unit WTScan;

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//                        Part of WindowsTree                              //
//                       Listeur d'arborescences                           //
//  2001 - 2011  Alain JAFFRE               http://jack.r.free.fr          //
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//                          Update history                                 //
//                                                                         //
// 20090123 Initialisation manquante de ffnumber dans TDirTreeEdit.Refresh //
// 20100226 Ajout du calcul de la somme de contrle des fichiers           //
//          Correction d'un bug sur le calcul de taille du rpertoire      //
//            lorsqu'il y avait une profondeur maxi                        //
// 20151108 Ajout du prefix '\\?\' a FRootDir dans BuildTreeList afin de   //
//            gerer des chemin de plus de 260 caracteres                   //
//            Voir: http://stackoverflow.com/questions/16903664/ ...       //
//  how-to-recurse-down-paths-over-255-characters-and-read-file-attributes //
//  https://msdn.microsoft.com/en-us/library/windows/desktop/aa365247.aspx //
//   #maxpath                                                              //
//                                                                         //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, veuillez consulter <http://www.gnu.org/licenses/>           }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, see <http://www.gnu.org/licenses/>.            }
{***************************************************************************}

interface

uses Windows, SysUtils, Classes, Forms, ComCtrls, Messages, Controls, Graphics,
  Masks, RichEdit, DCPCrypt2, DCPmd5, DCPsha1, DCPsha256, DCPsha512,
  WTLng;

const
  CARET_MESSAGE = WM_USER + 1;
  CARET_INSERT = 0;
  CARET_OVERWRITE = 1;

type
  TSortType = (soNone, soAlpha, soSize, soDate);
  THashType = (haNone, haMD5, haSHA1, haSHA256, haSHA512);

  TFileObject = class
  public
    IsDir: boolean;
    Filename: string;
    RelativePath: string;
    ComparePath: string;
    Time: integer;
    Size: int64;
    Attr: integer;
    Hash: string;
    FilesNb: int64;
    Number: string;
    Spacer: string;
    Depth: longint;
    constructor Create;
    destructor Destroy; override;
  end;

  TDirChangeEvent  = procedure (Sender: TObject; ADir: string) of object;

  TDirScanner = class(TComponent)
  private
    FDrive: char;                     // drive we scan
    FRootDir: string;                 // start directory for scanning
    FRootLength: integer;             // character length of root directory
    FRootSize: int64;                 // size of root directory
    FRootFilesNb: int64;              // number of files in the root directory
    FMaxDepth: longint;               // maximum directory depth for scanning
    FUseMask: boolean;                // did we use a mask for file selection
    FMaskList: TStringList;           // mask used for file selection
    FUseSize: boolean;                // did we use size for file selection
    FMinSize: int64;                  // minimum size in byte for file selection
    FMaxSize: int64;                  // maximum size in byte for file selection
    FSort: TSortType;                 // which sorting did we do on the result
    FHash: THashType;                 // which hash algorythm did we use
    FTreeList: TStringList;           // list all files and directories
    FCount: integer;                  // Number of tree items
    FStop: boolean;                   // interrupt processing

    FOnDirChange: TDirChangeEvent;    // event when scanner change directory

    FHorizontalChar: char;            // character use in spacer #196 for '-'
    FVerticalChar: char;              // character use in spacer #179 for '|'
    FMiddleChar: char;                // character use in spacer #195 for '+';
    FLastChar: char;                  // character use in spacer #192;

    procedure SetRootDir(ADirectory: string);
    procedure SetMaxDepth(ADepth: longint);
    procedure SetUseMask(AStatus: boolean);
    procedure SetUseSize(AStatus: boolean);
    procedure SetSort(ASortType: TSortType);
    procedure SetHash(AHashType: THashType);

    // function MatchMask(AFilename, AMask: string): boolean;
    function IsInMaskList(AFilename: string): boolean;
    function IsValidSize(ASize: int64): boolean;
    function CreateSpacer(Path: string): string;
    function AddFileObject(IsDir: boolean; Path, Name: string;
      Time: integer; Size: int64; Attr: integer; Hash: string; FilesNb: int64;
      Number, ComparePath: string; Depth: longint): integer;
    procedure BuildTreeList;
    procedure SortTreeList;
    procedure AdjustTreeList;
    function ComputeHash(AFilename: string): string;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

  published
    property Drive: char read FDrive;
    property RootDir: string read FRootDir write SetRootDir;
    property RootSize: int64 read FRootSize;
    property MaxDepth: longint read FMaxDepth write SetMaxDepth;
    property UseMask: boolean read FUseMask write SetUseMask;
    property UseSize: boolean read FUseSize write SetUseSize;
    property MinSize: int64 read FMinSize;
    property MaxSize: int64 read FMaxSize;
    property Sort: TSortType read FSort write SetSort;
    property Hash: THashType read FHash write SetHash;
    property Count: integer read FCount;
    property Stop: boolean read FStop write FStop;

    property HorizontalChar: char read FHorizontalChar;
    property VerticalChar: char read FVerticalChar;
    property MiddleChar: char read FMiddleChar;
    property LastChar: char read FLastChar;

    property OnDirChange: TDirChangeEvent read FOnDirChange write FOnDirChange;

    procedure SetMask(AMask: String);
    procedure SetSize(Min, Max: int64);
    procedure Process;
    function GetItem(ItemNumber: integer; var IsDir: boolean;
      var Filename, RelativePath: string; var Time: integer; var Size: int64;
      var Attr: integer; var Hash: string; var FilesNb: int64;
      var Number, Spacer: string; var Depth: longint): boolean;
  end;


  THighlight = class
    Styles: TFontStyles;
    Color: TColor;
    constructor Create;
    destructor Destroy; override;
  end;

  TDirTreeDisplay = (dtdEmptyFolder, dtdFile, dtdDate, dtdSize, dtdHash, dtdFilesNb);
  TDirTreeDisplays = set of TDirTreeDisplay;
  TDirTreeProcessingEvent  = procedure (Sender: TObject; ANumber: integer) of object;

  TDirTreeEdit = class(TRichEdit)
  // Specialized RichEdit for directory tree
  private
    FDirScanner: TDirScanner;           // directory scanner source
    FInsertMode: boolean;               // insert/overwrite mode
    FDisplays: TDirTreeDisplays;        // what will be displayed
    FDepth: longint;                    // max depth displayed
    FFileNumbering: boolean;            // do we display file number
    FFileNumber: integer;               // file number in directory
    FHighlight: boolean;                // do we highlight different parts
    FHighlightTree: THighlight;         // how we highlight tree
    FHighlightDir: THighlight;          // how we highlight directory
    FHighlightFile: THighlight;         // how we highlight file
    FHighlightDate: THighlight;         // how we highlight date
    FHighlightSize: THighlight;         // how we highlight size
    FHighlightHash: THighlight;         // how we highlight hash
    FLineSpaceDir: integer;             // number of line after each directory
    FLineSpaceFile: integer;            // number of line after each file
    FMaxLineSpace: integer;             // maximum number of line between items
    FStop: boolean;                     // interrupt processing
    FPreviousPath: string;              // path of previous item
    FPreviousIsDir: boolean;            // previous item was a directory
    FOnDirTreeProcessing: TDirTreeProcessingEvent; // event when displaying item
    procedure WMKeyDown(var Msg: TWMKeyDown); message WM_KEYDOWN;
    procedure SetDirScanner(ADirScanner: TDirScanner);
    procedure SetInsertMode(AStatus: boolean);
    procedure SetHighlight(AStatus: boolean);
    procedure SetLineSpaceDir(ANumber: integer);
    procedure SetLineSpaceFile(ANumber: integer);
    function BuildNumberString(ANumber: integer): string;
    function BuildSizeString(ASize: int64): string;
    function BuildDateString(ATime: integer): string;
    procedure AddItem(IsDir: boolean; ASpacer, AName, ARelativePath,
      ASize,ADate, AHash, AFilesNb: string);
  protected
    //
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Scanner: TDirScanner read FDirScanner write SetDirScanner;
    property InsertMode: boolean read FInsertMode write SetInsertMode;
    property Displays: TDirTreeDisplays read FDisplays write FDisplays;
    property Depth: longint read FDepth write FDepth;
    property FileNumbering: boolean read FFileNumbering write FFileNumbering;
    property Highlight: boolean read FHighlight write SetHighlight;
    property HighlightTree: THighlight read FHighlightTree write FHighlightTree;
    property HighlightDir: THighlight read FHighlightDir write FHighlightDir;
    property HighlightFile: THighlight read FHighlightFile write FHighlightFile;
    property HighlightDate: THighlight read FHighlightDate write FHighlightDate;
    property HighlightSize: THighlight read FHighlightSize write FHighlightSize;
    property HighlightHash: THighlight read FHighlightHash write FHighlightHash;
    property LineSpaceDir: integer read FLineSpaceDir write SetLineSpaceDir;
    property LineSpaceFile: integer read FLineSpaceFile write SetLineSpaceFile;
    property Stop: boolean read FStop write FStop;

    property OnDirTreeProcessing: TDirTreeProcessingEvent
      read FOnDirTreeProcessing
      write FOnDirTreeProcessing;

    procedure AddLine(AString: string);
    procedure InsertLines(ANumber: integer);
    procedure Refresh;
  end;

  function GetDriveTypeStr(ADrive: string): string;
  procedure GetVolumeInfo(ADrive: string;
    var VolumeName, FileSystemName, SerialNumber : string);
implementation

{*****************************************************************************}
{ Utilities                                                                   }
{*****************************************************************************}

function GetDriveTypeStr(ADrive: string): string;
// Return a string describing the type of drive
// Principle from http://delphi.developpez.com/faq/?page=lecteur
begin
  ADrive:= ADrive[1] + ':\';
  case GetDriveType(pchar(ADrive)) of
    1               : result:= GetMsg(0051);
    DRIVE_REMOVABLE : result:= GetMsg(0052);
    DRIVE_FIXED	    : result:= GetMsg(0053);
    DRIVE_REMOTE    : result:= GetMsg(0054);
    DRIVE_CDROM	    : result:= GetMsg(0055);
    DRIVE_RAMDISK   : result:= GetMsg(0056);
  else
    result:= 'Disque inconnu';
  end;
end;

{------------------------------------------------------------------------------}

procedure GetVolumeInfo(ADrive: string;
  var VolumeName, FileSystemName, SerialNumber : string);
// Get volume name, file systme name and volume serial number
// Principle from http://delphi.developpez.com/faq/?page=lecteur
var
  VolName     : Array[0..Max_Path]Of Char;
  FileSysName : Array[0..Max_Path]Of Char;
  VolSerial   : DWord;
  FileMaxLen  : DWord;
  FileFlags   : DWord;
begin
  ADrive:= ADrive[1] + ':\';
  if GetVolumeInformation(pchar(ADrive),VolName,Max_Path,@VolSerial,
             FileMaxLen, FileFlags,FileSysName,Max_Path) then
  begin
    VolumeName:= StrPas(VolName);
    FileSystemName:= StrPas(FileSysName);
    SerialNumber:= IntToStr(VolSerial);
  end
  else
  begin
    VolumeName:= '';
    FileSystemName:= '';
    SerialNumber:= '';
  end;
end;

{------------------------------------------------------------------------------}

function ExtractLastDir(APath: string): string;
// Get the last directory name of the specified path
var
  PathLength: integer;
begin
  PathLength:= length(APath);
  if PathLength <= 3 then result:= APath
  else
  begin
    result:= '';
    // Remove last \
    if APath[PathLength] = '\' then
    begin
      delete(APath,PathLength,1);
      dec(PathLength);
    end;
    // Find previous \
    while APath[PathLength] <> '\' do
    begin
      result:= APath[PathLength] + result;
      dec(PathLength);
    end;
  end;
end;

{------------------------------------------------------------------------------}


function CompareDateString(ATime: integer): string;
// Format properly the date  for comparison
begin
  if ATime <> 0 then
    DateTimeToString(result,'yyyymmddhhnnss',FileDateToDateTime(ATime))
  else
    result:= '';
end;

{------------------------------------------------------------------------------}
function CompareAlphaProc(List: TStringList;
  Index1, Index2: Integer): Integer;
var
  FileObject1: TFileObject;
  FileObject2: TFileObject;
  IsDir1: boolean;
  IsDir2: boolean;
  Str1: string;
  Str2: string;
begin
  FileObject1:= TFileObject(List.Objects[Index1]);
  FileObject2:= TFileObject(List.Objects[Index2]);
  IsDir1:= FileObject1.IsDir;
  IsDir2:= FileObject2.IsDir;
  // Compare RelativePath
  Str1:= FileObject1.RelativePath;
  Str2:= FileObject2.RelativePath;
  // If directory, add filename to relative path
  if IsDir1 then Str1:= Str1 + FileObject1.Filename;
  if IsDir2 then Str2:= Str2 + FileObject2.Filename;
  if (not IsDir1) and (not IsDir2) and (Str1 = Str2) then
  begin
    // Files on the same level, must compare Filename
    Str1:= FileObject1.Filename;
    Str2:= FileObject2.Filename;
  end;
  result:= AnsiCompareStr(Str1,Str2);

(*
  // More simple solution but mixe files and directories in the same level
  FileObject1:= TFileObject(List.Objects[Index1]);
  FileObject2:= TFileObject(List.Objects[Index2]);
  Str1:= FileObject1.RelativePath + FileObject1.Filename;
  Str2:= FileObject2.RelativePath + FileObject2.Filename;
  result:= AnsiCompareStr(Str1,Str2);
*)

end;

{------------------------------------------------------------------------------}

function CompareSizeProc(List: TStringList;
  Index1, Index2: Integer): Integer;
var
  FileObject1: TFileObject;
  FileObject2: TFileObject;
  IsDir1: boolean;
  IsDir2: boolean;
  Str1: string;
  Str2: string;
begin
  FileObject1:= TFileObject(List.Objects[Index1]);
  FileObject2:= TFileObject(List.Objects[Index2]);
  IsDir1:= FileObject1.IsDir;
  IsDir2:= FileObject2.IsDir;
  // Compare RelativePath
  Str1:= FileObject1.RelativePath;
  Str2:= FileObject2.RelativePath;
  if (IsDir1 = IsDir2) and (Str1 = Str2) then
  begin
    // File or directory on the same level, compare size. If same date compare filename
    if FileObject1.Size < FileObject2.Size then result := -1
    else
      if FileObject1.Size > FileObject2.Size then result := 1
      else
      begin
        Str1 := FileObject1.Filename;
        Str2 := FileObject2.Filename;
        result := AnsiCompareStr(Str1, Str2);
      end;
  end
  else
  begin
    // If directory, add filename to relative path
    if IsDir1 then Str1:= Str1 + FileObject1.Filename;
    if IsDir2 then Str2:= Str2 + FileObject2.Filename;
    result := AnsiCompareStr(Str1, Str2);
  end;
end;

{------------------------------------------------------------------------------}

function CompareDateProc(List: TStringList;
  Index1, Index2: Integer): Integer;
var
  FileObject1: TFileObject;
  FileObject2: TFileObject;
  IsDir1: boolean;
  IsDir2: boolean;
  Str1: string;
  Str2: string;
begin
  FileObject1:= TFileObject(List.Objects[Index1]);
  FileObject2:= TFileObject(List.Objects[Index2]);
  IsDir1:= FileObject1.IsDir;
  IsDir2:= FileObject2.IsDir;
  // Compare ComparePath
  Str1:= FileObject1.ComparePath;
  Str2:= FileObject2.ComparePath;
  // If directory, add filename to relative path
  if IsDir1 then Str1:= Str1 + CompareDateString(FileObject1.Time) + FileObject1.Filename;
  if IsDir2 then Str2:= Str2 + CompareDateString(FileObject2.Time) + FileObject2.Filename;
  if (not IsDir1) and (not IsDir2) and (Str1 = Str2) then
  begin
    // Files on the same level, must compare date and Filename
    Str1:= CompareDateString(FileObject1.Time) + FileObject1.Filename;
    Str2:= CompareDateString(FileObject2.Time) + FileObject2.Filename;
  end;
  result:= AnsiCompareStr(Str1,Str2);
end;

{*****************************************************************************}
{ File object                                                                 }
{*****************************************************************************}

{ TFileObject }

constructor TFileObject.Create;
begin
  inherited;
  IsDir:= false;
  Filename:= '';
  RelativePath:= '';
  Time:= 0;
  Size:= 0;
  Attr:= 0;
  Hash:= '';
  Number:= '';
  Spacer:= '';
end;

{------------------------------------------------------------------------------}

destructor TFileObject.Destroy;
begin
  inherited;
end;

{*****************************************************************************}
{ Directory scanner                                                           }
{*****************************************************************************}

{ TDirScanner }

{------------------------------------------------------------------------------}
{ Private                                                                      }
{------------------------------------------------------------------------------}

procedure TDirScanner.SetRootDir(ADirectory: string);
var
 ALength: integer;
begin
  ADirectory:= trim(ADirectory);
  ALength:= length(ADirectory);
  if (ALength > 2) and (ADirectory[ALength]<>'\') then
    ADirectory:= ADirectory + '\';
  FRootDir:= ADirectory;
  FRootLength:= length(FRootDir);
  FDrive:= FRootDir[1];
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SetMaxDepth(ADepth: longint);
begin
  if ADepth < 0 then ADepth:= 0;
  FMaxDepth:= ADepth;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SetUseMask(AStatus: boolean);
begin
  FUseMask:= AStatus;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SetUseSize(AStatus: boolean);
begin
  FUseSize:= AStatus;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SetSort(ASortType: TSortType);
begin
  try
    FSort:= ASortType;
  except
    FSort:= soNone;
  end;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SetHash(AHashType: THashType);
begin
  try
    FHash:= AHashType;
  except
    FHash:= haNone;
  end;
end;

{------------------------------------------------------------------------------}
{
  // 20100320 widowstree.1.2.3.ini doesn't match *.ini with that way of doing.
  // replaced by MatchesMask delphi function

function TDirScanner.MatchMask(AFilename, AMask: string): boolean;
// Return true if Filename match this mask
var
  Str1: string;
  Str2: string;

  function Match(Source,Ref: string): boolean;
  var
    Len: integer;
    Tmp: string;
  begin
    result:= true;
    if Ref = '*' then exit;
    // Ref must be smaller then source however it will never match
    if length(Ref) > length(Source) then result:= false
    else
    begin
      // *
      if pos('*',Ref) > 0  then
      begin
        // before *
        Len:= pos('*',Ref);
        if Len > 1 then
        begin
          Tmp:= copy(Source,1,Len - 1);
          system.delete(Source,1,len - 1);
        end;
        // after *
        Len:= length(Ref) - Len;
        if Len > 0 then
          Tmp:= Tmp + copy(Source,length(Source) - Len,Len);
        Len:= pos('*',Ref);
        // new element
        Source:= Tmp;
        system.delete(Ref,Len,1);
        if pos('*',Ref) > 0  then
        begin
          // Bad mask
          result:= false;
          exit;
        end;
      end;
      // Ref must have same length as source however it will never match
      if length(Ref) <> length(Source) then result:= false
      else
      begin
        // ?
        while result and (pos('?',Ref) > 0) do
        begin
          Len:= pos('?',Ref);
          system.delete(Source,Len,1);
          system.delete(Ref,Len,1);
        end;
        result:= Source = Ref;
      end;
    end;
  end;

begin
  if AMask= '*' then result:= true // always match
  else
  if AMask= '*.*' then result:= true // always match
  else
  if pos('.',AMask)= 0 then result:= Match(AFilename,AMask) // no extension
  else
  if pos('.',AFilename)= 0 then result:= false  // Mask has . but filename don't
  else
  begin
    // Extension
    Str1:= copy(AFilename,pos('.',AFilename) + 1, MaxInt);
    Str2:= copy(AMask,pos('.',AMask) + 1, Maxint);
    result:= Match(Str1,Str2);
    if result then
    begin
      // Name
      Str1:= copy(AFilename,1,pos('.',AFilename) - 1);
      Str2:= copy(AMask,1,pos('.',AMask) - 1);
      result:= Match(Str1,Str2);
    end;
  end;
end;
}
{------------------------------------------------------------------------------}

function TDirScanner.IsInMaskList(AFilename: string): boolean;
// Return true if AFilename match one of the mask of the FMaskList
var
  N: integer;
begin
  N:= 0;
  repeat
    result:= MatchesMask(AFilename,FMaskList.Strings[N]);
    inc(N);
  until result or (N = FMaskList.Count);
end;

{------------------------------------------------------------------------------}

function TDirScanner.IsValidSize(ASize: int64): boolean;
// Return true if ASize is the allowed range (FMinSize to FMaxSize)
begin
  if FMaxSize > 0 then
    result:= (ASize >= FMinSize) and (ASize <= FMaxSize)
  else
    result:= (ASize >= FMinSize);
end;

{------------------------------------------------------------------------------}

function TDirScanner.CreateSpacer(Path: string): string;
// Create the spacer corresponding to the relative path
var
  APos: integer;
begin
  result:= '  ' + FLastChar + FHorizontalChar + ' ';
  // All \
  APos:= pos('\',Path);
  while APos > 0 do
  begin
    result:= '       ' + result;
    system.delete(Path,1,APos);
    APos:= pos('\',Path);
  end;
end;

{------------------------------------------------------------------------------}

function TDirScanner.AddFileObject(IsDir: boolean; Path, Name: string;
  Time: integer; Size: int64; Attr: integer; Hash: string; FilesNb: int64;
  Number, ComparePath: string; Depth: longint): integer;
// Add a new FileObject in the TreeList
var
  FileObject: TFileObject;
begin
  FileObject:= TFileObject.Create;
  FileObject.IsDir:= IsDir;
  system.delete(Path,1,FRootLength);
  FileObject.RelativePath:= Path;
  FileObject.ComparePath:= ComparePath;
  FileObject.Filename:= Name;
  FileObject.Time:= Time;
  FileObject.Size:= Size;
  FileObject.Attr:= Attr;
  FileObject.Hash:= Hash;
  FileObject.FilesNb:= FilesNb;
  FileObject.Number:= Number;
  FileObject.Spacer:= CreateSpacer(Path);
  FileObject.Depth:= Depth;
  Result:= FTreeList.AddObject(Path + Name, FileObject);
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.BuildTreeList;
// Recurse in all folder to find files and folder
var
  ARootSize: int64;
  ARootFilesNb: int64;
  ARootDir: string;

  function ScanDirectory(var Path: string; var PathSize: int64;
    Depth: longint; var FilesNb: int64; ComparePath: string):boolean;
  var
    SRec: TSearchRec;
    PathLength: integer;
    SResult: integer;
    Index: longint;
    Number: string;
    SizePart: TULargeInteger;
    RealSize: int64;
    TotalSize: int64;
    ObjectIndex: integer;
    TmpFileObject: TFileObject;
    APathSize: int64;
    Valid: boolean;
    CurrentDepth: longint;
    AFilesNb: int64;
    TotalFilesNb: int64;
    Hash: string;
    OldComparePath: string;
  begin
    result:= true;
    Index:= 0;
    TotalSize:= 0;
    CurrentDepth:= Depth + 1;
    TotalFilesNb:= 0;
    // Adapt path string
    PathLength:= length(Path);
    if Path[PathLength]<>'\' then
    begin
      Path:= Path + '\';
      inc(PathLength);
    end;
    // Tell that we have change of directory
    if assigned(FOnDirChange) then FOnDirChange(self,Path);
    Application.ProcessMessages;

    // Look for files
    if (FMaskList[0]<>'') then
    begin
      SResult:= FindFirst(Path + '*.*',faAnyfile,SRec);
      try
        while SResult=0 do
        begin
          if ((SRec.Attr and faDirectory)<>faDirectory)
            and (SRec.Name<>'.') and (SRec.Name<>'..') then
          begin
            inc(Index);
            with SRec do
            begin
              Valid:= true;
              // Compute real size
              SizePart.LowPart := FindData.nFileSizeLow;
              SizePart.HighPart := FindData.nFileSizeHigh;
              RealSize := SizePart.QuadPart;
              if (FUseMask and
                (not IsInMaskList(AnsiLowerCase(Name)))) then Valid:= false;
              if (FUseSize and (not IsValidSize(RealSize))) then Valid:= false;
              if  Valid then
              begin
                Number:= Format('%4.4d',[Index]);
                if FHash = haNone then Hash:= ''
                                  else Hash:= ComputeHash(Path+Name);
                AddFileObject(false,Path,Name,Time,RealSize,Attr,Hash,1, Number, ComparePath, CurrentDepth);
                inc(TotalSize,RealSize);
                inc(TotalFilesNb);
              end;
            end;
          end;
          SResult:= FindNext(SRec);
          if FStop then exit;
        end;
      finally
        FindClose(SRec);
      end;
      Application.ProcessMessages;
    end;

    // Look for directories
    SResult:= FindFirst(Path + '*.*',faDirectory,SRec);
    try
      while (SResult=0) and result do
      begin
        if ((SRec.Attr and faDirectory)=faDirectory)
          and (SRec.Name<>'.') and (SRec.Name<>'..') then
        begin
          with SRec do
            ObjectIndex:= AddFileObject(true,Path,Name,Time,Size,Attr,'',0,'',ComparePath, CurrentDepth);
            // Check if we reach max depth
            if ((FMaxDepth = 0) or (CurrentDepth < FMaxDepth)) then
            begin
              // Recurse
              OldComparePath:= ComparePath;
              Path:= Path + SRec.Name + '\';
              case FSort of
                soNone: ComparePath:= Path;
                soAlpha: ComparePath:= Path;
                soSize: ComparePath:= Path;
                soDate: ComparePath:= ComparePath + CompareDateString(SRec.Time) + SRec.Name + '\';
              end;
              APathSize:= 0;
              result:= ScanDirectory(Path, APathSize, CurrentDepth, AFilesNb, ComparePath);
              // Go back to previous directory
              ComparePath:= OldComparePath;
              setlength(Path, PathLength);
              TmpFileObject:= TFileObject(FTreeList.Objects[ObjectIndex]);
              TmpFileObject.Size:= APathSize;
              TmpFileObject.FilesNb:= AFilesNb;
              inc(TotalSize,APathSize);
              inc(TotalFilesNb,AFilesNb);
            end;
        end;
        SResult:= FindNext(SRec);
        Application.ProcessMessages;
        if FStop then exit;
      end;
    finally
      FindClose(SRec);
    end;
    PathSize:= TotalSize;
    FilesNb:= TotalFilesNb;
  end;

begin
  FTreeList.Clear;
  ARootSize:= 0;
  ARootFilesNb:= 0;
  // Check if we need to add UNC header
  if copy(FRootDir,1,2)<>'\\' then
    ARootDir:= '\\?\'+FRootDir
  else
    ARootDir:= FRootDir;
  ScanDirectory(ARootDir,ARootSize, 0, ARootFilesNb, '');
  FRootSize:= ARootSize;
  FRootFilesNb:= ARootFilesNb;
  FCount:= FTreeList.Count;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SortTreeList;
// Sort the tree according to the selected sort option
begin
  case FSort of
    soAlpha : FTreeList.CustomSort(@CompareAlphaProc);
    soSize  : FTreeList.CustomSort(@CompareSizeProc);
    soDate  : FTreeList.CustomSort(@CompareDateProc);
  end;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.AdjustTreeList;
// Adapt all spacer regarding the next one in the tree
var
  N: integer;
  FileObject: TFileObject;
  Spacer: string;
  NextSpacer: string;
  APos: integer;
begin
  // If less than 2 elements, nothing to do
  if FCount < 1 then exit;
  if FStop then exit;
  // Get the last one
  FileObject:= TFileObject(FTreeList.Objects[FCount-1]);
  NextSpacer:= FileObject.Spacer;
  // Process all previous FileObject
  for N:= (FCount-1) downto 1 do
  begin
    FileObject:= TFileObject(FTreeList.Objects[N-1]);
    Spacer:= FileObject.Spacer;
    // Spacer longer than next one
    if length(Spacer) > length(NextSpacer) then
    begin
      APos:= length(NextSpacer) - 2;
      system.delete(Spacer,1,APos);
      Spacer:= copy(NextSpacer,1,APos) + Spacer;
      if Spacer[APos] = FLastChar then Spacer[APos] := FVerticalChar
      else
      if Spacer[APos] = FMiddleChar then Spacer[APos] := FVerticalChar
    end
    else
    // Spacer same length as next one
    if length(Spacer) = length(NextSpacer) then
    begin
      APos:= length(Spacer)-3;
      Spacer:= copy(NextSpacer,1,APos) + FMiddleChar + FHorizontalChar + ' ';
    end
    else
    // Spacer is shorter than next one
    if length(Spacer) < length(NextSpacer) then
    begin
      APos:= length(Spacer)-2;
      if APos > 0 then
      begin
        Spacer:= copy(NextSpacer,1,APos) + FHorizontalChar + ' ';
        if Spacer[APos]= FVerticalChar then Spacer[APos]:= FMiddleChar
        else
        if Spacer[APos]= ' ' then Spacer[APos]:= FLastChar;
      end;
    end;
    FileObject.Spacer:= Spacer;
    NextSpacer:= Spacer;
    Application.ProcessMessages;
    if FStop then exit;
  end;
end;

{------------------------------------------------------------------------------}

function TDirScanner.ComputeHash(AFilename: string): string;
// Compute corresponding hash for specified file
var
  DCP_hash: TDCP_hash;
  Buffer: array[0..65535] of byte;
  ReadBytes: cardinal;
  InputStream: TFileStream;
  Digest: array of byte;
  Hash: string;
  N: longint;
begin
  if FileExists(AFilename) then
  begin
    // Compute Hash
    try
      InputStream := TFileStream.Create(AFilename,fmOpenRead);
      case FHash of
        haMD5: DCP_hash:= TDCP_md5.Create(nil);
        haSHA1: DCP_hash:= TDCP_sha1.Create(nil);
        haSHA256: DCP_hash:= TDCP_sha256.Create(nil);
        haSHA512: DCP_hash:= TDCP_sha512.Create(nil);
      else
        DCP_hash:= TDCP_hash.Create(nil);
      end;
      DCP_hash.Init;
      repeat
        // read into the buffer
        ReadBytes := InputStream.Read(Buffer,Sizeof(Buffer));
        // hash the buffer
        DCP_hash.Update(Buffer,ReadBytes);
        Application.ProcessMessages;
      until (ReadBytes <> Sizeof(Buffer)) or FStop;
      InputStream.Free;
      SetLength(Digest,DCP_hash.HashSize div 8);
      DCP_hash.Final(Digest[0]);;
      // convert into an hexadecimal string
      Hash:= '';
      for N:= 0 to Length(Digest) - 1 do
        Hash:= Hash + IntToHex(Digest[N],2);
      Hash:= lowercase(Hash);
      DCP_hash.Free;
    except
      Hash:= '';
    end;
  end;
  if FStop then result:= ''
           else result:= Hash;
end;

{------------------------------------------------------------------------------}

{------------------------------------------------------------------------------}
{ Public                                                                       }
{------------------------------------------------------------------------------}

constructor TDirScanner.Create(AOwner: TComponent);
begin
  inherited;
    FRootDir:= '';
    FRootLength:= 0;
    FRootSize:= 0;
    FUseMask:= false;
    FMaskList:= TStringList.Create;
    FMaskList.Add('*');
    FUseSize:= false;
    FMinSize:= 0;
    FMaxSize:= 0;
    FSort:= soNone;
    FHash:= haNone;
    FTreeList:= TStringList.Create;
    FCount:= 0;

    FHorizontalChar:= #196; // '-'
    FVerticalChar:= #179;   // '|'
    FMiddleChar:= #195;     // '+';
    FLastChar:= #192;
end;

{------------------------------------------------------------------------------}

destructor TDirScanner.Destroy;
var
  N: integer;
begin
  for N:= 0 to (FCount - 1) do
  begin
    FTreeList.Objects[N].Free;
  end;
  FTreeList.Free;
  FMaskList.Free;
  inherited;
end;

{------------------------------------------------------------------------------}
{ Published                                                                    }
{------------------------------------------------------------------------------}

procedure TDirScanner.SetMask(AMask: String);
// Set and adapt mask in the mask list
var
  APos: integer;
  Tmp: string;
begin
  FMaskList.Clear;
  // If Mask empty then all files
  if AMask= '' then AMask:= '*';
  // Lower case to be case unsensitive
  AMask:= AnsiLowerCase(AMask);
  // Find and store all different mask
  APos:= pos(';',AMask);
  if APos = 0 then FMaskList.Add(trim(AMask))
  else
  begin
    while APos > 0 do
    begin
      Tmp:= copy(AMask,1,APos-1);
      if trim(Tmp) <> '' then FMaskList.Add(Tmp);
      system.delete(AMask,1,APos);
      APos:= pos(';',AMask);
    end;
    if trim(AMask) <> '' then FMaskList.Add(AMask);
  end;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.SetSize(Min, Max: int64);
// Set and adapt the min and max size
begin
  if (Max <> 0) and (Min > Max) then Min:= Max;
  FMinSize:= Min;
  FMaxSize:= Max;
end;

{------------------------------------------------------------------------------}

procedure TDirScanner.Process;
// Scan, build and arrange the result
begin
  BuildTreeList;
  SortTreeList;
  AdjustTreeList;
end;

{------------------------------------------------------------------------------}

function TDirScanner.GetItem(ItemNumber: integer; var IsDir: boolean;
  var Filename, RelativePath: string; var Time: integer; var Size: int64;
  var Attr: integer; var Hash: string; var FilesNb: int64;
  var Number, Spacer: string; var Depth: longint): boolean;
  // Return the corresponding item from the tree
var
  FileObject: TFileObject;
begin
  if ItemNumber <= FCount then
  begin
    FileObject:= TFileObject(FTreeList.Objects[ItemNumber-1]);
    IsDir:= FileObject.IsDir;
    Filename:= FileObject.Filename;
    RelativePath:= FileObject.RelativePath;
    Time:= FileObject.Time;
    Size:= FileObject.Size;
    Attr:= FileObject.Attr;
    Hash:= FileObject.Hash;
    FilesNb:= FileObject.FilesNb;
    Number:= FileObject.Number;
    Spacer:= FileObject.Spacer;
    Depth:= FileObject.Depth;
    result:= true;
  end
  else result:= false;
end;

{------------------------------------------------------------------------------}

{*****************************************************************************}
{ Highlighter                                                                 }
{*****************************************************************************}

{ THighlight }

constructor THighlight.Create;
begin
  Styles:= [];
  Color:= ClBlack;
end;

{------------------------------------------------------------------------------}

destructor THighlight.Destroy;
begin

  inherited;
end;

{*****************************************************************************}
{ Directory tree editor                                                       }
{*****************************************************************************}

{ TDirTreeEdit }

{------------------------------------------------------------------------------}
{ Private                                                                      }
{------------------------------------------------------------------------------}

procedure TDirTreeEdit.WMKeyDown(var Msg: TWMKeyDown);
// Key down event handler
begin
  case Msg.CharCode of
    VK_Insert : FInsertMode:= not FInsertMode;
  end;
  inherited;
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.SetDirScanner(ADirScanner: TDirScanner);
// Set DirScanner source for DirTreeEdit
begin
  FDirScanner:= ADirScanner;
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.SetInsertMode(AStatus: boolean);
// Set Insert mode
begin
  FInsertMode:= AStatus;
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.SetHighlight(AStatus: boolean);
// Set highlighting for tree, directories, files, date/time, size
begin
  FHighlight:= AStatus;
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.SetLineSpaceDir(ANumber: integer);
// Set number of spacing line after directories
begin
  if ANumber < 0 then ANumber:= 0;
  if ANumber > FMaxLineSpace then ANumber:= FMaxLineSpace;
  FLineSpaceDir:= ANumber;
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.SetLineSpaceFile(ANumber: integer);
// Set number of spacing line after files
begin
  if ANumber < 0 then ANumber:= 0;
  if ANumber > FMaxLineSpace then ANumber:= FMaxLineSpace;
  FLineSpaceFile:= ANumber;
end;

{------------------------------------------------------------------------------}

function TDirTreeEdit.BuildNumberString(ANumber: integer): string;
// Format properly the file number
begin
    result:= Format('%4.4d',[ANumber]) + '  ';
end;

{------------------------------------------------------------------------------}

function TDirTreeEdit.BuildSizeString(ASize: int64): string;
// Format properly the size
begin
  if ASize <> 0 then
    result:= '  ' + FloatToStrF(ASize / 1024,ffNumber,20,2) + ' Ko'
  else
    result:= '  0 Ko';
end;

{------------------------------------------------------------------------------}

function TDirTreeEdit.BuildDateString(ATime: integer): string;
// Format properly the date
begin
  if ATime <> 0 then
    result:= '  ' + DateTimeToStr(FileDateToDateTime(ATime))
  else
    result:= '';
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.AddItem(IsDir: boolean; ASpacer, AName, ARelativePath,
  ASize, ADate, AHash, AFilesNb: string);
// Add an item in the DirTreeEdit
begin
  if (IsDir or (dtdFile in Displays)) then
  begin
    // Display only if directory or if file display is requested
    if (IsDir and (AName <> '')) then AName:= '<' + AName + '>';
    if FHighlight then
    begin
      // With highlight
      // Tree
      if ASpacer <> '' then
      begin
        SelAttributes.Style:= FHighlightTree.Styles;
        SelAttributes.Color:= FHighlightTree.Color;
        SelText:= ASpacer;
      end;
      if AName <> '' then
      begin
        // Directory
        if IsDir then
        begin
          SelAttributes.Style:= FHighlightDir.Styles;
          SelAttributes.Color:= FHighlightDir.Color;
          SelText:= AName;
        end
        else
        // File
        begin
          if (dtdFile in Displays) then
          begin
            if FFileNumbering then
            begin
              if ARelativePath <> FPreviousPath then FFileNumber:= 1
                                                else inc(FFileNumber);
              FPreviousPath:= ARelativePath;
              SelText:= BuildNumberString(FFileNumber);
            end;
            SelAttributes.Style:= FHighlightFile.Styles;
            SelAttributes.Color:= FHighlightFile.Color;
            SelText:= AName + #9;
          end;
        end;
        FPreviousIsDir:= IsDir;
      end;
      // Date
      if ADate <> '' then
      begin
        if (dtdDate in Displays) then
        begin
          SelAttributes.Style:= FHighlightDate.Styles;
          SelAttributes.Color:= FHighlightDate.Color;
          SelText:= ADate + #9;
        end;
      end;
      // Size
      if ASize <> '' then
      begin
        if (dtdSize in Displays) then
        begin
          SelAttributes.Style:= FHighlightSize.Styles;
          SelAttributes.Color:= FHighlightSize.Color;
          SelText:= ASize + #9;
        end;
      end;
      // Hash
      if AHash <> '' then
      begin
        if (dtdHash in Displays) then
        begin
          SelAttributes.Style:= FHighlightHash.Styles;
          SelAttributes.Color:= FHighlightHash.Color;
          SelText:= AHash;
        end;
      end;
      // Number of files
      if ((AFilesNb <> '') and IsDir) then
      begin
        if dtdFilesNb in Displays then
        begin
          SelAttributes.Style:= FHighlightSize.Styles;
          SelAttributes.Color:= FHighlightSize.Color;
          SelText:= '  (' + AFilesNb +')';
        end;
      end;
    end
    else
    begin
      // No highlight
      // Tree
      if ASpacer <> '' then SelText:= ASpacer;
      // Directory or file
      if IsDir then
      begin
        // Directory
        if AName <> '' then SelText:= AName;
      end
      else
      begin
        // File
        if FFileNumbering then
        begin
          if ARelativePath <> FPreviousPath then FFileNumber:= 1
                                            else inc(FFileNumber);
          FPreviousPath:= ARelativePath;
          SelText:= BuildNumberString(FFileNumber);
        end;
        if AName <> '' then SelText:= AName + #9;
      end;
      // Date
      if ((ADate <> '') and (dtdDate in Displays)) then SelText:= ADate;
      // Size
      if ((ASize <> '') and (dtdSize in Displays)) then SelText:= ASize;
      // Hash
      if ((AHash <> '') and (dtdHash in Displays)) then SelText:= AHash;
      // Number of files
      if ((AFilesNb <> '') and (dtdFilesNb in Displays) and IsDir) then SelText:= '  (' + AFilesNb +')';
    end;
    SelText:= #13#10;
  end;
end;

{------------------------------------------------------------------------------}
{ Protected                                                                    }
{------------------------------------------------------------------------------}

{------------------------------------------------------------------------------}
{ Public                                                                       }
{------------------------------------------------------------------------------}

constructor TDirTreeEdit.Create(AOwner: TComponent);
begin
  inherited;
  FDirScanner:= nil;
  FInsertMode:= true;
  FDisplays:= [];
  FDepth:=0;
  FFileNumbering:= false;
  FFileNumber:= 0;
  FHighlight:= false;
  FHighlightTree:= THighlight.Create;
  FHighlightDir:= THighlight.Create;
  FHighlightFile:= THighlight.Create;
  FHighlightDate:= THighlight.Create;
  FHighlightSize:= THighlight.Create;
  FHighlightHash:= THighlight.Create;
  FLineSpaceDir:= 0;
  FLineSpaceFile:= 0;
  FMaxLineSpace:= 2;
  FStop:= false;
  FPreviousPath:= '';
  FPreviousIsDir:= false;
end;

{------------------------------------------------------------------------------}

destructor TDirTreeEdit.Destroy;
begin
  FHighlightTree.Free;
  FHighlightDir.Free;
  FHighlightFile.Free;
  FHighlightDate.Free;
  FHighlightSize.Free;
  FHighlightHash.Free;
  inherited;
end;


{------------------------------------------------------------------------------}
{ Published                                                                    }
{------------------------------------------------------------------------------}

procedure TDirTreeEdit.AddLine(AString: string);
// Add a new line in the DirTreeEdit
begin
  SelText:= AString + #13#10;
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.InsertLines(ANumber: integer);
// Add properly formatted lines below de current one
var
  PosY: longint;
  ALine: string;
  ASpacer: string;
  N: integer;
begin
  PosY:= SendMessage(Handle, EM_EXLINEFROMCHAR, 0, SelStart);
  inc(PosY);
  // Get line below
  ALine:= Lines[PosY];
  // Get spacer
  N:= 1;
  ASpacer:= '';
  while (N <= length(ALine)) and
    (ALine[N] in [' ', FDirScanner.VerticalChar, FDirScanner.MiddleChar, FDirScanner.LastChar]) do
  begin
    if ALine[N] in [FDirScanner.MiddleChar, FDirScanner.LastChar] then
      ALine[N]:= FDirScanner.VerticalChar;
    ASpacer:= ASpacer + ALine[N];
    inc(N);
  end;
  // Move to start of below line
  SelStart:= Perform(EM_LINEINDEX,PosY,0);
  // Add lines
  for N:= 1 to ANumber do SelText:= ASpacer + #13#10;
  // Move to the end of the first added line
  SelStart:= Perform(EM_LINEINDEX,PosY,0) + length(ASpacer);
end;

{------------------------------------------------------------------------------}

procedure TDirTreeEdit.Refresh;
var
  N: integer;
  I: integer;
  AFilename: string;
  ARelativePath: string;
  ATime: integer;
  ASize: int64;
  AnAttr: integer;
  AHash: string;
  AFilesNb: int64;
  ANumber: string;
  ASpacer: string;
  ADepth: longint;
  IsDir: boolean;

  SpacingSpacer: string;
  SpcLength: integer;
  SizeStr: string;
  DateStr: string;
  HashStr: string;
  FilesNbStr: string;

  EmptyFolder: boolean;
begin
  if FDirScanner <> nil then
  begin
    Lines.BeginUpdate;
    SelStart:= getTextLen;

    Paragraph.TabCount:= 4;
    Paragraph.Tab[0] := 0;
    Paragraph.Tab[1] := 0;
    Paragraph.Tab[2] := 0;
    Paragraph.Tab[3] := 0;

    AFilename:= ExtractLastDir(FDirScanner.RootDir);
    SizeStr:= BuildSizeString(FDirScanner.RootSize);
    FFileNumber:= 0;
    AddItem(true,'', AFilename, '',SizeStr , '', '', '');
    // Display items
    FPreviousPath:= '';
    N:= 1;
    while N <= FDirScanner.Count do
    begin
      if FDirScanner.GetItem(N, IsDir, AFilename, ARelativePath,
        ATime, ASize, AnAttr, AHash, AFilesNb, ANumber, ASpacer, ADepth) then
      begin
        if (FDepth = 0) or (ADepth <= FDepth) then
        begin
          if (dtdEmptyFolder in Displays) then
            EmptyFolder:= false
          else
            EmptyFolder:= IsDir and (AFilesNb = 0);
          if not EmptyFolder then
          begin
            SizeStr:= BuildSizeString(ASize);
            DateStr:= BuildDateString(ATime);
            HashStr:= '    ' + AHash;
            FilesNbStr:= intTostr(AFilesNb);
            if ((ASpacer <> '') and
                (((FLineSpaceDir > 0) and (FPreviousIsDir)) or
                 ((FLineSpaceFile > 0) and (not FPreviousIsDir)))
               ) then
            begin
              // Compute new spacer
              SpacingSpacer:= copy(ASpacer,1,length(ASpacer) - 2);
              SpcLength:= length(SpacingSpacer);
              if (SpacingSpacer[SpcLength] = FDirScanner.MiddleChar) or
                (SpacingSpacer[SpcLength] = FDirScanner.LastChar) then
                SpacingSpacer[SpcLength]:= FDirScanner.VerticalChar
              else
                SpacingSpacer[SpcLength]:= ' ';
              SpacingSpacer:= SpacingSpacer + ' ';
              // Add spacing line between items
              if FPreviousIsDir then
              begin
                for I:= 1 to FLineSpaceDir do
                  AddItem(IsDir,SpacingSpacer, '', '', '', '', '', '');
              end
              else
              begin
                for I:= 1 to FLineSpaceFile do
                  AddItem(IsDir,SpacingSpacer, '', '', '', '','', '');
              end;
            end;
            Paragraph.Tab[0] := (length(ASpacer) div 3 + 20)* font.Size;
            Paragraph.Tab[1] := Paragraph.Tab[0] + 50;
            Paragraph.Tab[2] := Paragraph.Tab[1] + 50;
            Paragraph.Tab[3] := Paragraph.Tab[2] + 60;
            AddItem(IsDir,ASpacer, AFilename, ARelativePath, SizeStr, DateStr, HashStr, FilesNbStr);
          end;
        end;
      end;
      if assigned(FOnDirTreeProcessing) then FOnDirTreeProcessing(self,N);
      Application.ProcessMessages;
      if FStop then N:= FDirScanner.Count;
      inc(N);
    end;

    // Get back to default style and color
    SelAttributes.Style:= [];
    SelAttributes.Color:= clWindowText;
    SelText:= #13#10;

    Lines.EndUpdate;
  end;
end;

end.
