unit UTCDateTime;

{***************************************************************************}
{ L'usage, la modification, la diffusion de ce code source est libre.       }
{                                                                           }
{ Ce code est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE      }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER.                                      }
{                                                                           }
{ Code d'origine: UTCDateTime.pas                                           }
{                                                                           }
{ Developpeur d'origine: A.Jaffre   jack.r@free.fr   http://jack.r.free.fr  }
{***************************************************************************}

{***************************************************************************}
{ Usage, modification, distribution of this source code is free.            }
{                                                                           }
{ This source is distributed in the hope it will be useful, but WITHOUT     }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE.                                         }
{                                                                           }
{ Original code: UTCDateTime.pas                                            }
{                                                                           }
{ Original developer: A.Jaffre   jack.r@free.fr   http://jack.r.free.fr     }
{***************************************************************************}

interface

uses
  Windows,SysUtils;

  function AdaptSystemTime(SystemTime: TSystemTime): TDateTime;

  function UTCToDateTime(UTC: longint): TDateTime;
  function DateTimeToUTC(DateTime: TDateTime): longint;
  function UTCToLocalDateTime(UTC: longint): TDateTime;
  function LocalDateTimeToUTC(LocalDateTime: TDateTime): longint;

  function SecondFromRefToDateTime(Seconds: longint;RefDate: string): TDateTime;
  function DateTimeToSecondFromRef(DateTime: TDateTime; RefDate: string): longint;

implementation

const
  DaysInMonths: array [1..12] of word = (31,28,31,30,31,30,31,31,30,31,30,31);
  SecondPerDay = 86400;

function AdaptSystemTime(SystemTime: TSystemTime): TDateTime;
// Convert SystemDate to DateTime.
// Take in count the special meaning for TimeZone
var
  Year: word;
  Month: word;
  Days: word;
  TmpDate: TDateTime;
begin
  if SystemTime.wDayOfWeek = 0 then
  begin
    Year:= SystemTime.wYear;
    Month:= SystemTime.wMonth;
    // Special meaning: last day of a particular week of the month
    if SystemTime.wDay < 5 then Days:= SystemTime.wDay * 7
    else
    begin
      Days:= DaysInMonths[Month];
      if (Month = 2) and IsLeapYear(Year) then Days:= 29;
    end;
    TmpDate:= EncodeDate(Year,Month,Days);
    // Find the last day of that week
    while DayOfWeek(TmpDate) > 1 do
    begin
      dec(Days);
      TmpDate:= EncodeDate(Year,Month,Days);
    end;
    with SystemTime do
      TmpDate:= TmpDate + EncodeTime(wHour,wMinute,wSecond,wMilliseconds);
    result:= TmpDate;
  end
  else
    result:= SystemTimeToDateTime(SystemTime);
end;

function UTCToDateTime(UTC: longint): TDateTime;
// Return TDateTime corresponding to UTC
// UTC is number of seconds elapse since 1 January 1970 00:00:00
// TDateTime integer parts is number of day since 31 december 1899 00:00:00
// TDateTime decimal parts is the elapse part of the current day
// 25569 = TDateTime of 1 January 1970 00:00:00
begin
  result:= UTC / SecondPerDay;
  result:= result + 25569;
end;

function DateTimeToUTC(DateTime: TDateTime): longint;
// Return UTC corresponding to DateTime
// UTC is number of seconds elapse since 1 January 1970 00:00:00
// TDateTime integer parts is number of day since 31 december 1899 00:00:00
// TDateTime decimal parts is the elapse part of the current day
// 25569 = TDateTime of 1 January 1970 00:00:00
begin
  result:= trunc((DateTime - 25569) * SecondPerDay);
end;

function UTCToLocalDateTime(UTC: longint): TDateTime;
// Return the local DateTime corresponding to UTC
// UTC is number of seconds elapse since 1 January 1970 00:00:00
var
  LocalDateTime: TDateTime;
  TimeZoneInfo: TTimeZoneInformation;
  Year: word;
  Month: word;
  Day: word;
  StandardDateTime: TDateTime;
  DayLightDateTime: TDateTime;
begin
  GetTimeZoneInformation(TimeZoneInfo);
  // Convert from UTC
  LocalDateTime:= UTCToDateTime(UTC);
  // Compensate bias to get LocalDateTime
  LocalDateTime:= LocalDateTime - ((TimeZoneInfo.Bias*60)/SecondPerDay);
  // Find the year
  DecodeDate(LocalDateTime,Year,Month,Day);
  // Correct TimeZoneInfo for that Year
  TimeZoneInfo.StandardDate.wYear:= Year;
  TimeZoneInfo.DayLightDate.wYear:= Year;
  // Calculate corresponding TDateTime
  StandardDateTime:= AdaptSystemTime(TimeZoneInfo.StandardDate);
  DayLightDateTime:= AdaptSystemTime(TimeZoneInfo.DayLightDate);
  // Compensate with the corresponding bias
  if (LocalDateTime >= DayLightDateTime) and (LocalDateTime < StandardDateTime) then
    LocalDateTime:= LocalDateTime - ((TimeZoneInfo.DayLightBias*60)/SecondPerDay)
  else
    LocalDateTime:= LocalDateTime - ((TimeZoneInfo.StandardBias*60)/SecondPerDay);
  result:= LocalDateTime;
end;

function LocalDateTimeToUTC(LocalDateTime: TDateTime): longint;
var
  DateTime: TDateTime;
  TimeZoneInfo: TTimeZoneInformation;
  Year: word;
  Month: word;
  Day: word;
  StandardDateTime: TDateTime;
  DayLightDateTime: TDateTime;
begin
  GetTimeZoneInformation(TimeZoneInfo);
  // Find the year
  DecodeDate(LocalDateTime,Year,Month,Day);
  // Correct TimeZoneInfo for that Year
  TimeZoneInfo.StandardDate.wYear:= Year;
  TimeZoneInfo.DayLightDate.wYear:= Year;
  // Calculate corresponding TDateTime
  StandardDateTime:= AdaptSystemTime(TimeZoneInfo.StandardDate);
  DayLightDateTime:= AdaptSystemTime(TimeZoneInfo.DayLightDate);
  // Compensate with the corresponding bias
  StandardDateTime:= StandardDateTime + ((TimeZoneInfo.StandardBias*60)/SecondPerDay);
  DayLightDateTime:= DayLightDateTime + ((TimeZoneInfo.DayLightBias*60)/SecondPerDay);
  // Correct if needed
  if (LocalDateTime >= DayLightDateTime) and (LocalDateTime < StandardDateTime) then
    DateTime:= LocalDateTime + ((TimeZoneInfo.DayLightBias*60)/SecondPerDay)
  else
    DateTime:= LocalDateTime + ((TimeZoneInfo.StandardBias*60)/SecondPerDay);
  // Compensate bias to get UTCDateTime
  DateTime:= DateTime + ((TimeZoneInfo.Bias*60)/SecondPerDay);
  // Convert to UTC
  result:= DateTimeToUTC(DateTime);
end;

function SecondFromRefToDateTime(Seconds: longint;RefDate: string): TDateTime;
// Return TDateTime corresponding to Seconds elapse since RefDate
// RefDate must be in the ShortDateFormat and LongTimeFormat 01/01/2001 10:05:02
begin
  result:= Seconds / SecondPerDay;
  result:= result + StrToDateTime(RefDate);
end;

function DateTimeToSecondFromRef(DateTime: TDateTime; RefDate: string): longint;
// Return number of seconds elapse since RefDate corresponding to DateTime
// RefDate must be in the ShortDateFormat and LongTimeFormat 01/01/2001 10:05:02
begin
  result:= trunc((DateTime - StrToDateTime(RefDate)) * SecondPerDay);
end;

end.
