#!/usr/bin/env tclsh
# -*- mode: tcl; coding: utf-8-unix; -*-
# Time-stamp: <2016-05-14 22:35:50 alain>%
# $Id: netinfo_win_9x.tcl,v 1.8 2009-01-25 17:59:01 cvsalain Exp $
#-------------------------------------------------------------------------------
# 20090122 version 1.1
#   Addition of differenciation between Microsoft workgroup and domain
# 20090124 version 1.2
#   Addition of operating system support level
# 20090125 version 1.3
#   Windows 98 support working
#   All informations are got in one query
#
#   Todo: detect if we are in a Microsoft domain and corresponding domain name
#-------------------------------------------------------------------------------

package provide netinfo_win_9x 1.3

if {![info exists ::netinfo]} {
	# Adjust auto_path to contain script home and current directory
	set script_home [string trimright [file dirname [info script]] ./]
	set script_home [file join [pwd] $script_home]
	lappend auto_path $script_home
}

package require netinfo_common
package require registry
package require debug

set ::net_info::workgroupKey \
	"HKEY_LOCAL_MACHINE\\System\\CurrentControlSet\\Services\\VxD\\VNETSUP\\"

#-------------------------------------------------------------------------------
# Return specific value from specified key in registry
#-------------------------------------------------------------------------------
proc ::net_info::getValue {key valueName} {
	if [catch {registry get $key $valueName} value] {
		::debug::msg 1 "Unable to read value \"$valueName\" in key \"$key\""
		return ""
	} else {
		return $value
	}
}

#-------------------------------------------------------------------------------
# Return level of support for that operationg system
# It can be full, partial or none
# Partial as we did not succeed to detect interface boards, if we are on a
# Microsoft domain or workgroup
#-------------------------------------------------------------------------------
proc ::net_info::getSupportLevel {} {
	return "partial"
}

#-------------------------------------------------------------------------------
# Return computer operating system name
#-------------------------------------------------------------------------------
proc ::net_info::getOS {} {
	if { $::tcl_platform(os) == "Windows 95" } {
		switch -exact -- $::tcl_platform(osVersion) \
			"4.0" { return "Windows 95" } \
			"4.10" { return "Windows 98" } \
			"4.90" { return "Windows Millenium" } \
			default { return $::tcl_platform(os) }
	} else {
		return $::tcl_platform(os)
	}
}

#-------------------------------------------------------------------------------
# Return computer operating system version
#-------------------------------------------------------------------------------
proc ::net_info::getOSVersion {} {
	return $::tcl_platform(osVersion)
}

#-------------------------------------------------------------------------------
# Return computer current logged user
#-------------------------------------------------------------------------------
proc ::net_info::getCurrentUser {} {
	return $::tcl_platform(user)
}

#-------------------------------------------------------------------------------
# Return computer name
#-------------------------------------------------------------------------------
proc ::net_info::getComputerName {} {
	return [info hostname]
}

#-------------------------------------------------------------------------------
# Return computer dns domain name
#-------------------------------------------------------------------------------
proc ::net_info::getDnsDomainName {fullname} {
	set name "[info hostname]."
	return [string map -nocase [list $name ""] $fullname]
}

#-------------------------------------------------------------------------------
# Return if computer is in a Microsoft domain
#-------------------------------------------------------------------------------
proc ::net_info::isInMicrosoftDomain {} {
	return false
}

#-------------------------------------------------------------------------------
# Return Microsoft workgroup name
#-------------------------------------------------------------------------------
proc ::net_info::getMicrosoftWorkgroupName {} {
	return [ ::net_info::getValue $::net_info::workgroupKey "Workgroup" ]
}

#-------------------------------------------------------------------------------
# Return Microsoft domain name
#-------------------------------------------------------------------------------
proc ::net_info::getMicrosoftDomainName {} {
	return [ ::net_info::getValue $::net_info::workgroupKey "Workgroup" ]
}

#-------------------------------------------------------------------------------
# Return value after the column
#-------------------------------------------------------------------------------
proc ::net_info::extractValue {line} {
	set pos [expr {[string first ":" $line] + 2}]
	return [string range $line $pos end]
}

#-------------------------------------------------------------------------------
# Return all informations regarding computer and interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInfos {} {
	array set Computer [::net_info::initComputer]

	if { [catch { eval ::exec -- "ipconfig /all" } RawData ] } {
		set Datas [ split $RawData \n ]
		# Remove empty lines
		set Lines {}
		foreach Data $Datas {
			if {$Data != ""} {
				lappend Lines $Data
			}
		}

		set Max [llength $Lines]
		if { $Max > 7} {
			# Computer datas
			#  0 title
			#  1 hostname
			#  2 DNS server
			#  3 node type
			#  4 NetBios ID
			#  5 IP routing
			#  6 Wins proxy
			#  7 NetBios resolution by DNS
			set Computer(Os)             [::net_info::getOS]
			set Computer(OsVersion)      [::net_info::getOSVersion]
			set Computer(OsSupportLevel) [::net_info::getSupportLevel]
			set Computer(Username)       [::net_info::getCurrentUser]
			set Computer(Hostname)       [::net_info::getComputerName]
			set Computer(DnsServer)      [::net_info::extractValue [lindex $Lines 2]]
			set Computer(DnsDomainName)  [::net_info::getDnsDomainName \
											  [::net_info::extractValue [lindex $Lines 1]]]
			if {[::net_info::isInMicrosoftDomain]} {
				set Computer(MicrosoftDomain) true
				set Computer(MicrosoftDomainType) domain
				set Computer(MicrosoftDomainName) [::net_info::getMicrosoftDomainName]
			} else {
				set Computer(MicrosoftDomain) false
				set Computer(MicrosoftDomainType) workgroup
				set Computer(MicrosoftDomainName) [::net_info::getMicrosoftWorkgroupName]
			}

			set n 8
			set Interfaces {}
			# Interfaces datas
			while {[expr {$n + 12}] <= $Max} {
				# For each interface
				#  0 interface number and title
				#  1 description
				#  2 mac address
				#  3 DHCP
				#  4 IP address
				#  5 subnet mask
				#  6 gateway
				#  7 DHCP server
				#  8 main WINS server
				#  9 secondary WINS server
				# 10 start date
				# 11 expiration date
				array set Interface [::net_info::initInterface]
				set Interface(Id) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 1}]]]
				set Interface(Description) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 1}]]]
				set Interface(MacAddress) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 2}]]]
				set Interface(IpAddress) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 4}]]]
				set Interface(SubnetMask) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 5}]]]
				set Interface(Gateway) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 6}]]]
				set dhcp [::net_info::extractValue [lindex $Lines [expr {$n + 3}]]]
				if {[string index $dhcp 0] eq "n"} {
					set Interface(Dhcp) {0}
				} else {
					set Interface(Dhcp) {1}
				}
				set Interface(DhcpServer) \
					[::net_info::extractValue [lindex $Lines [expr {$n + 7}]]]

				lappend Interfaces [array get Interface]
				incr n 12
			}
		}
	}
	return [list Computer [array get Computer] Interfaces $Interfaces]
}

#-------------------------------------------------------------------------------
# If that file is called directly, display test suite on command line
#-------------------------------------------------------------------------------

if {![info exist ::netinfo]} {
	::net_info::testSuite
}
