#!/usr/bin/env tclsh
# -*- mode: tcl; coding: utf-8-unix; -*-
# Time-stamp: <2016-05-14 22:31:42 alain>%
# $Id: netinfo_linux.tcl,v 1.10 2009-05-27 13:53:30 cvsalain Exp $
#-------------------------------------------------------------------------------
# 20090122 version 1.1
#   Addition of differenciation between Microsoft workgroup and domain
# 20090124 version 1.2
#   Addition of operating system support level
# 20090125 version 1.3
#   Collect all informations in only one call
#-------------------------------------------------------------------------------

package provide netinfo_linux 1.3

if {![info exists ::netinfo]} {
	# Adjust auto_path to contain script home and current directory
	set script_home [string trimright [file dirname [info script]] ./]
	set script_home [file join [pwd] $script_home]
	lappend auto_path $script_home
}

package require netinfo_common

#-------------------------------------------------------------------------------
# Remove extra spaces
#-------------------------------------------------------------------------------
proc ::net_info::removeSpaces {value} {
	set value [ string trim $value ]
	while { [ string first "  " $value ] != -1 } {
		set value [ string map { "  " " " } $value ]
	}
	return $value
}

#-------------------------------------------------------------------------------
# Return level of support for that operationg system
# It can be full, partial or none
# Partial as Microsoft domain, domain name and workgroup name are probably wrong
#-------------------------------------------------------------------------------
proc ::net_info::getSupportLevel {} {
	return "partial"
}

#-------------------------------------------------------------------------------
# Return computer operating system name
#-------------------------------------------------------------------------------
proc ::net_info::getOS {} {
	return $::tcl_platform(os)
}

#-------------------------------------------------------------------------------
# Return computer operating system version
#-------------------------------------------------------------------------------
proc ::net_info::getOSVersion {} {
	return $::tcl_platform(osVersion)
}

#-------------------------------------------------------------------------------
# Return computer current logged user
#-------------------------------------------------------------------------------
proc ::net_info::getCurrentUser {} {
	return $::tcl_platform(user)
}

#-------------------------------------------------------------------------------
# Return computer name
#-------------------------------------------------------------------------------
proc ::net_info::getComputerName {} {
	return [info hostname]
}

#-------------------------------------------------------------------------------
# Return if computer is in a Microsoft domain
#-------------------------------------------------------------------------------
proc ::net_info::isInMicrosoftDomain {} {
	set result ""
	if { $::net_info::sambaFile != "" } {
		if { [ catch { open $::net_info::sambaFile "r" } fileid ] } {
			::debug::msg 1 "Unable to open \"$::net_info::sambaFile\""
		} else {
			set contents [ split [ read $fileid ] "\n" ]
			close $fileid
			foreach line $contents {
				set line [ string trim $line ]
				if { [string first "#" $line] != 0 } {
					set pos [ string first "domain logons" $line ]
					if { $pos > -1 } {
						set result [ string range \
										 $line \
										 [expr {[ string first "=" $line ]+1}]\
										 end]
					}
				}
			}
		}

	}
	return [string match -nocase "yes" [string trim $result ]]
}

#-------------------------------------------------------------------------------
# Return Microsoft workgroup name
#-------------------------------------------------------------------------------
proc ::net_info::getMicrosoftWorkgroupName {} {
	set result ""
	if { $::net_info::sambaFile != "" } {
		if { [ catch { open $::net_info::sambaFile "r" } fileid ] } {
			::debug::msg 1 "Unable to open \"$::net_info::sambaFile\""
		} else {
			set contents [ split [ read $fileid ] "\n" ]
			close $fileid
			foreach line $contents {
				set line [ string trim $line ]
				if { [string first "#" $line] != 0 } {
					set pos [ string first "workgroup" $line ]
					if { $pos > -1 } {
						set result [ string range \
										 $line \
										 [expr {[ string first "=" $line ]+1}]\
										 end]
					}
				}
			}
		}

	}
	return [ string trim $result ]
}

#-------------------------------------------------------------------------------
# Return Microsoft domain name
#-------------------------------------------------------------------------------
proc ::net_info::getMicrosoftDomainName {} {
	set result ""
	if { $::net_info::sambaFile != "" } {
		if { [ catch { open $::net_info::sambaFile "r" } fileid ] } {
			::debug::msg 1 "Unable to open \"$::net_info::sambaFile\""
		} else {
			set contents [ split [ read $fileid ] "\n" ]
			close $fileid
			foreach line $contents {
				set line [ string trim $line ]
				if { [string first "#" $line] != 0 } {
					set pos [ string first "workgroup" $line ]
					if { $pos > -1 } {
						set result [ string range \
										 $line \
										 [expr {[ string first "=" $line ]+1}]\
										 end]
					}
				}
			}
		}

	}
	return [ string trim $result ]
}


#-------------------------------------------------------------------------------
# Return dns domain name
#-------------------------------------------------------------------------------
proc ::net_info::getDnsDomainName {} {
	set domain ""
	if { $::net_info::resolvconfFile != "" } {
		if { [ catch { open $::net_info::resolvconfFile "r" } fileid ] } {
			::debug::msg 1 "Unable to open \"$::net_info::resolvconfFile\""
		} else {
			set contents [ split [ read $fileid ] "\n" ]
			close $fileid
			foreach line $contents {
				set line [ string trim $line ]
				if { [string first "#" $line] != 0 } {
					set pos [ string first "search" $line ]
					if { $pos > -1 } {
						set domain [ string range $line \
										 [ expr { $pos + 6 } ] end]
					}
				}
			}
		}

	}
	return [ string trim $domain ]
}


#-------------------------------------------------------------------------------
# Return domain name servers
#-------------------------------------------------------------------------------
proc ::net_info::getDnsServer {} {
	set dns ""
	if { $::net_info::resolvconfFile != "" } {
		if { [ catch { open $::net_info::resolvconfFile "r" } fileid ] } {
			::debug::msg 1 "Unable to open \"$::net_info::resolvconfFile\""
		} else {
			set contents [ split [ read $fileid ] "\n" ]
			close $fileid
			foreach line $contents {
				set line [ string trim $line ]
				if { [string first "#" $line] != 0 } {
					set pos [ string first "nameserver" $line ]
					if { $pos > -1 } {
						set dns [ string range $line [ expr { $pos + 10 } ] end]
					}
				}
			}
		}

	}
	return [ string trim $dns ]
}

#-------------------------------------------------------------------------------
# Return corresponding mask in its dotted version
#-------------------------------------------------------------------------------
proc ::net_info::convertMask {value} {
	set binaryMask [ string repeat "1" $value ]
	append binaryMask [ string repeat "0" [ expr { 32 - $value } ] ]
	binary scan [ binary format B* $binaryMask ] c1c1c1c1 mask1 mask2 mask3 mask4
	set mask "[ expr ( $mask1 + 0x100 ) % 0x100 ]."
	append mask "[ expr ( $mask2 + 0x100 ) % 0x100 ]."
	append mask "[ expr ( $mask3 + 0x100 ) % 0x100 ]."
	append mask "[ expr ( $mask4 + 0x100 ) % 0x100 ]"
	return $mask
}

#-------------------------------------------------------------------------------
# Return mac address corresponding to that interface id
#-------------------------------------------------------------------------------
proc ::net_info::getMac {id} {
	set mac ""
	if { $::net_info::ipPrg != "" } {
		catch { eval ::exec -- $::net_info::ipPrg \
					-0 addr show $id | grep link } linkData
		set mac [ lindex [ split [ string trim $linkData ] ] 1 ]
	}
	return [ string toupper $mac ]
}

#-------------------------------------------------------------------------------
# Return gateway corresponding to that interface id
#-------------------------------------------------------------------------------
proc ::net_info::getGateway {id} {
	# find program to use for gateway
	if { $::net_info::routePrg != "" } {
		# get gateway
		catch { eval ::exec -- $::net_info::routePrg \
					-n | grep UG | grep $id } routeData
		set routeData [ ::net_info::removeSpaces $routeData ]
		set route [ split [ string trim $routeData ] ]
		return [ lindex $route 1 ]
	} else {
		return ""
	}
}

#-------------------------------------------------------------------------------
# Return if that interface id use dhcp
#-------------------------------------------------------------------------------
proc ::net_info::getDhcp {id} {
	set dhcp 0
	if { [ file exists $::net_info::interfacesFile ] } {
		catch { eval ::exec -- cat $::net_info::interfacesFile \
					| grep iface | grep $id } dhcpData
		if { [string first "#" $dhcpData] != 0 } {
			if { [ string first "dhcp" $dhcpData ] != -1 } {
				set dhcp 1
			}
		}
	} else {
		::debug::msg 1 "Can't find \"interfaces\" file"
	}

	return $dhcp
}

#-------------------------------------------------------------------------------
# Return all informations regarding a computer
#-------------------------------------------------------------------------------
proc ::net_info::getComputer {} {
	array set Computer [::net_info::initComputer]

	set Computer(Os)             [::net_info::getOS]
	set Computer(OsVersion)      [::net_info::getOSVersion]
	set Computer(OsSupportLevel) [::net_info::getSupportLevel]
	set Computer(Username)       [::net_info::getCurrentUser]
	set Computer(Hostname)       [::net_info::getComputerName]
	set Computer(DnsServer)      [::net_info::getDnsServer]
	set Computer(DnsDomainName)  [::net_info::getDnsDomainName]
	if {[::net_info::isInMicrosoftDomain]} {
		set Computer(MicrosoftDomain) 1
		set Computer(MicrosoftDomainType) domain
		set Computer(MicrosoftDomainName) \
			[::net_info::getMicrosoftDomainName]
	} else {
		set Computer(MicrosoftDomain) 0
		set Computer(MicrosoftDomainType) workgroup
		set Computer(MicrosoftDomainName) \
			[::net_info::getMicrosoftWorkgroupName]
	}

	return [array get Computer]
}

#-------------------------------------------------------------------------------
# Return all informations regarding all interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInterfaces {} {
	set Interfaces {}
	if { $::net_info::ipPrg != "" } {
		catch { eval ::exec -- $::net_info::ipPrg \
					-4 addr show | grep inet } inetDatas
		foreach line [ split $inetDatas "\n" ] {
			set data [ split [ string trim $line ] ]
			set id [ lindex $data end ]
			if { $id != "lo" } {
				set ipData [ split [ lindex $data 1 ] "/" ]
				set ip [ lindex $ipData 0 ]
				set mask [ ::net_info::convertMask [ lindex $ipData 1 ] ]
				# result
				array set Interface [::net_info::initInterface]
				set Interface(Id) $id
				set Interface(Description) $id
				set Interface(MacAddress) [::net_info::getMac $id]
				set Interface(IpAddress) $ip
				set Interface(SubnetMask) $mask
				set Interface(Gateway) [::net_info::getGateway $id]
				set Interface(Dhcp) [::net_info::getDhcp $id]
				set Interface(DhcpServer) "????"

				lappend Interfaces [array get Interface]
			}
		}
	}
	return $Interfaces
}

#-------------------------------------------------------------------------------
# Return all informations regarding computer and interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInfos {} {
	return [list Computer [::net_info::getComputer] \
				Interfaces [::net_info::getInterfaces]]
}

#-------------------------------------------------------------------------------
# Detect program and files to be used
#-------------------------------------------------------------------------------
proc ::net_info::detectDataSources {} {
	# Ip
	set ::net_info::ipPrg ""
	if { [ file executable /sbin/ip ] } {
		set ::net_info::ipPrg "/sbin/ip"
	} elseif { [ file executable /usr/sbin/ip ] } {
		set ::net_info::ipPrg "/usr/sbin/ip"
	} else {
		::debug::msg 1 "Can't find \"ip\" executable"
	}

	# Route
	set ::net_info::routePrg ""
	if { [ file executable /sbin/route2 ] } {
		set ::net_info::routePrg "/sbin/route"
	} elseif { [ file executable /usr/sbin/route2 ] } {
		set ::net_info::routePrg "/usr/sbin/route"
	} else {
		::debug::msg 1 "Can't find \"route\" executable"
	}

	# Interfaces
	set ::net_info::interfacesFile ""
	if { [ file exists /etc/network/interfaces ] } {
		set ::net_info::interfacesFile "/etc/network/interfaces"
	}

	# Resolv.conf
	set ::net_info::resolvconfFile ""
	if { [ file exists /etc/resolv.conf ] } {
		set ::net_info::resolvconfFile "/etc/resolv.conf"
	}
	# Smb.conf
	set ::net_info::sambaFile ""
	if { [ file exists /etc/samba/smb.conf ] } {
		set ::net_info::sambaFile "/etc/samba/smb.conf"
	}
}


#-------------------------------------------------------------------------------
# Initialization
#-------------------------------------------------------------------------------

::net_info::detectDataSources

#-------------------------------------------------------------------------------
# If that file is called directly, display test suite on command line
#-------------------------------------------------------------------------------

if {![info exist ::netinfo]} {
	::net_info::testSuite
}
