#!/usr/bin/env tclsh
# -*- mode: tcl; coding: utf-8-unix; -*-
# Time-stamp: <2016-05-14 22:33:18 alain>%
# $Id: netinfo_darwin.tcl,v 1.1 2009-05-27 13:56:46 cvsalain Exp $
#-------------------------------------------------------------------------------
# 20090527 version 1.0
#   First realease for Darwin
#-------------------------------------------------------------------------------

package provide netinfo_darwin 1.0

if {![info exists ::netinfo]} {
	# Adjust auto_path to contain script home and current directory
	set script_home [string trimright [file dirname [info script]] ./]
	set script_home [file join [pwd] $script_home]
	lappend auto_path $script_home
}

package require netinfo_common

#-------------------------------------------------------------------------------
# Return level of support for that operationg system
# It can be full, partial or none
#-------------------------------------------------------------------------------
proc ::net_info::getSupportLevel {} {
	return "partial"
}

#-------------------------------------------------------------------------------
# Return computer operating system name
#-------------------------------------------------------------------------------
proc ::net_info::getOS {} {
	return $::tcl_platform(os)
}

#-------------------------------------------------------------------------------
# Return computer operating system version
#-------------------------------------------------------------------------------
proc ::net_info::getOSVersion {} {
	return $::tcl_platform(osVersion)
}

#-------------------------------------------------------------------------------
# Return computer current logged user
#-------------------------------------------------------------------------------
proc ::net_info::getCurrentUser {} {
	return $::tcl_platform(user)
}

#-------------------------------------------------------------------------------
# Return computer name
#-------------------------------------------------------------------------------
proc ::net_info::getComputerName {} {
	return [info hostname]
}

#-------------------------------------------------------------------------------
# Return all informations regarding a computer
#-------------------------------------------------------------------------------
proc ::net_info::getComputer {} {
	array set Computer [::net_info::initComputer]

	set Computer(Os)             [::net_info::getOS]
	set Computer(OsVersion)      [::net_info::getOSVersion]
	set Computer(OsSupportLevel) [::net_info::getSupportLevel]
	set Computer(Username)       [::net_info::getCurrentUser]
	set Computer(Hostname)       [::net_info::getComputerName]

	return [array get Computer]
}

#-------------------------------------------------------------------------------
# Return corresponding mask in its dotted version
#-------------------------------------------------------------------------------
proc convertMask {value} {
	binary scan [binary format H8 [string range $value 2 end]] c1c1c1c1 mask1 mask2 mask3 mask4
	set mask "[ expr ( $mask1 + 0x100 ) % 0x100 ]."
	append mask "[ expr ( $mask2 + 0x100 ) % 0x100 ]."
	append mask "[ expr ( $mask3 + 0x100 ) % 0x100 ]."
	append mask "[ expr ( $mask4 + 0x100 ) % 0x100 ]"
 	return $mask
}

#-------------------------------------------------------------------------------
# Return all informations regarding all interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInterfaces {} {
	set Interfaces {}
	array unset Interface

	catch { eval ::exec -- ifconfig } ifconfig
	if {$ifconfig != ""} {
		set lines [ split $::contents "\n" ]
		foreach line $lines {
			set data [split $line " "]
			if {[string range $line 0 0] == "\t"} {
				# Mac address
				set pos [lsearch $data "\tether"]
				if {$pos != -1} {
					incr pos
					set Interface(MacAddress) [lindex $data $pos]
				}
 				# Ip address V4
				set pos [lsearch $data "\tinet"]
				if {$pos != -1} {
					incr pos
					set Interface(IpAddress) [lindex $data $pos]
				}
				# Subnet mask
				set pos [lsearch $data "netmask"]
				if {$pos != -1} {
					incr pos
					set Interface(SubnetMask) [convertMask [lindex $data $pos]]
				}
 				# Ip address V6
				set pos [lsearch $data "\tinet6"]
				if {$pos != -1} {
					incr pos
					set Interface(IpV6) [lindex [split [lindex $data $pos] "%"] 0]
				}
			} else {
				if {[array exists Interface]} {
					if {($Interface(IpAddress) != {}) \
							&& ([string range $Interface(Id) 0 1] != "lo")} {
						lappend Interfaces [array get Interface]
					}
					array unset Interface
				}
				set id [lindex $data 0 0]
				if {[string range $id end end] == ":" } {
					# new interface
					array set Interface [::net_info::initInterface]
					set Interface(Id) [string range $id 0 end-1]
				}
			}
		}
	}
	# Add remaining data
	if {[array exists Interface]} {
		if {($Interface(IpAddress) != {}) \
				&& ([string range $Interface(Id) 0 1] != "lo")} {
			lappend Interfaces [array get Interface]
		}
		array unset Interface
	}
	return $Interfaces
}

#-------------------------------------------------------------------------------
# Return all informations regarding computer and interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInfos {} {
	return [list Computer [::net_info::getComputer] \
				Interfaces [::net_info::getInterfaces]]
}

#-------------------------------------------------------------------------------
# If that file is called directly, display test suite on command line
#-------------------------------------------------------------------------------

if {![info exist ::netinfo]} {
	::net_info::testSuite
}
