#!/usr/bin/env tclsh
# -*- mode: tcl; coding: latin-9-unix; -*-
# Time-stamp: <2016-05-14 22:34:03 alain>%
# $Id: netinfo_common.tcl,v 1.10 2009-05-27 13:53:55 cvsalain Exp $
#-------------------------------------------------------------------------------
# Common part for all netinfo_ packages
# All platform specifc package must supply code for all exported procedure
#-------------------------------------------------------------------------------
# 20090125 version 1.3
#   Informations are now collected in only one call
# 20090527 version 1.4
#   Addition of partial Darwin support
#   Addition of IP V6 address
#-------------------------------------------------------------------------------

package provide netinfo_common 1.4

if {![info exists ::netinfo]} {
	# Adjust auto_path to contain script home and current directory
	set script_home [string trimright [file dirname [info script]] ./]
	set script_home [file join [pwd] $script_home]
	lappend auto_path $script_home
}

package require debug

namespace eval ::net_info:: {
	namespace export getInfos
	# Array which contain:
	# - a Computer array with computer informations
	#   + operating system
	#   + operating system version
	#   + operating system support level by netinfo
	#   + current username
	#   + computer hostname
	#   + dns domain name
	#   + dns server
	#   + Microsoft domain (0 or 1)
	#   + Microsoft domain type (domain or workgroup)
	#   + Microsoft domain name
	# - an Interfaces list which contain for each interface
	#   + interface identification
	#   + interface description
	#   + mac address
	#   + ip address
	#   + subnet mask
	#   + gateway
	#   + dhcp (0 or 1)
	#   + dhcp server
}

# Platform specific packages
switch -exact -- $::tcl_platform(os) \
	"Windows 95" { package require netinfo_win_9x } \
	"Windows NT" { package require netinfo_win_nt } \
	"Linux" { package require netinfo_linux } \
	"Darwin" { package require netinfo_darwin } \
	default {package require netinfo_generic }


#-------------------------------------------------------------------------------
# Initialize a computer array
#-------------------------------------------------------------------------------
proc ::net_info::initComputer {} {
	return [list \
				Os                  {} \
				OsVersion           {} \
				OsSupportLevel      {none} \
				Username            {} \
				Hostname            {} \
				DnsDomainName       {} \
				DnsServer           {} \
				MicrosoftDomain     {0} \
				MicrosoftDomainType {} \
				MicrosoftDomainName {} \
			   ]
}

#-------------------------------------------------------------------------------
# Initialize an interface array
#-------------------------------------------------------------------------------
proc ::net_info::initInterface {} {
	return [ list \
				 Id          {} \
				 Description {} \
				 MacAddress  {} \
				 IpAddress   {} \
				 SubnetMask  {} \
				 IpV6        {} \
				 Gateway     {} \
				 Dhcp        {?} \
				 DhcpServer  {} \
				]
}

#-------------------------------------------------------------------------------
# Format output data
#-------------------------------------------------------------------------------
proc ::net_info::frmData {title value size} {
	set frmString [ string repeat " ." $size ]
	return "[ string replace $frmString 0 \
      [expr {[ string length $title ]-1}] $title ] \
		: $value"
}

#-------------------------------------------------------------------------------
# Build information text
#-------------------------------------------------------------------------------
proc ::net_info::infoText {title} {
	array set Infos [::net_info::getInfos]
	array set Computer $Infos(Computer)
	set Interfaces $Infos(Interfaces)

	set result "\n"
	append result $title\n
	append result [ string repeat "-" [ string length $title ] ]\n
	append result "\n"
	append result [ ::net_info::frmData "Operating system" \
						$Computer(Os) 15 ]\n
	append result [ ::net_info::frmData "OS version" \
						$Computer(OsVersion) 15 ]\n
	append result [ ::net_info::frmData "OS support level" \
						$Computer(OsSupportLevel) 15 ]\n
	append result "\n"
	append result [ ::net_info::frmData "Current user" \
						$Computer(Username) 15 ]\n
	append result [ ::net_info::frmData "Computer name" \
						$Computer(Hostname) 15 ]\n
	append result [ ::net_info::frmData "  Dns domain name" \
						$Computer(DnsDomainName) 15 ]\n
	append result [ ::net_info::frmData "  Dns server" \
						$Computer(DnsServer) 15 ]\n
	append result [ ::net_info::frmData "  Microsoft domain" \
						$Computer(MicrosoftDomain) 15 ]\n
	append result [ ::net_info::frmData "  Microsoft domain type" \
						$Computer(MicrosoftDomainType) 15 ]\n
	append result [ ::net_info::frmData "  Microsoft domain name" \
						$Computer(MicrosoftDomainName) 15 ]\n
	append result "\n"
	foreach interfaceData $Interfaces {
		array set interface $interfaceData
		append result [ ::net_info::frmData "Interface" \
							$interface(Id) 15 ]\n
		append result [ ::net_info::frmData "  Description" \
							$interface(Description) 15 ]\n
		append result [ ::net_info::frmData "  Mac address" \
							$interface(MacAddress) 15 ]\n
		append result [ ::net_info::frmData "  Ip address" \
							$interface(IpAddress) 15 ]\n
		append result [ ::net_info::frmData "  Subnet mask" \
							$interface(SubnetMask) 15 ]\n
		append result [ ::net_info::frmData "  Gateway" \
							$interface(Gateway) 15 ]\n
		append result [ ::net_info::frmData "  Ip V6" \
							$interface(IpV6) 15 ]\n
		append result "\n"
		append result [ ::net_info::frmData "  Dhcp" $interface(Dhcp) 15 ]\n
		append result [ ::net_info::frmData "  Dhcp server" \
							$interface(DhcpServer) 15 ]\n
		append result "\n"
	}
	return $result
}

#-------------------------------------------------------------------------------
# Test suite to check package
#-------------------------------------------------------------------------------
proc ::net_info::testSuite {} {
	puts [ ::net_info::infoText "Running $::argv0 test suite" ]
}

#-------------------------------------------------------------------------------
# Save interface informations in a text file
#-------------------------------------------------------------------------------
proc ::net_info::saveToFile {filename title} {
	if { [ catch { open $filename "w" } fileid ] } {
		::debug::msg 1 "Unable to open \"$filename\" to write data"
		return false
	} else {
		puts $fileid [ ::net_info::infoText $title ]
		close $fileid
		return true
	}
}
