#
# Copyright (c) 2006, 2007 Ashok P. Nadkarni
# All rights reserved.
#
# See the file LICENSE for license

# TBD - object identity comparison 
#   - see http://blogs.msdn.com/ericlippert/archive/2005/04/26/412199.aspx

namespace eval twapi {
    # Maps TYPEKIND data values to symbols
    array set _typekind_map {
        0 enum
        1 record
        2 module
        3 interface
        4 dispatch
        5 coclass
        6 alias
        7 union
    }

    # Cache of Interface names - IID mappings
    array set _iid_to_name_cache {
    }
    array set _name_to_iid_cache {
        idispatch {{00020400-0000-0000-C000-000000000046}}
        iunknown  {{00000000-0000-0000-C000-000000000046}}
        ipersist  {{0000010c-0000-0000-C000-000000000046}}
        ipersistfile {{0000010b-0000-0000-C000-000000000046}}
        itasktrigger {{148BD52B-A2AB-11CE-B11F-00AA00530503}}
        ischeduleworkitem {{a6b952f0-a4b1-11d0-997d-00aa006887ec}}
        itask {{148BD524-A2AB-11CE-B11F-00AA00530503}}
        ienumworkitems {{148BD528-A2AB-11CE-B11F-00AA00530503}}
        itaskscheduler {{148BD527-A2AB-11CE-B11F-00AA00530503}}
        iprovidetaskpage {{4086658a-cbbb-11cf-b604-00c04fd8d565}}
    }

    # Prototype definitions in the format generated by
    # idispatch_fill_prototypes
    array set idispatch_prototypes {}

    # Mapping table from COM instances to related data. We
    # need this level of indirection because multiple object creation
    # can return the same interface pointer if the referenced resource
    # is the same. We need to treat these as separate objects from
    # the application's point of view.
    # Indexed by ($comobj,field) where field may be
    #  ifc  - corresponding idispatch interface
    #  sinks,SINKID - event sinks with id SINKID bound to this object
    #  connpts,SINKID - connection points with id SINKID bound to this object
    array set com_instance_data {}

    # Counter of COM instances - used for name generation
    variable com_instance_counter 0

    # Controls debug checks
    variable com_debug 1
}

# Get the CLSID for a ProgID
proc twapi::progid_to_clsid {progid} {
    return [CLSIDFromProgID $progid]
}

# Get the ProgID for a CLSID
proc twapi::clsid_to_progid {progid} {
    return [ProgIDFromCLSID $progid]
}

# Increment ref count for an interface
proc twapi::iunknown_release {ifc} {
    if {$ifc eq "NULL"} {
        error "NULL interface pointer passed."
    }
    if {$::twapi::com_debug} {
        # Check if we are releaseing once too often
        # We may even crash if the memory has been reclaimed already
        set refs [IUnknown_AddRef $ifc]
        if {$refs >= 2} {
            # Fine. Undo our addref
            IUnknown_Release $ifc
        } else {
            error "Internal error: attempt to release interface that's already released"
            # TBD - should really exit
            # Fall thru to undo a single addref (our addref)
        }
    }
    IUnknown_Release $ifc
}

# Release an interface
proc twapi::iunknown_addref {ifc} {
    if {$ifc eq "NULL"} {
        error "NULL interface pointer passed."
    }
    IUnknown_AddRef $ifc
}

# Query interface
proc twapi::iunknown_query_interface {ifc name_or_iid} {
    if {$ifc eq "NULL"} {
        error "NULL interface pointer passed."
    }
    foreach {iid name} [_resolve_iid $name_or_iid] break
    return [IUnknown_QueryInterface $ifc $iid $name]
}

#
# Get an existing active object interface
proc twapi::get_iunknown_active {clsid} {
    return [GetActiveObject $clsid]
}

#
# Create a new object and get an interface to it
# Generates exception if no such interface
proc twapi::com_create_instance {clsid name_or_iid args} {
    array set opts [parseargs args {
        {model.arg any}
        download.bool
        {disablelog.bool false}
        enableaaa.bool
        {nocustommarshal.bool false}
    } -maxleftover 0]

    # CLSCTX_NO_CUSTOM_MARSHAL ?
    set flags [expr { $opts(nocustommarshal) ? 0x1000 : 0}]

    set model 0
    if {[info exists opts(model)]} {
        foreach m $opts(model) {
            switch -exact -- $m {
                any           {setbits model 23}
                inprocserver  {setbits model 1}
                inprochandler {setbits model 2}
                localserver   {setbits model 4}
                remoteserver  {setbits model 16}
            }
        }
    }

    setbits flags $model

    if {[info exists opts(download)]} {
        if {$opts(download)} {
            setbits flags 0x2000;       # CLSCTX_ENABLE_CODE_DOWNLOAD
        } else {
            setbits flags 0x400;       # CLSCTX_NO_CODE_DOWNLOAD
        }
    }

    if {$opts(disablelog)} {
        setbits flags 0x4000;           # CLSCTX_NO_FAILURE_LOG
    }

    if {[info exists opts(enableaaa)]} {
        if {$opts(enableaaa)} {
            setbits flags 0x10000;       # CLSCTX_ENABLE_AAA
        } else {
            setbits flags 0x8000;       # CLSCTX_DISABLE_AAA
        }
    }

    foreach {iid iid_name} [_resolve_iid $name_or_iid] break

    # In some cases, like Microsoft Office getting an interface other
    # than IUnknown fails fails.
    # We need to get IUnknown, wait for the object to run, and then
    # get the desired interface from IUnknown.
    #  We could check for a specific error code but no guarantee that
    #  the error is same in all versions so we catch and retry on all errors
    if {[catch {set ifc [Twapi_CoCreateInstance $clsid NULL $flags $iid $iid_name]}]} {
        # Try through IUnknown
        set iunk [Twapi_CoCreateInstance $clsid NULL $flags [_iid_iunknown] IUnknown]
        try {
            # Wait for it to run, then get IDispatch from it
            twapi::OleRun $iunk
            set ifc [iunknown_query_interface $iunk $iid]
        } finally {
            iunknown_release $iunk
        }
    }

    return $ifc
}


#
proc twapi::get_iunknown {clsid args} {
    return [eval [list com_create_instance $clsid IUnknown] $args]
}

#
# IDispatch commands


# Get the IDispatch interface for a class
# Generates exception if no such interface
#
proc twapi::get_idispatch {clsid args} {
    return [eval [list com_create_instance $clsid IDispatch] $args]
}

#
# Has type information?
proc twapi::idispatch_has_typeinfo {ifc} {
    return [IDispatch_GetTypeInfoCount $ifc]
}

#
# Returns the type information for a IDispatch interface
proc twapi::idispatch_get_itypeinfo {ifc args} {
    array set opts [parseargs args {
        lcid.int
    } -maxleftover 0 -nulldefault]

    # TBD - what is the second param (0) supposed to be?
    IDispatch_GetTypeInfo $ifc 0 $opts(lcid)
}

#
# Get ids of names
proc twapi::idispatch_names_to_ids {ifc name args} {
    array set opts [parseargs args {
        lcid.int
        paramnames.arg
    } -maxleftover 0 -nulldefault]
    
    
    return [IDispatch_GetIDsOfNames $ifc [concat [list $name] $opts(paramnames)] $opts(lcid)]
}


#
# Invoke an IDispatch function
# prototype should consist of basically params to IDispatch_Invoke - this
# format is happily returned by idispatch_fill_prototypes
proc twapi::idispatch_invoke {ifc prototype args} {
    if {$prototype eq ""} {
        # Treat as a property get DISPID_VALLUE (default value)
        # {dispid=0, riid="" lcid=0 cmd=propget(2) ret type=bstr(8) {} (no params)}
        set prototype {0 {} 0 2 8 {}}
    }
    uplevel 1 [list twapi::IDispatch_Invoke $ifc $prototype] $args
}

#
# NULL comobj object
proc twapi::comobj_null {args} {
    switch -exact -- [lindex $args 0] {
        -isnull    { return true }
        -interface { return NULL }
        -destroy   { return }
        default {
            error "NULL comobj called with arguments <[join $args ,]>."
        }
    }
}

#
# Creates an object command for a COM object from a dispatch interface
# need_addref should be false if the object will own the interface
# and true if the caller will be independently using it (and releasing it)
# as well
proc twapi::comobj_idispatch {ifc need_addref {objclsid ""}} {
    if {$ifc eq "NULL"} {
        return ::twapi::comobj_null
    }

    if {$need_addref} {
        iunknown_addref $ifc
    }

    set objname ::twapi::com_[incr twapi::com_instance_counter]
    set ::twapi::com_instance_data($objname,ifc) $ifc
    interp alias {} $objname {} ::twapi::_comobj_wrapper $objname $objclsid
    return $objname
}

#
# Create a object command for a COM object
# comid is either a CLSID or a PROGID
proc twapi::comobj {comid args} {
    set clsid [_convert_to_clsid $comid]
    return [comobj_idispatch [eval [list get_idispatch $clsid] $args] false $clsid]
}


#
# Returns the "prototypes" for idispatch methods (non-dispatch methods
# are ignored!
#
# $args is a list of names.
#
# Stores prototypes for each function in array $v_protos. 
# The array elements are indexed as 
# ($ifc,$name,$lcid,$invokeflag)  - contains the full prototype in a form
#     that can be passed to idispatch_invoke. This is a list with the
#     elements {DISPID "" LCID INVOKEFLAGS RETTYPE PARAMTYPES}
# Entries are created for every match, so for example if the name
# was a property, entries would be created for both the set property
# and the get property method ($invokeflag would be different in the
# two cases
#
# Returns number of entries found
proc twapi::idispatch_fill_prototypes {ifc v_protos lcid args} {
    upvar $v_protos protos

    array set protos {};                #  Just to make sure array is created

    # Filter out the names we already have
    set names [list ]
    foreach name $args {
        set count [llength [array names protos $ifc,$name,$lcid*]]
        if {$count} {
            # Already have the prototypes
            return $count
        }
    }

    set count 0
    try {
        set ti [idispatch_get_itypeinfo $ifc -lcid $lcid]

        # In case of dual interfaces, we need the typeinfo for the dispatch
        switch -exact -- [lindex [itypeinfo_get_info $ti -typekind] 1] {
            dispatch {
                # Fine, just what we want
            }
            interface {
                # Get the dispatch interface
                set ti2 [itypeinfo_get_referenced_itypeinfo $ti -1]
                iunknown_release $ti
                set ti $ti2
            }
            default {
                error "Interface is not a dispatch interface"
            }
        }

        set tc [itypeinfo_get_itypecomp $ti]
        
        foreach name $args {
            # Check for existence of method, propget, propput
            foreach invkind {1 2 4} {
                if {![catch {
                    set binddata [ITypeComp_Bind $tc $name $invkind $lcid]
                }]} {
                    if {[llength $binddata] == 0} {
                        continue;       # Not found
                    }
                    foreach {type data ti2} $binddata break
                    iunknown_release $ti2; # Don't need this but must release
                    if {$type ne "funcdesc"} continue
                    array set bindings $data
                    set protos($ifc,$name,$lcid,$bindings(invkind)) [list $bindings(memid) "" $lcid $bindings(invkind) $bindings(elemdescFunc.tdesc) $bindings(lprgelemdescParam)]
                    incr count
                }
            }
        }

    } onerror {TWAPI_WIN32 0x80004002} {
        # Interface not supported
        # Ignore the error - we will try below using another method
    } finally {
        if {[info exists tc]} {
            iunknown_release $tc
        }
        if {[info exists ti]} {
            iunknown_release $ti
        }
    }    

    if {$count} {
        return $count
    }

    # No interfaces found. See if we have a IDispatchEx interface that
    # will has dynamic members. Note that these DISPID will work with
    # the original IDispatch as well
    try {
        set dispex [iunknown_query_interface $ifc IDispatchEx]
        if {$dispex ne ""} {
            # flags = 10 - case insensitive, create if required
            set dispid [IDispatchEx_GetDispID $dispex $name 10]
            # No type information is available for dynamic members.
            # Try at least getting the invocation type but even that is not
            # supported by all objects

            # Invoke kind - 1 (method), 2 (propget), 4 (propput)
            set invkinds [list 1 2 4];      # In case call below fails

            # 1+4+10+100
            # We look for the following flags
            #  0x1 - property get
            #  0x4 - property put
            #  0x10 - property putref
            #  0x100 - method call
            if {! [catch {set flags [IDispatchEx_GetMemberProperties $dispex 0x115] }]} {
                set invkinds [list ]
                if {$flags & 0x100} {lappend invkinds 1}
                if {$flags & 0x1} {lappend invkinds 2}
                if {$flags & 0x14} {
                    # TBD - we are marking putref and put the same. Is that OK?
                    lappend invkinds 4
                }
            }

            foreach invkind $invkinds {
                # Note that the last element in prototype is missing indicating
                # we do not have parameter information. Also, we assume return
                # type of 8 (BSTR) (although the actual return type doesn't matter)
                set protos($ifc,$name,$lcid,$invkind) [list $dispid "" $lcid $invkind 8]
                incr count
            }
        }
    } onerror {} {
        # Ignore errors, just means prototypes not filled
    } finally {
        if {[info exists dispex] && $dispex ne ""} {
            iunknown_release $dispex
        }
    }

    return $count
}

# Define a prototype manually in the same format as idispatch_fill_prototypes
# TBD - should we document this?
proc twapi::idispatch_define_prototype {ifc name args} {
    # Parse out options.
    # Return type is assumed 8 (BSTR) but does
    # not matter as automatic type conversion will be done on
    # the return value.
    array set opts [parseargs args {
        {lcid.int 0}
        {type.arg 1 {-get get -set set -call call 1 2 4}}
        {rettype.arg bstr}
        params.arg
    } -maxleftover 0]

    set dispid [lindex [idispatch_names_to_ids $ifc $name] 1]
    if {$dispid eq ""} {
        win32_error 0x80020003 "No property or method found with name '$name'."
    }

    switch -exact -- $opts(type) {
        "call"  -
        "-call" {set flags 1 }
        "get"   -
        "-get" { set flags 2 }
        "set"   -
        "-set" { set flags 4 }
        default {
            set flags $opts(type)
        }
    }

    # Create prototype. The 6th element - parameter description -
    # if missing which means we will just to default parameter
    # type handling. This is different from an empty element which
    # would mean no parameters
    set proto [list $dispid "" $opts(lcid) $flags $opts(rettype)]
    if {[info exists opts(params)]} {
        lappend proto $opts(params)
    }

    return $proto
}


#
# Return attributes of a ITypeInfo
proc twapi::itypeinfo_get_info {ifc args} {
    array set opts [parseargs args {
        all
        guid
        lcid
        constructorid
        destructorid
        schema
        instancesize
        typekind
        fncount
        varcount
        interfacecount
        vtblsize
        alignment
        majorversion
        minorversion
        aliasdesc
        flags
        idldesc
        memidmap
    } -maxleftover 0]

    array set data [ITypeInfo_GetTypeAttr $ifc]
    set result [list ]
    foreach {opt key} {
        guid guid
        lcid lcid
        constructorid memidConstructor
        destructorid  memidDestructor
        schema lpstrSchema
        instancesize cbSizeInstance
        fncount cFuncs
        varcount cVars
        interfacecount cImplTypes
        vtblsize cbSizeVft
        alignment cbAlignment
        majorversion wMajorVerNum
        minorversion wMinorVerNum
        aliasdesc tdescAlias
    } {
        if {$opts(all) || $opts($opt)} {
            lappend result -$opt $data($key)
        }
    }

    if {$opts(all) || $opts(typekind)} {
        set typekind $data(typekind)
        if {[info exists ::twapi::_typekind_map($typekind)]} {
            set typekind $::twapi::_typekind_map($typekind)
        }
        lappend result -typekind $typekind
    }

    if {$opts(all) || $opts(flags)} {
        lappend result -flags [_make_symbolic_bitmask $data(wTypeFlags) {
            appobject       1
            cancreate       2
            licensed        4
            predeclid       8
            hidden         16
            control        32
            dual           64
            nonextensible 128
            oleautomation 256
            restricted    512
            aggregatable 1024
            replaceable  2048
            dispatchable 4096
            reversebind  8192
            proxy       16384
        }]
    }

    if {$opts(all) || $opts(idldesc)} {
        lappend result -idldesc [_make_symbolic_bitmask $data(idldescType) {
            in 1
            out 2
            lcid 4
            retval 8
        }]
    }

    if {$opts(all) || $opts(memidmap)} {
        set memidmap [list ]
        for {set i 0} {$i < $data(cFuncs)} {incr i} {
            array set fninfo [itypeinfo_get_func_info $ifc $i -memid -name]
            lappend memidmap $fninfo(-memid) $fninfo(-name)
        }
        lappend result -memidmap $memidmap
    }

    return $result
}

#
# Get the referenced typeinfo of a typeinfo
proc twapi::itypeinfo_get_referenced_itypeinfo {ifc index} {
    set hreftype [ITypeInfo_GetRefTypeOfImplType $ifc $index]
    return [ITypeInfo_GetRefTypeInfo $ifc $hreftype]
}

#
# Get the containing typelib
proc twapi::itypeinfo_get_itypelib {ifc} {
    return [ITypeInfo_GetContainingTypeLib $ifc]
}

#
# Get the typecomp for a typeinfo
proc twapi::itypeinfo_get_itypecomp {ifc} {
    return [ITypeInfo_GetTypeComp $ifc]
}

#
# Get a function definition
proc twapi::itypeinfo_get_name {ifc} {
    return [lindex [itypeinfo_get_doc $ifc -1 -name] 1]
}


#
# Get a variable description associated with a type
proc twapi::itypeinfo_get_var_info {ifc index args} {
    # TBD - add support for retrieving elemdescVar.paramdesc fields

    array set opts [parseargs args {
        all
        name
        memid
        schema
        datatype
        value
        valuetype
        varkind
        flags
    } -maxleftover 0]

    array set data [ITypeInfo_GetVarDesc $ifc $index]
    
    set result [list ]
    foreach {opt key} {
        memid memid
        schema lpstrSchema
        datatype elemdescVar.tdesc
    } {
        if {$opts(all) || $opts($opt)} {
            lappend result -$opt $data($key)
        }
    }

    if {$opts(all) || $opts(value)} {
        if {[info exists data(lpvarValue)]} {
            # Const value
            lappend result -value [lindex $data(lpvarValue) 1]
        } else {
            lappend result -value $data(oInst)
        }
    }

    if {$opts(all) || $opts(valuetype)} {
        if {[info exists data(lpvarValue)]} {
            lappend result -valuetype [lindex $data(lpvarValue) 0]
        } else {
            lappend result -valuetype int
        }
    }

    if {$opts(all) || $opts(varkind)} {
        lappend result -varkind [string map {
            0 perinstance
            1 static
            2 const
            3 dispatch
        } $data(varkind)]
    }

    if {$opts(all) || $opts(flags)} {
        lappend result -flags [_make_symbolic_bitmask $data(wVarFlags) {
            readonly       1
            source       2
            bindable        4
            requestedit       8
            displaybind         16
            defaultbind        32
            hidden           64
            restricted 128
            defaultcollelem 256
            uidefault    512
            nonbrowsable 1024
            replaceable  2048
            immediatebind 4096
        }]
    }
    
    if {$opts(all) || $opts(name)} {
        set result [concat $result [itypeinfo_get_doc $ifc $data(memid) -name]]
    }    

    return $result
}

#
# Get a function definition
proc twapi::itypeinfo_get_func_info {ifc index args} {

    array set opts [parseargs args {
        all
        name
        memid
        funckind
        invkind
        callconv
        params
        paramnames
        flags
        datatype
        resultcodes
        vtbloffset
    } -maxleftover 0]

    array set data [ITypeInfo_GetFuncDesc $ifc $index]
    set result [list ]

    if {$opts(all) || $opts(paramnames)} {
        lappend result -paramnames [lrange [itypeinfo_get_names $ifc $data(memid)] 1 end]
    }
    foreach {opt key} {
        memid       memid
        vtbloffset  oVft
        datatype    elemdescFunc.tdesc
        resultcodes lprgscode
    } {
        if {$opts(all) || $opts($opt)} {
            lappend result -$opt $data($key)
        }
    }

    if {$opts(all) || $opts(funckind)} {
        lappend result -funckind [string map {
            0 virtual
            1 purevirtual
            2 nonvirtual
            3 static
            4 dispatch
        } $data(funckind)]
    }

    if {$opts(all) || $opts(invkind)} {
        lappend result -invkind [string map {
            0 func
            1 propget
            2 propput
            3 propputref
        } $data(invkind)]
    }

    if {$opts(all) || $opts(callconv)} {
        lappend result -callconv [string map {
            0 fastcall
            1 cdecl
            2 pascal
            3 macpascal
            4 stdcall
            5 fpfastcall
            6 syscall
            7 mpwcdecl
            8 mpwpascal
        } $data(callconv)]
    }

    if {$opts(all) || $opts(flags)} {
        lappend result -flags [_make_symbolic_bitmask $data(wFuncFlags) {
            restricted   1
            source       2
            bindable     4
            requestedit  8
            displaybind  16
            defaultbind  32
            hidden       64
            usesgetlasterror  128
            defaultcollelem 256
            uidefault    512
            nonbrowsable 1024
            replaceable  2048
            immediatebind 4096
        }]
    }

    if {$opts(all) || $opts(params)} {
        set params [list ]
        foreach param $data(lprgelemdescParam) {
            foreach {paramtype paramdesc} $param break
            set paramflags [_make_symbolic_bitmask [lindex $paramdesc 0] {
                in 1
                out 2
                lcid 4
                retval 8
                optional 16
                hasdefault 32
                hascustom  64
            }]
            if {[llength $paramdesc] > 1} {
                # There is a default value associated with the parameter
                lappend params [list $paramtype $paramflags [lindex $paramdesc 1]]
            } else {
                lappend params [list $paramtype $paramflags]
            }
        }
        lappend result -params $params
    }

    if {$opts(all) || $opts(name)} {
        set result [concat $result [itypeinfo_get_doc $ifc $data(memid) -name]]
    }    

    return $result
}

#
# Get documentation for a element of a type
proc twapi::itypeinfo_get_doc {ifc memid args} {
    array set opts [parseargs args {
        all
        name
        docstring
        helpctx
        helpfile
    } -maxleftover 0]

    foreach {name docstring helpctx helpfile} [ITypeInfo_GetDocumentation $ifc $memid] break

    set result [list ]
    foreach opt {name docstring helpctx helpfile} {
        if {$opts(all) || $opts($opt)} {
            lappend result -$opt [set $opt]
        }
    }
    return $result
}

#
# Get ids of names
proc twapi::itypeinfo_names_to_ids {ifc name args} {
    array set opts [parseargs args {
        paramnames.arg
    } -maxleftover 0 -nulldefault]

    return [ITypeInfo_GetIDsOfNames $ifc [concat [list $name] $opts(paramnames)]]
}


#
# Get type information
proc twapi::itypeinfo_get_impl_type_flags {ifc index} {
    return [_make_symbolic_bitmask \
                [ITypeInfo_GetImplTypeFlags $ifc $index] \
                {
                    default      1
                    source       2
                    restricted   4
                    defaultvtable 8
                }]    
}

#
# Get names in a typeinfo
proc twapi::itypeinfo_get_names {ifc memid} {
    return [ITypeInfo_GetNames $ifc $memid]
}


#
# ITypeLib commands
#

# Return an interface to a typelib
proc twapi::get_itypelib {path args} {
    array set opts [parseargs args {
        {registration.arg none {none register default}}
    } -maxleftover 0]

    return [LoadTypeLibEx $path [string map {default 0 register 1 none 2} $opts(registration)]]
}

#
# Return an interface to a typelib from the registry
proc twapi::get_registered_itypelib {uuid major minor args} {
    array set opts [parseargs args {
        lcid.int
    } -maxleftover 0 -nulldefault]
    
    return [LoadRegTypeLib $uuid $major $minor $opts(lcid)]
}

#
# Register a typelib
proc twapi::itypelib_register {ifc path helppath args} {
    RegisterTypeLib $ifc $path $helppath
}

#
# Unregister a typelib
proc twapi::itypelib_unregister {uuid major minor args} {
    array set opts [parseargs args {
        lcid.int
    } -maxleftover 0 -nulldefault]

    UnRegisterTypeLib $uuid $major $minor $opts(lcid) 1
}



#
# Return count of entries in a typelib
proc twapi::itypelib_count {ifc} {
    return [ITypeLib_GetTypeInfoCount $ifc]
}

#
# Returns the type of a type description
proc twapi::itypelib_get_entry_typekind {ifc id} {
    set typekind [ITypeLib_GetTypeInfoType $ifc $id]
    if {[info exists ::twapi::_typekind_map($typekind)]} {
        set typekind $::twapi::_typekind_map($typekind)
    }
}

#
# Get documentation for a element of a typelib
proc twapi::itypelib_get_entry_doc {ifc id args} {
    array set opts [parseargs args {
        all
        name
        docstring
        helpctx
        helpfile
    } -maxleftover 0]

    foreach {name docstring helpctx helpfile} [ITypeLib_GetDocumentation $ifc $id] break

    set result [list ]
    foreach opt {name docstring helpctx helpfile} {
        if {$opts(all) || $opts($opt)} {
            lappend result -$opt [set $opt]
        }
    }
    return $result
}

#
# Returns the ITypeInfo interface for a typelib entry
interp alias {} twapi::itypelib_get_entry_itypeinfo {} twapi::ITypeLib_GetTypeInfo


#
# Returns the ITypeInfo interface for a typelib guid
interp alias {} twapi::itypelib_get_registered_itypeinfo {} ITypeLib_GetTypeInfoOfGuid


#
# Returns the path to the typelib based on a guid
proc twapi::itypelib_get_registered_path {guid major minor args} {
    array set opts [parseargs args {
        lcid.int
    } -maxleftover 0 -nulldefault]


    set path [QueryPathOfRegTypeLib $guid $major $minor $opts(lcid)]
    # At least some versions have a bug in that there is an extra \0
    # at the end.
    if {[string equal [string index $path end] \0]} {
        set path [string range $path 0 end-1]
    }
    return $path
}

#
# Get attributes of a library
proc twapi::itypelib_get_info {ifc args} {
    array set opts [parseargs args {
        all
        guid
        lcid
        syskind
        majorversion
        minorversion
        flags
    } -maxleftover 0]

    array set data [ITypeLib_GetLibAttr $ifc]
    set result [list ]
    foreach {opt key} {
        guid guid
        lcid lcid
        majorversion wMajorVerNum
        minorversion wMinorVerNum
    } {
        if {$opts(all) || $opts($opt)} {
            lappend result -$opt $data($key)
        }
    }

    if {$opts(all) || $opts(flags)} {
        lappend result -flags [_make_symbolic_bitmask $data(wLibFlags) {
            restricted      1
            control         2
            hidden          4
            hasdiskimage    8
        }]
    }

    if {$opts(all) || $opts(syskind)} {
        lappend result -syskind [string map {
            0 win16
            1 win32
            2 mac
        } $data(syskind)]
    }

    return $result
}

#
# Iterate through a typelib. Caller is responsible for each itypeinfo
# passed
proc twapi::itypelib_foreach {args} {

    array set opts [parseargs args {
        type.arg
        name.arg
        guid.arg
    } -maxleftover 3]

    if {[llength $args] != 3} {
        error "Syntax error: Should be 'itypelib_foreach ?options? VARNAME ITYPELIB SCRIPT'"
    }

    foreach {varname tl script} $args break

    set count [itypelib_count $tl]

    for {set i 0} {$i < $count} {incr i} {
        if {[info exists opts(type)] &&
            $opts(type) ne [itypelib_get_entry_typekind $tl $i]} {
            continue;                   # Type does not match
        }
        if {[info exists opts(name)] &&
            [string compare -nocase $opts(name) [lindex [itypelib_get_entry_doc $tl $i -name] 1]]} {
            continue;                   # Name does not match
        }
        upvar $varname ti
        set ti [itypelib_get_entry_itypeinfo $tl $i]
        if {[info exists opts(guid)]} {
            if {[string compare -nocase [lindex [itypeinfo_get_info $ti -guid] 1] $opts(guid)]} {
                continue
            }
        }
        set ret [catch {uplevel $script} msg]
        switch -exact -- $ret {
            1 {
                error $msg $::errorInfo $::errorCode
            }
            2 {
                return; # TCL_RETURN
            }
            3 {
                set i $count; # TCL_BREAK
            }
        }
    }
    return
}





#
# Map interface name to IID
proc twapi::name_to_iid {iname} {

    set iname [string tolower $iname]

    if {[info exists ::twapi::_name_to_iid_cache($iname)]} {
        return $::twapi::_name_to_iid_cache($iname)
    }

    # Look up the registry
    foreach iid [registry keys HKEY_CLASSES_ROOT\\Interface] {
        if {![catch {
            set val [registry get HKEY_CLASSES_ROOT\\Interface\\$iid ""]
        }]} {
            if {[string equal -nocase $iname $val]} {
                return [set ::twapi::_name_to_iid_cache($iname) $iid]
            }
        }
    }
    return [set ::twapi::_name_to_iid_cache($iname) ""]
}


#
# Map interface IID to name
proc twapi::iid_to_name {iid} {
    set iname ""
    catch {set iname [registry get HKEY_CLASSES_ROOT\\Interface\\$iid ""]}
    return $iname
}

#
# Given a collection, iterates over it and returns a list consisting of
# {name value} pairs. Assumes the collection is a list of COM objects
# which have properties Name and value
# obj is a COM object as returned by comobj command
proc twapi::com_named_property_list {obj} {
    set result [list ]
    $obj -iterate itemobj {
        lappend result [$itemobj Name] [$itemobj]
        $itemobj -destroy
    }
    return $result
}


#
# Get the typeinfo for the default source interface of a coclass
# $coti is the typeinfo of the coclass
proc twapi::get_coclass_default_source_itypeinfo {coti} {
    set count [lindex [itypeinfo_get_info $coti -interfacecount] 1]
    for {set i 0} {$i < $count} {incr i} {
        set flags [ITypeInfo_GetImplTypeFlags $coti $i]
        # default 0x1, source 0x2
        if {($flags & 3) == 3} {
            return [itypeinfo_get_referenced_itypeinfo $coti $i]
        }
    }
    return ""
}

#
# Convert a variant time to a time list
proc twapi::variant_time_to_timelist {double} {
    return [VariantTimeToSystemTime $double]
}

#
# Convert a time list time to a variant time
proc twapi::timelist_to_variant_time {timelist} {
    return [SystemTimeToVariantTime $timelist]
}


################################################################

#
# Test code
proc twapi::_print_typelib {path args} {
    array set opts [parseargs args {
        type.arg
        name.arg
    } -maxleftover 0]

    
    set ifc [get_itypelib $path -registration none]
    set count [itypelib_count $ifc]

    for {set i 0} {$i < $count} {incr i} {
        set type [itypelib_get_entry_typekind $ifc $i]
        if {[info exists opts(type)] && $opts(type) ne $type} continue
        array set tlinfo [itypelib_get_entry_doc $ifc $i -all]
        if {[info exists opts(name)] && [string compare -nocase $opts(name) $tlinfo(-name)]} continue
        set desc [list "$i:\t$type\t$tlinfo(-name) - $tlinfo(-docstring)"]
        set ti [twapi::itypelib_get_entry_itypeinfo $ifc $i]
        array set attrs [itypeinfo_get_info $ti -all]
        switch -exact -- $type {
            record -
            union  -
            enum {
                for {set j 0} {$j < $attrs(-varcount)} {incr j} {
                    array set vardata [itypeinfo_get_var_info $ti $j -all]
                    set vardesc "\t\t$vardata(-varkind) $vardata(-datatype) $vardata(-name)"
                    if {$type eq "enum"} {
                        append vardesc " = $vardata(-value)"
                    } else {
                        append vardesc " (offset $vardata(-value))"
                    }
                    lappend desc $vardesc
                }
            }
            alias {
                lappend desc "\t\ttypedef $attrs(-aliasdesc)"
            }
            dispatch -
            interface {
                for {set j 0} {$j < $attrs(-fncount)} {incr j} {
                    array set funcdata [itypeinfo_get_func_info $ti $j -all] 
                    if {$funcdata(-funckind) eq "dispatch"} {
                        set funckind "(dispid $funcdata(-memid))"
                    } else {
                        set funckind "(vtable $funcdata(-vtbloffset))"
                    }
                    lappend desc "\t\t$funckind [_resolve_com_type $ti $funcdata(-datatype)] $funcdata(-name) [_resolve_com_params $ti $funcdata(-params) $funcdata(-paramnames)]"
                }
            }
            coclass {
                for {set j 0} {$j < $attrs(-interfacecount)} {incr j} {
                    set ti2 [itypeinfo_get_referenced_itypeinfo $ti $j]
                    set idesc "\t\t[itypeinfo_get_name $ti2]"
                    set iflags [itypeinfo_get_impl_type_flags $ti $j]
                    if {[llength $iflags]} {
                        append idesc " ([join $iflags ,])"
                    }
                    lappend desc $idesc
                    iunknown_release $ti2
                }
            }
        }
        puts [join $desc \n]
        iunknown_release $ti
    }

    iunknown_release $ifc
    return
}


#
# Print methods in an interface
proc twapi::_print_interface {ifc} {
    set ti [idispatch_get_itypeinfo $ifc]
    twapi::_print_interface_helper $ti
    iunknown_release $ti
}

proc twapi::_print_interface_helper {ti {names_already_done ""}} {
    set name [itypeinfo_get_name $ti]
    if {[lsearch -exact $names_already_done $name] >= 0} {
        # Already printed this
        return $names_already_done
    }
    lappend names_already_done $name
    array set attrs [itypeinfo_get_info $ti -all]
    for {set j 0} {$j < $attrs(-fncount)} {incr j} {
        array set funcdata [itypeinfo_get_func_info $ti $j -all] 
        if {$funcdata(-funckind) eq "dispatch"} {
            set funckind "(dispid $funcdata(-memid))"
        } else {
            set funckind "(vtable $funcdata(-vtbloffset))"
        }
        lappend desc "\t$funckind [_resolve_com_type $ti $funcdata(-datatype)] $funcdata(-name) [_resolve_com_params $ti $funcdata(-params) $funcdata(-paramnames)]"
    }

    puts $name
    puts [join $desc \n]

    # Now get any referenced typeinfos and print them
    for {set j 0} {$j < $attrs(-interfacecount)} {incr j} {
        set ti2 [itypeinfo_get_referenced_itypeinfo $ti $j]
        set names_already_done [_print_interface_helper $ti2 $names_already_done]
        iunknown_release $ti2
    }
    return $names_already_done
}



#
# Resolves references to parameter definition
proc twapi::_resolve_com_params {ti params paramnames} {
    set result [list ]
    foreach param $params paramname $paramnames {
        set paramdesc [lreplace $param 0 0 [_resolve_com_type $ti [lindex $param 0]]]
        lappend paramdesc $paramname
        lappend result $paramdesc
    }
    return $result
}

#
# Resolves typedefs
proc twapi::_resolve_com_type {ti typedesc} {
    switch -exact -- [lindex $typedesc 0] {
        ptr {
            # Recurse to resolve any inner types
            set typedesc [list ptr [_resolve_com_type $ti [lindex $typedesc 1]]]
        }
        userdefined {
            set hreftype [lindex $typedesc 1]
            set ti2 [ITypeInfo_GetRefTypeInfo $ti $hreftype]
            set typedesc [list userdefined [itypeinfo_get_name $ti2]]
            iunknown_release $ti2
        }
        default {
        }
    }

    return $typedesc
}

#
# Returns a string value from a formatted variant value pair {VT_xxx value}
# $addref controls whether we do an AddRef when the value is a pointer to
# an interface. $raw controls whether interfaces are converted to comobjs
# or not.
proc twapi::_convert_from_variant {variant addref {raw false}} {
    # TBD - format appropriately depending on variant type for dates and
    # currency
    if {[llength $variant] == 0} {
        return ""
    }
    set vt [lindex $variant 0]

    if {$vt & 0x2000} {
        # VT_ARRAY
        if {[llength $variant] < 3} {
            return [list ]
        }
        set vt [expr {$vt & ~ 0x2000}]
        if {$vt == 12} {
            # Array of variants. Recursively convert values
            set result [list ]
            foreach elem [lindex $variant 2] {
                lappend result [_convert_from_variant $elem $addref $raw]
            }
            return $result
        } else {
            return [lindex $variant 2]
        }
    } else {
        if {$vt == 9} {
            # IDispatch - return as comobj
            set idisp [lindex $variant 1]; # May be NULL!
            if {$raw} {
                if {$addref && $idisp ne "NULL"} {
                    iunknown_addref $idisp
                }
                return $idisp
            } else {
                # Note comobj_idispatch takes care of NULL
                return [comobj_idispatch $idisp $addref]
            }
        } elseif {$vt == 13} {
            # IUnknown - try converting to IDispatch
            set iunk [lindex $variant 1]; # May be NULL!
            if {$raw} {
                if {$addref && $iunk ne "NULL"} {
                    iunknown_addref $iunk
                }
                return $iunk
            } else {
                # Try to return as idispatch
                if {$iunk eq "NULL"} {
                    return ::twapi::comobj_null
                }
                set idisp [iunknown_query_interface $iunk IDispatch]
                if {$idisp eq ""} {
                    # No IDispatch - return as is
                    if {$addref} {
                        iunknown_addref $iunk
                    }
                    return $iunk
                } else {
                    # If caller did not need us to do an addref, implies
                    # it would already have done it, and app code is expected
                    # to release. However, what we pass the appcode is
                    # a converted idispatch. So we have to do the release
                    # on the original interface ourselves.
                    if {! $addref} {
                        iunknown_release $iunk
                    }
                    return [comobj_idispatch $idisp false]
                }
            }
        }
    }
    return [lindex $variant 1]
}


#
# Wrapper used by comobj
proc twapi::_comobj_wrapper {comobj clsid args} {

    # Look for the interface object corresponding to this object
    if {![info exists ::twapi::com_instance_data($comobj,ifc)]} {
        error "Missing COM interface"
    }
    set ifc $::twapi::com_instance_data($comobj,ifc)

    set nargs [llength $args]
    # parse to figure out what the command is
    switch -exact -- [lindex $args 0] {
        -get {
            if {$nargs < 2} {
                error "Insufficient number of arguments supplied for method call"
            }
            set name [lindex $args 1]
            set params [lrange $args 2 end]
            set flags  2;           # Property get
        }
        -set {
            if {$nargs < 3} {
                error "Insufficient number of arguments supplied for method call"
            }
            set name [lindex $args 1]
            set params [lrange $args 2 end]
            set flags  4;           # Property set
        }
        -call {
            if {$nargs < 2} {
                error "Insufficient number of arguments supplied for method call"
            }
            set name [lindex $args 1]
            set params [lrange $args 2 end]
            set flags  1;           # Method call
        }
        -destroy {
            # Release sinks, connection points
            foreach sink_item [array names ::twapi::com_instance_data "$comobj,sink,*"] {
                set sinkid [lindex [split $sink_item ,] 2]
                $comobj -unbind $sinkid
            }

            # Get rid of dispatch prototypes. This is important for two
            # reasons. A different idispatch allocated later for a different
            # interface may have the same pointer value but will have
            # totally different prototypes. Even if it is the same
            # object and interface type, in the case of dynamic properties
            # we need to call IDispatchEx::GetDispIDs for the dynamic
            # properties to show up for the new object.
            array unset twapi::idispatch_prototypes ${ifc}*
            twapi::iunknown_release $ifc
            rename $comobj ""
            return
        }
        -isnull {
            return false
        }
        -precache {
            # TBD - document
            foreach {name proto} [lindex $args 1] {
                set flags [lindex $proto 3]
                set ::twapi::idispatch_prototypes($ifc,$name,0,$flags) $proto
            }
            return
        }
        "" {
            return [_convert_from_variant [twapi::idispatch_invoke $ifc ""] false]
        }
        -print {
            _print_interface $ifc
            return
        }
        -interface {
            return $ifc
        }
        -queryinterface {
            return [iunknown_query_interface $ifc [lindex $args 1]]
        }
        -with {
            # $obj -with SUBOBJECTPATHLIST arguments
            # where SUBOBJECTPATHLIST is list each element of which is
            # either a property or a method of the previous element in
            # the list. The element may itself be a list in which case
            # the first element is the property/method and remaining
            # are passed to it
            #
            # Note that 'arguments' may themselves be comobj subcommands!
            set subobjlist [lindex $args 1]
            set next $comobj
            set releaselist [list ]

            try {
                while {[llength $subobjlist]} {
                    set nextargs [lindex $subobjlist 0]
                    set subobjlist [lrange $subobjlist 1 end]
                    set next [uplevel [list $next] $nextargs]
                    lappend releaselist $next
                }
                # We use uplevel here because again we want to run in caller
                # context 
                return [uplevel [list $next] [lrange $args 2 end]]
            } finally {
                foreach next $releaselist {
                    $next -destroy
                }
            }
        }
        -iterate {
            # $obj -iterate VARNAME arguments
            if {[llength $args] < 3} {
                error "Insufficient arguments. Syntax '$comobj -iterate VARNAME CODEBLOCK'"
            }
            upvar [lindex $args 1] var
            # First get IEnumVariant iterator using the _NewEnum method
            set enum_disp [$comobj -get _NewEnum]
            # This gives us an IUnknown.
            try {
                # Convert the IUnknown to IEnumVARIANT
                set iter [iunknown_query_interface $enum_disp IEnumVARIANT]
                if {$iter ne ""} {
                    while {1} {
                        # Get the next item from iterator
                        set next [IEnumVARIANT_Next $iter 1]
                        foreach {more values} $next break
                        if {[llength $values]} {
                            # TBD - does var have to be released?
                            set var [_convert_from_variant [lindex $values 0] false]
                            set ret [catch {uplevel [lindex $args 2]} msg]
                            switch -exact -- $ret {
                                1 {
                                    error $msg $::errorInfo $::errorCode
                                }
                                2 {
                                    return; # TCL_RETURN
                                }
                                3 {
                                    set more 0; # TCL_BREAK
                                }
                            }
                        }
                        if {! $more} break
                    }
                }
            } finally {
                iunknown_release $enum_disp
                if {[info exists iter] && $iter ne ""} {
                    iunknown_release $iter
                }
            }
            return
        }
        -bind {
            # $comobj -bind callbackscript
            if {[llength $args] != 2} {
                error "Syntax error: should be '$comobj -bind SCRIPT"
            }

            # We need the typeinfo for the coclass. We get this one of
            # two ways - 
            # If the object supports IProvideClassInfo, we use it. Else
            # we try the following:
            #   - from the idispatch, we get its typeinfo
            #   - from the typeinfo, we get the containing typelib
            #   - then we search the typelib for the coclass clsid
            # Once we have the coclass we can locate the source interface
            # within it and retrieve disp id mappings
            try {
                set pci [iunknown_query_interface $ifc IProvideClassInfo]
                if {$pci ne ""} {
                    # Great, get the coclass from it
                    catch {set coti [IProvideClassInfo_GetClassInfo $pci]}
                }

                if {![info exists coti]} {
                    # Couldn't get the coclass above, try the second way
                    if {$clsid eq ""} {
                        error "Do not have class information for binding"
                    }

                    set ti [idispatch_get_itypeinfo $ifc]
                    set tl [lindex [itypeinfo_get_itypelib $ti] 0]
                    itypelib_foreach -guid $clsid -type coclass coti $tl {
                        break
                    }
                }
                if {![info exists coti]} {
                    error "Could not find coclass for binding"
                }
                # $coti is the coclass information. Get dispids for the default
                # source interface for events and its guid
                set srcti [get_coclass_default_source_itypeinfo $coti]
                array set srcinfo [itypeinfo_get_info $srcti -memidmap -guid]

                # Now we need to get the actual connection point itself
                set container [iunknown_query_interface $ifc IConnectionPointContainer]
                if {$container eq ""} {
                    error "Object does not have any event source interfaces"
                }
                set connpt [IConnectionPointContainer_FindConnectionPoint $container $srcinfo(-guid)]
                if {$connpt eq ""} {
                    error "Object has no matching event source"
                }

                # Finally, create our sink object
                # TBD - need to make sure comobj is not deleted or
                # should the callback itself check?
                set sink [ComEventSink $srcinfo(-guid) [list ::twapi::_eventsink_callback $comobj $srcinfo(-memidmap) [lindex $args 1]]]

                # OK, we finally have everything we need. Tell the event source
                set sinkid [IConnectionPoint_Advise $connpt $sink]
                set ::twapi::com_instance_data($comobj,sink,$sinkid) $sink
                set ::twapi::com_instance_data($comobj,connpt,$sinkid) $connpt
                return $sinkid
            } onerror {} {
                # These are released only on error as otherwise they have
                # to be kept until unbind time
                foreach x {connpt sink} {
                    if {[info exists $x] && [set $x] ne ""} {
                        iunknown_release [set $x]
                    }
                }
                error $errorResult $errorInfo $errorCode
            } finally {
                # In all cases, release any interfaces we created
                # Note connpt and sink are released at unbind time except
                # on error
                foreach x {ti tl coti srcti container pci} {
                    if {[info exists $x] && [set $x] ne ""} {
                        iunknown_release [set $x]
                    }
                }
            }
        }
        -unbind {
            # $obj -unbind SINKID
            if {[llength $args] != 2} {
                error "Syntax error: Should be '$comobj -unbind BINDID'"
            }
            set sinkid [lindex $args 1]
            if {[info exists ::twapi::com_instance_data($comobj,connpt,$sinkid)]} {
                IConnectionPoint_Unadvise $::twapi::com_instance_data($comobj,connpt,$sinkid) $sinkid
                unset ::twapi::com_instance_data($comobj,connpt,$sinkid)
            }
            if {[info exists ::twapi::com_instance_data($comobj,sink,$sinkid)]} {
                iunknown_release $::twapi::com_instance_data($comobj,sink,$sinkid)
                unset ::twapi::com_instance_data($comobj,sink,$sinkid)
            }
            return
        }
        default {
            # Try to figure out whether it is a name or method
            set name [lindex $args 0]
            set params [lrange $args 1 end]
            twapi::idispatch_fill_prototypes $ifc ::twapi::idispatch_prototypes 0 $name
            # We have to figure out if it is a property get, property put
            # or a method. We will check in that order. If multiple matches
            # we check if number of parameters matches that in prototype

            set flags 0
            if {[info exists ::twapi::idispatch_prototypes($ifc,$name,0,2)]} {
                # Property get
                set flags [expr {$flags | 2}]
            }
            if {[info exists ::twapi::idispatch_prototypes($ifc,$name,0,4)]} {
                # Property set
                set flags [expr {$flags | 4}]
            }
            if {[info exists ::twapi::idispatch_prototypes($ifc,$name,0,1)]} {
                # Property set
                set flags [expr {$flags | 1}]
            }
            # If only one of the bits is set, then that's what we go with
            # else we have to use a priority scheme
            if {$flags != 0 && $flags != 1 && $flags != 2 && $flags != 4} {
                # More than 1 possibility
                # Check for match on exact number of parameters
                set nparams [llength $params]
                foreach flag {1 2 4} {
                    if {$flags & $flag} {
                        set proto $::twapi::idispatch_prototypes($ifc,$name,0,$flag)
                        # See if we do have parameter info in prototype
                        if {[llength $proto] > 5} {
                            # Yes we do. see if number matches supplied args
                            if {$nparams == [llength [lindex $proto 5]]} {
                                # Matched
                                set matched_flags $flag
                                break
                            }
                        }
                    }
                }
                if {![info exists matched_flags]} {
                    # Still ambiguity. Use following heruristic. 
                    # If possible get, and no args supplied, assume get
                    # If possible put, and one arg supplied, assume put
                    # Else method if set
                    # TBD - maybe we should use return type to distinguish
                    # between put and get? But not sure that's always
                    # supplied in prototype
                    if {($flags & 2) && $nparams == 0} {
                        set matched_flags 2
                    } elseif {($flags & 4) && $nparams == 1} {
                        set matched_flags 4
                    } elseif {$flags & 1} {
                        set matched_flags 1
                    }
                }
                if {[info exists matched_flags]} {
                    set flags $matched_flags
                } else {
                    set flags 0
                }
            }
            if {$flags == 0} {
                # Could not figure out, assume method
                set flags 1
            }
        }
    }

    # Check if a prototype exists
    if {![info exists ::twapi::idispatch_prototypes($ifc,$name,0,$flags)]} {
        twapi::idispatch_fill_prototypes $ifc ::twapi::idispatch_prototypes 0 $name
        if {![info exists ::twapi::idispatch_prototypes($ifc,$name,0,$flags)]} {
            # Don't have prototype. Try to get the dispatch id of the
            # function, use a default prototype and hope for the best
            set dispid [lindex [idispatch_names_to_ids $ifc $name] 1]
            if {$dispid eq ""} {
                win32_error 0x80020003 "No property or method found with name '$name'."
            }
            # Create prototype. Return type is assumed 8 (BSTR) but does
            # not matter as automatic type conversion will be done on
            # the return value. The 6th element - parameter description -
            # is missing which means we will just to default parameter
            # type handling
            set ::twapi::idispatch_prototypes($ifc,$name,0,$flags) [list $dispid "" 0 $flags 8]
        }
    }
    
    # Invoke the function. TBD - should we do a uplevel instead of eval
    # here so variables if any are in caller's context ?
    return [_convert_from_variant [eval [list twapi::idispatch_invoke $ifc $::twapi::idispatch_prototypes($ifc,$name,0,$flags)] $params] false]
}


proc twapi::_comobj_active {comobj} {
    # TBD - temporary
    if {[info exists ::twapi::com_instance_data($comobj,ifc)]} {
        return 1
    } else {
        return 0
    }
}

#
# General dispatcher for callbacks from event sinks. Invokes the actual
# registered script after mapping dispid's
proc twapi::_eventsink_callback {comobj dispidmap script dispid lcid flags params} {
    # Check if the comobj is still active
    if {![_comobj_active $comobj]} {
        if {$::twapi::com_debug} {
            debug_puts "COM event received for inactive object"
        }
        return;                         # Object has gone away, ignore
    }

    set result ""
    set retcode [catch {
        # Map dispid to event if possible
        set dispid [twapi::kl_get_default $dispidmap $dispid $dispid]
        set converted_params [list ]
        foreach param $params {
            lappend converted_params [_convert_from_variant $param false true]
        }
        set result [uplevel \#0 $script [list $dispid] $converted_params]
    } msg]

    if {$::twapi::com_debug && $retcode} {
        debug_puts "Event sink callback error ($retcode): $msg\n$::errorInfo"
    }

    # $retcode is returned as HRESULT by the Invoke
    return -code $retcode $result
}

#
# Return clsid from a string. If $clsid is a valid CLSID - returns as is
# else tries to convert it from progid. An error is generated if neither
# works
proc twapi::_convert_to_clsid {comid} {
    if {[catch {IIDFromString $comid}]} {
        return [progid_to_clsid $comid]
    }
    return $comid
}


#
# Get WMI service
proc twapi::_wmi {} {
    return [comobj_idispatch [::twapi::Twapi_GetObjectIDispatch "winmgmts:{impersonationLevel=impersonate}!//./root/cimv2"] false]
}


# Get cached IDispatch and IUNknown IID's
proc twapi::_iid_iunknown {} {
    return $::twapi::_name_to_iid_cache(iunknown)
}
proc twapi::_iid_idispatch {} {
    return $::twapi::_name_to_iid_cache(idispatch)
}

#
# Return IID and name given a IID or name
proc twapi::_resolve_iid {name_or_iid} {

    # IID -> name mapping is more efficient so first assume it is
    # an IID else we will unnecessarily trundle through the whole
    # registry area looking for an IID when we already have it
    # Assume it is a name
    set other [iid_to_name $name_or_iid]
    if {$other ne ""} {
        # It was indeed the IID. Return the pair
        return [list $name_or_iid $other]
    }

    # Else resolve as a name
    set other [name_to_iid $name_or_iid]
    if {$other ne ""} {
        # Yep
        return [list $other $name_or_iid]
    }

    win32_error 0x80004002 "Could not find IID $name_or_iid"
}


#
# Some simple tests

proc twapi::_com_tests {} {
    puts "Invoking Internet Explorer"
    set ie [comobj InternetExplorer.Application -enableaaa true]
    $ie Visible 1
    $ie Navigate http://www.google.com
    after 2000
    puts "Exiting Internet Explorer"
    $ie Quit
    $ie -destroy
    puts "Internet Explorer done."

    puts "------------------------------------------"

    puts "Invoking Word"
    set word [comobj Word.Application]
    set doc [$word -with Documents Add]
    $word Visible 1
    puts "Inserting text"
    $word -with {selection font} name "Courier New"
    $word -with {selection font} size 10.0
    $doc -with content text "Text in Courier 10 point"
    after 2000
    puts "Exiting Word"
    $word Quit 0
    puts "Word done."
    
    puts "------------------------------------------"

    puts "WMI BIOS test"
    puts [get_bios_info]
    puts "WMI BIOS done."

    puts "------------------------------------------"
    
    puts "WMI direct property access test (get bios version)"
    set wmi [twapi::_wmi]
    $wmi -with {{ExecQuery "select * from Win32_BIOS"}} -iterate biosobj {
        puts "BIOS version: [$biosobj BiosVersion]"
        $biosobj -destroy
    }
    $wmi -destroy

    puts "------------------------------------------"

    puts " Starting process tracker. Type 'twapi::_stop_process_tracker' to stop it."
    twapi::_start_process_tracker
    vwait ::twapi::_stop_tracker
}

#
proc twapi::_wmi_read_popups {} {
    set res {}
    set wmi [twapi::_wmi]
    set wql {select * from Win32_NTLogEvent where LogFile='System' and \
                 EventType='3'    and \
                 SourceName='Application Popup'}
    set svcs [$wmi ExecQuery $wql]

    # Iterate all records
    $svcs -iterate instance {
        set propSet [$instance Properties_]
        # only the property (object) 'Message' is of interest here
        set msgVal [[$propSet Item Message] Value]
        lappend res $msgVal
    }
    return $res
}

#
proc twapi::_wmi_read_popups_succint {} {
    set res [list ]
    set wmi [twapi::_wmi]
    $wmi -with {
        {ExecQuery "select * from Win32_NTLogEvent where LogFile='System' and EventType='3' and SourceName='Application Popup'"}
    } -iterate event {
        lappend res [$event Message]
    }
    return $res
}

#
proc twapi::_wmi_get_autostart_services {} {
    set res [list ]
    set wmi [twapi::_wmi]
    $wmi -with {
        {ExecQuery "select * from Win32_Service where StartMode='Auto'"}
    } -iterate svc {
        lappend res [$svc DisplayName]
    }
    return $res
}

proc twapi::get_bios_info {} {
    set wmi [twapi::_wmi]
    array set entries [list ]
    $wmi -with {{ExecQuery "select * from Win32_BIOS"}} -iterate elem {
        set propset [$elem Properties_]
        array set entries [com_named_property_list $propset]
        $elem -destroy
        $propset -destroy
    }
    $wmi -destroy
    return [array get entries]
}

# Handler invoked when a process is started.  Will print exe name of process.
proc twapi::_process_start_handler {wmi_event args} {
    if {$wmi_event eq "OnObjectReady"} {
        # First arg is a IDispatch interface of the event object
        # Create a TWAPI COM object out of it
        set event_obj [comobj_idispatch [lindex $args 0] true]

        # Get and print the Name property
        puts "Process [$event_obj ProcessID] [$event_obj ProcessName] started at [clock format [large_system_time_to_secs [$event_obj TIME_CREATED]] -format {%x %X}]"

        # Get rid of the event object
        $event_obj -destroy
    }
}

# Call to begin tracking of processes.
proc twapi::_start_process_tracker {} {
    # Get local WMI root provider
    set ::twapi::_process_wmi [twapi::_wmi]

    # Create an WMI event sink
    set ::twapi::_process_event_sink [comobj wbemscripting.swbemsink]

    # Attach our handler to it
    set ::twapi::_process_event_sink_id [$::twapi::_process_event_sink -bind twapi::_process_start_handler]

    # Associate the sink with a query that polls every 1 sec for process
    # starts.
    $::twapi::_process_wmi ExecNotificationQueryAsync [$::twapi::_process_event_sink -interface] "select * from Win32_ProcessStartTrace"
}

# Stop tracking of process starts
proc twapi::_stop_process_tracker {} {
    # Cancel event notifications
    $::twapi::_process_event_sink Cancel

    # Unbind our callback
    $::twapi::_process_event_sink -unbind $::twapi::_process_event_sink_id

    # Get rid of all objects
    $::twapi::_process_event_sink -destroy
    $::twapi::_process_wmi -destroy

    set ::twapi::_stop_tracker 1
    return
}


# Handler invoked when a service status changes.
proc twapi::_service_change_handler {wmi_event args} {
    if {$wmi_event eq "OnObjectReady"} {
        # First arg is a IDispatch interface of the event object
        # Create a TWAPI COM object out of it
        set event_obj [twapi::comobj_idispatch [lindex $args 0] true]

        puts "Previous: [$event_obj PreviousInstance]"
        #puts "Target: [$event_obj -with TargetInstance State]"

        # Get rid of the event object
        $event_obj -destroy
    }
}

# Call to begin tracking of service state
proc twapi::_start_service_tracker {} {
    # Get local WMI root provider
    set ::twapi::_service_wmi [twapi::_wmi]

    # Create an WMI event sink
    set ::twapi::_service_event_sink [twapi::comobj wbemscripting.swbemsink]

    # Attach our handler to it
    set ::twapi::_service_event_sink_id [$::twapi::_service_event_sink -bind twapi::_service_change_handler]

    # Associate the sink with a query that polls every 1 sec for service
    # starts.
    $::twapi::_service_wmi ExecNotificationQueryAsync [$::twapi::_service_event_sink -interface] "select * from __InstanceModificationEvent within 1 where TargetInstance ISA 'Win32_Service'"
}

# Stop tracking of services
proc twapi::_stop_service_tracker {} {
    # Cancel event notifications
    $::twapi::_service_event_sink Cancel

    # Unbind our callback
    $::twapi::_service_event_sink -unbind $::twapi::_service_event_sink_id

    # Get rid of all objects
    $::twapi::_service_event_sink -destroy
    $::twapi::_service_wmi -destroy
}
