#!/usr/bin/env tclsh
# -*- mode: tcl; coding: utf-8-unix; -*-
# Time-stamp: <2016-05-14 22:32:53 alain>%
# $Id: netinfo_win_nt.tcl,v 1.15 2009-01-25 18:26:08 cvsalain Exp $
#-------------------------------------------------------------------------------
# 2007/03/19
# Bug:
# - By reading card data in
#   HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\Tcpip\Parameters\Interfaces
#   it was possible to read data from a removed interface. Correct it by reading
#   HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Services\_service_name_\Parameters\Tcpip\
	#   and checking that HKEY_LOCAL_MACHINE\SYSTEM\CurrentControlSet\Control\
	#   DeviceClasses\{ad498944-762f-11d0-8dcb-00c04fc3358c}\_some_identifier_\
	#   _service_name_\Control exist
#
# Addition
# - MAC address detection
# - use of common part for all OS through netinfo_common
#
# 20090122 version 1.1
#   Addition of differenciation between Microsoft workgroup and domain
#   and use of TWAPI for Windows 2000, XP, server 2003, vista
# 20090124 version 1.2
#   Addition of operating system support level
# 20090125 version 1.3
#   Collect all informations in only one call
# 20160515 version 1.4
#   Correct data collection on Windows seven by checking 
#    HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Parameters\\Interfaces\\
#    if not all data have been found when in DHCP
#-------------------------------------------------------------------------------

package provide netinfo_win_nt 1.4

if {![info exists ::netinfo]} {
	# Adjust auto_path to contain script home and current directory
	set script_home [string trimright [file dirname [info script]] ./]
	set script_home [file join [pwd] $script_home]
	lappend auto_path $script_home
}

# Adjust auto_path to contain twapi path
set lib_path [file join $script_home "twapi"]
if {[lsearch $auto_path $lib_path] == -1} {
	lappend auto_path $lib_path
}

package require netinfo_common
package require registry
package require twapi

# Network card
set ::net_info::networkCardsKey \
	"HKEY_LOCAL_MACHINE\\SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\NetworkCards\\"

# Microsoft Workgroup
set ::net_info::explorerKey \
	"HKEY_CURRENT_USER\\SOFTWARE\\Microsoft\\Windows\\CurrentVersion\\Explorer\\"

# Microsoft Domain
set ::net_info::winlogonKey \
	"HKEY_LOCAL_MACHINE\\SOFTWARE\\Microsoft\\Windows NT\\CurrentVersion\\Winlogon\\"

# Classes
set ::net_info::classesKey \
	"HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Control\\DeviceClasses\\{ad498944-762f-11d0-8dcb-00c04fc3358c}\\"

# Services
set ::net_info::servicesKey \
	"HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\"

# Network cards configuration
set ::net_info::computerTcpIpParametersSubKey \
	"Tcpip\\Parameters\\"

# Specific network card configuration
set ::net_info::cardTcpIpParametersSubKey \
	"Parameters\\Tcpip\\"

set ::net_info::adaptersKey \
	"HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Parameters\\Adapters\\"
set ::net_info::interfaceKey \
	"HKEY_LOCAL_MACHINE\\SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Parameters\\Interfaces\\"

#-------------------------------------------------------------------------------
# Return specific value from specified key in registry
#-------------------------------------------------------------------------------
proc ::net_info::getValue {key valueName} {
	if [catch {registry get $key $valueName} value] {
		::debug::msg 1 "Unable to read value \"$valueName\" in key \"$key\""
		return ""
	} else {
		return $value
	}
}

#-------------------------------------------------------------------------------
# Return level of support for that operationg system
# It can be full, partial or none
#-------------------------------------------------------------------------------
proc ::net_info::getSupportLevel {} {
	return "full"
}

#-------------------------------------------------------------------------------
# Return computer operating system name
#-------------------------------------------------------------------------------
proc ::net_info::getOS {} {
	if { $::tcl_platform(os) == "Windows NT" } {
		switch -exact -- $::tcl_platform(osVersion) \
			"4.0" { return "Windows NT 4.0" } \
			"5.0" { return "Windows 2000" } \
			"5.1" { return "Windows XP" } \
			"5.2" { return "Windows Server 2003" } \
			"6.0" { return "Windows Vista" } \
			"6.1" { return "Windows Seven" } \
			default { return $::tcl_platform(os) }
	} else {
		return $::tcl_platform(os)
	}
}

#-------------------------------------------------------------------------------
# Return computer operating system version
#-------------------------------------------------------------------------------
proc ::net_info::getOSVersion {} {
	return $::tcl_platform(osVersion)
}

#-------------------------------------------------------------------------------
# Return computer current logged user
#-------------------------------------------------------------------------------
proc ::net_info::getCurrentUser {} {
	return $::tcl_platform(user)
}

#-------------------------------------------------------------------------------
# Return computer name
#-------------------------------------------------------------------------------
proc ::net_info::getComputerName {} {
	return [info hostname]
}

#-------------------------------------------------------------------------------
# Return if computer is in a Microsoft domain
#-------------------------------------------------------------------------------
proc ::net_info::isInMicrosoftDomain {} {
	if {[twapi::min_os_version 5]} {
		array set domaininfo [twapi::get_primary_domain_info -type]
		if {$domaininfo(-type) eq "workgroup"} {
			return false
		} else {
			return true
		}
	} else {
		if {[ ::net_info::getValue $::net_info::winlogonKey "CachePrimaryDomain" ] eq ""} {
			return false
		} else {
			return true
		}
	}
}

#-------------------------------------------------------------------------------
# Return Microsoft workgroup name
#-------------------------------------------------------------------------------
proc ::net_info::getMicrosoftWorkgroupName {} {
	set domainname ""
	if {[twapi::min_os_version 5]} {
		array set domaininfo [twapi::get_primary_domain_info -name]
		set domainname $domaininfo(-name)
	}

	if {$domainname == ""} {
		set value [ ::net_info::getValue $::net_info::explorerKey "Last Domain" ]
		set pos [string last "," $value]
		incr pos
		set domainname [string range $value $pos end]
	}

	return $domainname
}

#-------------------------------------------------------------------------------
# Return Microsoft domain name
#-------------------------------------------------------------------------------
proc ::net_info::getMicrosoftDomainName {} {
	set domainname ""
	if {[twapi::min_os_version 5]} {
		array set domaininfo [twapi::get_primary_domain_info -name]
		set domainname $domaininfo(-name)
	}

	if {$domainname == ""} {
		set domainname [ ::net_info::getValue $::net_info::winlogonKey "DefaultDomainName" ]
	}

	return $domainname
}

#-------------------------------------------------------------------------------
# Return dns domain name
#-------------------------------------------------------------------------------
proc ::net_info::getDnsDomainName {} {
	set domain ""
	if {[twapi::min_os_version 5]} {
		array set domaininfo [twapi::get_primary_domain_info -dnsdomainname]
		set domain $domaininfo(-dnsdomainname)
	}

	if {$domain == ""} {
		set key $::net_info::servicesKey$::net_info::computerTcpIpParametersSubKey
		set domain [ ::net_info::getValue $key "DhcpDomain" ]
		if { $domain == "" } {
			set domain [ ::net_info::getValue $key "Domain" ]
		}
	}
	return $domain
}

#-------------------------------------------------------------------------------
# Return interface check list
# Try to prevent finding data for network cards which have been removed
# but which still have data in registry
# Still existing one will have their own "Control" key in that part of registry
# so get all keys which have their own "Control" key
#-------------------------------------------------------------------------------
proc ::net_info::getInterfaceCheckList {} {
	set result {}
	# Get all known network card
	if { ![catch {registry keys $::net_info::classesKey} classKeys] } {
		foreach classKey $classKeys {
			set key "$::net_info::classesKey$classKey\\"
			if { ! [catch {registry keys $key #\{*\} } subKeys] } {
				foreach subKey $subKeys {
					set fullKey "$key$subKey\\"
					if { ! [catch {registry keys $fullKey "Control"} subSubKeys] } {
						if { $subSubKeys != {} } {
							lappend result [ string range $subKey 1 end ]
						}
					}
				}
			}
		}
	}
	return $result
}

#-------------------------------------------------------------------------------
# Return domain name servers
#-------------------------------------------------------------------------------
proc ::net_info::getDnsServer {} {
	set dns ""
	set key $::net_info::servicesKey$::net_info::computerTcpIpParametersSubKey
	set dns [ ::net_info::getValue $key "DhcpNameServer" ]
	if { $dns == "" } {
		set dns [ ::net_info::getValue $key "NameServer" ]
	}
	return $dns
}

#-------------------------------------------------------------------------------
# Return MAC address data from getmac
#-------------------------------------------------------------------------------
proc ::net_info::getMacDataByGetmac {} {
	set macList {}
	set searchString "\\Device\\Tcpip_"
	if { ! [catch { eval ::exec -- "getmac.exe /V /NH /FO csv" } rawData ] } {
		set Lines [ split $rawData \n ]
		foreach line $Lines {
			if { [ string first $searchString $line ] != -1 } {
				set datas [ split $line , ]
				# type, description, mac address, service name
				set id [ lindex $datas 3 ]
				set long [ string length $searchString ]
				set id [ string range $id $long end ]
				set long [ expr { [ string length $id ] - 2 } ]
				set id [ string range $id 1 $long ]
				set mac [ string map { - : } \
							  [ string toupper [ lindex $datas 2 ] ] ]
				set long [ expr { [ string length $mac ] -2 } ]
				set mac [ string range $mac 1 $long ]
				lappend macList $id
				lappend macList $mac
			}
		}
	}
	return $macList
}

#-------------------------------------------------------------------------------
# Return MAC address data from NetBEUI and IPX
# http://support.microsoft.com/kb/118623/fr
#-------------------------------------------------------------------------------
proc ::net_info::getMacDataByNetBEUI {} {
	set macList {}
	set searchString "NetBT_Tcpip_"
	if { [ info exists starkit::topdir ] } {
		set filename [ file join $starkit::topdir getmac.bat ]
		if { [ file exists $filename ] } {
			if { [ catch { open $filename "r" } fileid ] } {
                ::debug::msg 1 "File $filename cannot be open"
                set contents ""
			} else {
                set contents [ read $fileid ]
                close $fileid
			}
			if { [ catch { open getmac.bat "w" } fileid ] } {
                ::debug::msg 1 "Impossible to build file getmac.bat"
			} else {
                puts $fileid $contents
                close $fileid
			}
		} else {
			::debug::msg 1 "File $filename not available"
		}
	}
	if { ! [catch { eval ::exec -- "getmac.bat" } rawData ] } {
		set Lines [ split $rawData \n ]
		foreach line $Lines {
			set pos [ string first $searchString $line ]
			if { $pos != -1 } {
				set data [ split [ string replace $line 0 \
									   [ expr { $pos + [ string length $searchString ] - 1 } ] ] ]
				lappend macList [ lindex $data 0 ]
				set mac [ string toupper [ lindex $data 1 ] ]
				set frmMac "[string range $mac 1 2]:[string range $mac 3 4]:"
				append frmMac "[string range $mac 5 6]:[string range $mac 7 8]:"
				append frmMac "[string range $mac 9 10]:[string range $mac 11 12]"
				lappend macList $frmMac
			}
		}
	}
	if { [ info exists starkit::topdir ] } {
		catch { [ file delete "getmac.bat" ] }
	}
	return $macList
}

# #-------------------------------------------------------------------------------
# # Return MAC address data with TWAPI
# #-------------------------------------------------------------------------------
# proc ::net_info::getMacDataByTWAPI {} {
# 	set macList {}

# 	package require twapi

# 	set searchString "NetBT_Tcpip_"
# 	foreach ifx [twapi::get_netif_indices] {
# 		array set netif [twapi::get_netif_info $ifx -all]
# 		lappend macList $netif(-adaptername)
# 		lappend macList [ string map { - : } \
	# 							  [ string toupper $netif(-physicaladdress) ] ]
# 	}
# 	return $macList
# }

# #-------------------------------------------------------------------------------
# # Return MAC address data from WMI
# #
# # Use of WMI in TCL: http://wiki.tcl.tk/11633
# #
# # Win32_NetworkAdapter: http://msdn2.microsoft.com/en-us/library/aa394216.aspx
# # Script examples: http://msdn2.microsoft.com/en-us/library/aa394585.aspx
# #
# # Win32_NetworkAdapterConfiguration:
# #   http://msdn2.microsoft.com/en-us/library/aa394217.aspx
# # Script example:
# #   http://www.commentcamarche.net/forum/affich-1953934-script-et-adresse-ip#12
# #-------------------------------------------------------------------------------
# proc ::net_info::getMacDataByWMI {} {
# 	set macList {}

# 	package require tcom

# 	set wbemFlagReturnImmediately [scan \x10 %c]
# 	set wbemFlagForwardOnly       [scan \x20 %c]

# 	set objWMIService [::tcom::ref getobject "winmgmts:{impersonationLevel=impersonate}!#"]
# 	set colItems \
	# 		[ $objWMIService ExecQuery \
	# 			  {SELECT * FROM Win32_NetworkAdapterConfiguration WHERE IPEnabled=TRUE} \
	# 			  WQL [expr {$wbemFlagReturnImmediately + $wbemFlagForwardOnly}]]

# 	::tcom::foreach objItem $colItems {
# 		set propSet [ $objItem Properties_ ]
# 		lappend macList [ [ $propSet Item SettingId ] Value ]
# 		lappend macList [ [ $propSet Item MACAddress ] Value ]
# 	}

# 	return $macList
# }

#-------------------------------------------------------------------------------
# Get MAC address of all interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getMacData {} {
	set ::net_info::MacData {}
	# try by getmac
	if { $::net_info::MacData == {} } {
		set ::net_info::MacData [ ::net_info::getMacDataByGetmac ]
		::debug::msg 2 "Run getMacDataByGetmac"
	}
	# try by NetBEUI and IPX
	if { $::net_info::MacData == {} } {
		set ::net_info::MacData [ ::net_info::getMacDataByNetBEUI ]
		::debug::msg 2 "Run getMacDataByNetBEUI"
	}
	# 	# try by TWAPI
	# 	if { $::net_info::MacData == {} } {
	# 		set ::net_info::MacData [ ::net_info::getMacDataByTWAPI ]
	# 	}
	# 	# try by WMI
	# 	if { $::net_info::MacData == {} } {
	# 		set ::net_info::MacData [ ::net_info::getMacDataByWMI ]
	# 	}
}

#-------------------------------------------------------------------------------
# Return mac address
#-------------------------------------------------------------------------------
proc ::net_info::getMacAddress {id} {
	set mac ""
	set pos [ lsearch $::net_info::MacData $id ]

	if {$pos != -1 } {
		incr pos
		set mac [ lindex $::net_info::MacData $pos ]
	}
	return $mac
}

#-------------------------------------------------------------------------------
# Return all informations regarding a computer
#-------------------------------------------------------------------------------
proc ::net_info::getComputer {} {
	array set Computer [::net_info::initComputer]

	set Computer(Os)             [::net_info::getOS]
	set Computer(OsVersion)      [::net_info::getOSVersion]
	set Computer(OsSupportLevel) [::net_info::getSupportLevel]
	set Computer(Username)       [::net_info::getCurrentUser]
	set Computer(Hostname)       [::net_info::getComputerName]
	set Computer(DnsServer)      [::net_info::getDnsServer]
	set Computer(DnsDomainName)  [::net_info::getDnsDomainName]
	if {[::net_info::isInMicrosoftDomain]} {
		set Computer(MicrosoftDomain) 1
		set Computer(MicrosoftDomainType) domain
		set Computer(MicrosoftDomainName) \
			[::net_info::getMicrosoftDomainName]
	} else {
		set Computer(MicrosoftDomain) 0
		set Computer(MicrosoftDomainType) workgroup
		set Computer(MicrosoftDomainName) \
			[::net_info::getMicrosoftWorkgroupName]
	}

	return [array get Computer]
}

#-------------------------------------------------------------------------------
# Return all informations regarding an interface
#-------------------------------------------------------------------------------
proc ::net_info::getInterface {cardKey checklist} {
	set id [ ::net_info::getValue $cardKey "ServiceName" ]
	if { $id != "" } {
		# Check if this board  is still available as it could be a board
		# which have been there and since removed
		if {[ lsearch $checklist $id ] > -1  } {
			set cardDataKey $::net_info::servicesKey
			append cardDataKey "$id\\$::net_info::cardTcpIpParametersSubKey"
			puts "cardKey: $cardKey"
			puts "cardDataKey: $cardDataKey"
			if ![catch {registry keys $cardDataKey} keys] {
				# That card is still known
				array set Interface [ ::net_info::initInterface ]
				set Interface(Id) $id
				set Interface(Description) \
					[ ::net_info::getValue $cardKey "Description" ]
				set Interface(MacAddress) \
					[ ::net_info::getMacAddress $id ]
				set Interface(DnsServer) \
					[ ::net_info::getDnsServer ]
				set Interface(DnsDomainName) \
					[ ::net_info::getDnsDomainName ]

				set dhcp [ ::net_info::getValue $cardDataKey "EnableDHCP" ]
				set Interface(Dhcp) $dhcp
				if { $dhcp == 1 } {
					set Interface(DhcpServer) \
						[ ::net_info::getValue $cardDataKey "DhcpServer" ]
					set Interface(IpAddress) \
						[ ::net_info::getValue $cardDataKey "DhcpIPAddress" ]
					set Interface(SubnetMask) \
						[ ::net_info::getValue $cardDataKey "DhcpSubnetMask" ]
					set Interface(Gateway) \
						[ ::net_info::getValue $cardDataKey "DhcpDefaultGateway" ]
		
					# If some are not found there, try at alternate key
					set cardDataKey2 $::net_info::interfaceKey
					append cardDataKey2 "$id"
					if { $Interface(DhcpServer) == "" } {
						set Interface(DhcpServer) \
							[ ::net_info::getValue $cardDataKey2 "DhcpServer" ]
					}
					if {$Interface(IpAddress) == "" } {
						set Interface(IpAddress) \
							[ ::net_info::getValue $cardDataKey2 "DhcpIPAddress" ]
					}
					if {$Interface(SubnetMask) == "" } {
						set Interface(SubnetMask) \
							[ ::net_info::getValue $cardDataKey2 "DhcpSubnetMask" ]
					}
					if {$Interface(Gateway) == "" } {
						set Interface(Gateway) \
						[	 ::net_info::getValue $cardDataKey2 "DhcpDefaultGateway" ]
					}
				} else {
					set Interface(IpAddress) \
						[ ::net_info::getValue $cardDataKey "IPAddress" ]
					set Interface(SubnetMask) \
						[ ::net_info::getValue $cardDataKey "SubnetMask" ]
					set Interface(Gateway) \
						[ ::net_info::getValue $cardDataKey "DefaultGateway" ]
				}
			} else {
				set Interface {}
			}
		} else {
			set Interface {}
		}
	} else {
		set Interface {}
	}

	return [ array get Interface ]
}

#-------------------------------------------------------------------------------
# Return all informations regarding all interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInterfaces {} {
	set interfaces {}
	set key $::net_info::networkCardsKey
	# Get all known network card
	if [catch {registry keys $key} keys] {
		::debug::msg 1 "Unable to get keys \"$key\" from registry."
		::debug::msg 1 "Error: $keys"
	} else {
		# Build interface check list
		set checkList [ ::net_info::getInterfaceCheckList ]
		# Get MAC address informations
		::net_info::getMacData
		# Process each known network card
		foreach subkey $keys {
			set interfaceData \
				[ ::net_info::getInterface "$key$subkey\\" $checkList ]
			if { $interfaceData != {} } {
				lappend interfaces $interfaceData
			}
		}
	}

	return $interfaces
}

#-------------------------------------------------------------------------------
# Return all informations regarding computer and interfaces
#-------------------------------------------------------------------------------
proc ::net_info::getInfos {} {
	return [list Computer [::net_info::getComputer] \
				Interfaces [::net_info::getInterfaces]]
}

#-------------------------------------------------------------------------------
# If that file is called directly, display test suite on command line
#-------------------------------------------------------------------------------

if {![info exist ::netinfo]} {
	::net_info::testSuite
}
