#!/usr/bin/env tclsh
# -*- mode: tcl; coding: utf-8-unix; -*-
# Time-stamp: <2009-01-25 18:56:36 alain>%
# $Id: netinfo.tcl,v 1.9 2009-05-27 13:54:53 cvsalain Exp $
#-------------------------------------------------------------------------------
#
# Way of doing for pseudo tab is from
#    Richard Suchenwirth, http://wiki.tcl.tk/17297
#
# Addition of differenciation between Microsoft workgroup and domain
# 20090124 version 1.2
#   Addition of operating system support level
# 20090125 version 1.3
#   Addition Windows 98 support
#   Addition of file save as option
# 20090527 version 1.4
#   Addition Darwin support
#   Start to prepare for IP V6
# 20160515 version 1.5
#   Correct Windows seven management
#-------------------------------------------------------------------------------

# To be able to use/make starkit/starpack
package provide app-netinfo 1.5

# Adjust auto_path to contain script home and current directory
set script_home [string trimright [file dirname [info script]] ./]
set script_home [file join [pwd] $script_home]
lappend auto_path $script_home

# Adjust auto_path to contain lib path
set lib_path [file join [file dirname [file dirname $script_home]] "lib"]
if {[lsearch $auto_path $lib_path] == -1} {
	lappend auto_path $lib_path
}

# Define some application constants
set ::APPNAME "Network informations"
set ::VERSION [ package require app-netinfo ]
set ::APPSTRING "$APPNAME $VERSION"
set ::AUTHOR "Jack.R"

# Specify required packages
set ::TCLVERSION [ package require Tcl ]
set ::MCVERSION [ package require msgcat ]
namespace import ::msgcat::*
package require lngcode
namespace import ::lngcode::*
package require debug

set ::netinfo 1

namespace eval ::netinfo:: {
	variable Obj
	variable Idx
	variable Hints
	variable StatusText

	variable IfQty
	variable CurrentTab

	set StatusText "                                                           "
}

#*******************************************************************************
#                                   GUI                                        *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Build main menu
#-------------------------------------------------------------------------------
proc ::netinfo::BuildMenu {mb} {
	set ::netinfo::Obj(mnu) [ menu $mb -borderwidth 1 ]
	set index 0

	# File
	set ::netinfo::Obj(mnuFile) [ menu $::netinfo::Obj(mnu).file -tearoff 0 ]
	$::netinfo::Obj(mnu) add cascade -menu $::netinfo::Obj(mnuFile)
	set ::netinfo::Idx(mnuFile) [ incr index ]
	# Option
	set ::netinfo::Obj(mnuTools) [ menu $::netinfo::Obj(mnu).option -tearoff 0 ]
	$::netinfo::Obj(mnu) add cascade -menu $::netinfo::Obj(mnuTools)
	set ::netinfo::Idx(mnuTools) [ incr index ]
	# Help
	set ::netinfo::Obj(mnuHelp) [ menu $::netinfo::Obj(mnu).help -tearoff 0 ]
	$::netinfo::Obj(mnu) add cascade -menu $::netinfo::Obj(mnuHelp)
	set ::netinfo::Idx(mnuHelp) [ incr index ]

	# File menu
	set index -1
	$::netinfo::Obj(mnuFile) add command -command ::netinfo::FileSaveAs
	set ::netinfo::Idx(mnuFileSaveAs) [ incr index ]
	$::netinfo::Obj(mnuFile) add separator
	set ::netinfo::Idx(mnuFileSeparator) [ incr index ]
	$::netinfo::Obj(mnuFile) add command -command ::netinfo::FileQuit
	set ::netinfo::Idx(mnuFileQuit) [ incr index ]

	# Option menu
	set index -1
	$::netinfo::Obj(mnuTools) add command -command ::netinfo::BuildTabs
	set ::netinfo::Idx(mnuRefresh) [ incr index ]
	set ::netinfo::Obj(mnuLanguage) [ menu $::netinfo::Obj(mnuTools).language -tearoff 0 ]
	$::netinfo::Obj(mnuTools) add cascade -menu $::netinfo::Obj(mnuLanguage)
 	set ::netinfo::Idx(mnuToolsLanguage) [ incr index ]

	# Language
	if { [ ::lngcode::getCodeList "$::script_home/msg" ] } {
		foreach code $::lngcode::CodeList {
			$::netinfo::Obj(mnuLanguage) add command \
				-label [ getLngName $code 0 ] \
				-command [ list ::netinfo::SwitchLanguage $code ]
		}
	}

	# Help menu
	set index -1
	$::netinfo::Obj(mnuHelp) add command -command ::netinfo::About
	set ::netinfo::Idx(mnuHelpAbout) [ incr index ]
	$::netinfo::Obj(mnuHelp) add command -command ::netinfo::Versions
	set ::netinfo::Idx(mnuHelpVersions) [ incr index ]

	. config -menu $::netinfo::Obj(mnu)
}

#-------------------------------------------------------------------------------
# Build status bar
#-------------------------------------------------------------------------------
proc ::netinfo::BuildStatusBar {sb} {
	set f [ frame $sb ]
	label $sb.label -textvariable ::netinfo::StatusText \
		-relief sunken -border 1 -anchor w
	pack $sb.label -side left -padx 2 -expand yes -fill both

	return $f
}

#-------------------------------------------------------------------------------
# Raise tab
#-------------------------------------------------------------------------------
proc ::netinfo::RaiseTab {idx} {
	for { set i 0 } { $i <= $::netinfo::IfQty } { incr i } {
		set btnName "tab$i"
 		$::netinfo::Obj($btnName) config -borderwidth 0
	}
	set btnName "tab$idx"
 	$::netinfo::Obj($btnName) config -borderwidth 1

	for { set i 0 } { $i <= $::netinfo::IfQty } { incr i } {
		set fName "f$i"
 		if { $i ne $idx } { pack forget $::netinfo::Obj($fName) }
	}
	set fName "f$idx"
    pack $::netinfo::Obj($fName) -fill both -expand 1

	focus $::netinfo::Obj($btnName)
	set ::netinfo::CurrentTab $idx
}

#-------------------------------------------------------------------------------
# Add item
# If title = "-" then it is just a spacer
#-------------------------------------------------------------------------------
proc ::netinfo::AddItem {f i title value} {

	if {![info exists ::netinfo::Obj($f.t$i)]} {
		set ::netinfo::Obj($f.t$i) [ label $f.t$i -anchor nw -pady 0 ]
		if { $title != "-" } {
			$::netinfo::Obj($f.t$i) configure -background "white smoke"
		}
	}
	if {![info exists ::netinfo::Obj($f.v$i)]} {
		if { $title != "-" } {
			set ::netinfo::Obj($f.v$i) [ entry $f.v$i -width 50 -borderwidth 1 \
											 -selectborderwidth 2 -relief flat \
											 -state readonly ]
			$::netinfo::Obj($f.v$i) configure -readonlybackground white
		} else {
			set ::netinfo::Obj($f.v$i) [ label $f.v$i -anchor nw -width 50 -pady 0 ]
		}
	}

	if { $title != "-" } {
		set ::netinfo::Obj($f.t$i.text) $title
		set ::netinfo::Obj($f.v$i.text) $value
		$::netinfo::Obj($f.t$i) configure -textvariable ::netinfo::Obj($f.t$i.text)
		$::netinfo::Obj($f.v$i) configure -textvariable ::netinfo::Obj($f.v$i.text)
	}
    grid $f.t$i $f.v$i -sticky ew -pady 0
    grid configure $f.t$i -pady 1
	grid configure $f.v$i -pady 1
}

#-------------------------------------------------------------------------------
# Build os tab
#-------------------------------------------------------------------------------
proc ::netinfo::BuildOsTab {idx computerdata} {

	set btnName "tab$idx"
	set fName "f$idx"
	array set Computer $computerdata

	if {![info exists ::netinfo::Obj($btnName)]} {
		set ::netinfo::Obj($btnName) \
			[ button $::netinfo::Obj(ftab).$btnName \
				  -relief raised -borderwidth 2 \
				  -command [ list ::netinfo::RaiseTab $idx ] ]
	}
	$::netinfo::Obj($btnName) configure -text [ mc "msg Computer" ]
	pack $::netinfo::Obj($btnName) -side left -ipadx 5

	# Tab contents
	if {![info exists ::netinfo::Obj($fName)]} {
		set ::netinfo::Obj($fName) [ frame $::netinfo::Obj(fcontents).$fName \
										 -padx 10 -pady 10 \
										 -relief raised -borderwidth 1 ]
	}
	set f $::netinfo::Obj($fName)

	::netinfo::AddItem $f "os" [ mc "msg Os" ] $Computer(Os)
	::netinfo::AddItem $f "version" [ mc "msg Os version" ] \
		$Computer(OsVersion)
	::netinfo::AddItem $f "support" [ mc "msg Os support" ] \
		$Computer(OsSupportLevel)
	::netinfo::AddItem $f 0 "-" ""
	::netinfo::AddItem $f "user" [ mc "msg User" ] $Computer(Username)
	::netinfo::AddItem $f "computername" [ mc "msg Computer name" ] \
		$Computer(Hostname)
	::netinfo::AddItem $f "dnsDomain" [ mc "msg DNS domain" ] \
		$Computer(DnsDomainName)
	::netinfo::AddItem $f "dnsServer" [ mc "msg DNS server" ] \
		$Computer(DnsServer)
	::netinfo::AddItem $f 1 "-" ""
	if {$Computer(MicrosoftDomain)} {
		::netinfo::AddItem $f "microsoftdomaintype" \
			[ mc "msg Microsoft domain type" ] [mc "msg domain"]
	} else {
		::netinfo::AddItem $f "microsoftdomaintype" \
			[ mc "msg Microsoft domain type" ] [mc "msg workgroup"]
	}
	::netinfo::AddItem $f "microsoftdomainname" [ mc "msg Microsoft domain name" ] \
		$Computer(MicrosoftDomainName)

    pack $::netinfo::Obj($fName) -fill both -expand 1
}

#-------------------------------------------------------------------------------
# Build interface tab
#-------------------------------------------------------------------------------
proc ::netinfo::BuildInterfaceTab {idx interfacedata} {

	set btnName "tab$idx"
	set fName "f$idx"
	array set Interface $interfacedata

	if {![info exists ::netinfo::Obj($btnName)]} {
		set ::netinfo::Obj($btnName) [ button $::netinfo::Obj(ftab).$btnName \
										   -relief raised -borderwidth 2 \
										   -command [ list ::netinfo::RaiseTab $idx ] ]
	}
	$::netinfo::Obj($btnName) configure -text $idx
	pack $::netinfo::Obj($btnName) -side left -ipadx 5

	# Tab contents
	if {![info exists ::netinfo::Obj($fName)]} {
		set ::netinfo::Obj($fName) [ frame $::netinfo::Obj(fcontents).$fName \
										 -padx 10 -pady 10 \
										 -relief raised -borderwidth 1 ]
	}
	set f $::netinfo::Obj($fName)

	::netinfo::AddItem $f "interface" [ mc "msg Interface" ] $Interface(Id)
	::netinfo::AddItem $f 0 "-" ""
	::netinfo::AddItem $f "description" [ mc "msg Description" ] \
		$Interface(Description)

	::netinfo::AddItem $f "mac" [ mc "msg Mac" ] $Interface(MacAddress)
	::netinfo::AddItem $f "ip" [ mc "msg IP" ] $Interface(IpAddress)
	::netinfo::AddItem $f "subnet" [ mc "msg Subnet" ] $Interface(SubnetMask)
# 	::netinfo::AddItem $f "ipV6" [ mc "msg IPV6" ] $Interface(IpV6)
	::netinfo::AddItem $f "gateway" [ mc "msg Gateway" ] \
		$Interface(Gateway)

	if {$Interface(Dhcp) == 1} {
		::netinfo::AddItem $f "dhcp" [ mc "msg DHCP" ] [ mc "msg Yes" ]
	} elseif {$Interface(Dhcp) == 0} {
		::netinfo::AddItem $f "dhcp" [ mc "msg DHCP" ] [ mc "msg No" ]
	} else {
		::netinfo::AddItem $f "dhcp" [ mc "msg DHCP" ] "?" ]
	}
	::netinfo::AddItem $f "dhcpserver" [ mc "msg DHCP server" ] \
		$Interface(DhcpServer)

    pack $::netinfo::Obj($fName) -fill both -expand 1
}

#-------------------------------------------------------------------------------
# Build all tabs
#-------------------------------------------------------------------------------
proc ::netinfo::BuildTabs {} {
	# Hide existing tabs
	for { set idx 1 } { $idx <= $::netinfo::IfQty } { incr idx } {
		set btnName "tab$idx"
		set fName "f$idx"

		pack forget $::netinfo::Obj($btnName)
		pack forget $::netinfo::Obj($fName)

	}
	# Get informations
	array set Infos [::net_info::getInfos]
	# Build new tabs
	set ::netinfo::IfQty 0
	BuildOsTab $::netinfo::IfQty $Infos(Computer)

	foreach InterfaceData $Infos(Interfaces) {
		array set Interface $InterfaceData
		# Show only active ones
		if {($Interface(IpAddress)!= "") && ($Interface(IpAddress)!= "0.0.0.0")} {
			incr ::netinfo::IfQty
			BuildInterfaceTab $::netinfo::IfQty $InterfaceData
		}
	}
	::netinfo::RaiseTab 0
}

#-------------------------------------------------------------------------------
# Build window contents
#-------------------------------------------------------------------------------
proc ::netinfo::BuildContents {c} {

	# Frame for contents
	set ::netinfo::Obj(fcontents) [ frame $c ]
	# Frame for tab buttons
	set ::netinfo::Obj(ftab) [ frame $::netinfo::Obj(fcontents).tab \
								   -relief raised -borderwidth 1 ]
	pack $::netinfo::Obj(ftab) -side top -fill x -anchor w

	return $::netinfo::Obj(fcontents)
}

#-------------------------------------------------------------------------------
# Build GUI
#-------------------------------------------------------------------------------
proc ::netinfo::BuildGUI {} {
	::netinfo::BuildMenu .mb
	pack [ ::netinfo::BuildContents .fcontents ] -fill both -expand 1
	pack [ ::netinfo::BuildStatusBar .statusBar ] -side bottom -fill x -pady 2
}

#*******************************************************************************
#                                 Languages                                    *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Load a new language file
#-------------------------------------------------------------------------------
proc ::netinfo::LoadLanguage {code} {
	if {$code == {} } {
		# get user default language
		set code [ eval "::lngcode::getValidCode [ mcpreferences ]" ]
	} else {
		# be sure supplyed code is a valid language code
		set code [ ::lngcode::getValidCode $code ]
	}
	mclocale $code
	# load language file stored in "msg" folder
	mcload "$::script_home/msg"
}

#-------------------------------------------------------------------------------
# Change language
#-------------------------------------------------------------------------------
proc ::netinfo::SwitchLanguage {code} {
	::netinfo::LoadLanguage $code
	::netinfo::UpdateAll
}

#*******************************************************************************
#                              Text changes                                    *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Update a menu item
#-------------------------------------------------------------------------------
proc ::netinfo::SetMnuTxt {menu index name hint} {
	set underline [ string first & $name ]
	if { $underline < 0 } {
		set label $name
	} else {
		if { $underline == 0 } {
			set label [ string range $name 1 end ]
		} else {
			set label [ string range $name 0 [ expr { $underline - 1 } ] ]
			append label [ string range $name [ expr { $underline + 1 } ] end ]
		}
	}

	$menu entryconfigure $index -label $label
	if { $underline >= 0 } {
		$menu entryconfigure $index -underline $underline
	}

	# Hint
	set ::netinfo::Hints($menu-$index) $hint
}

#-------------------------------------------------------------------------------
# Update main window title
#-------------------------------------------------------------------------------
proc ::netinfo::UpdateMainWindow {} {
	wm title . [ mc "msg Net info" ]
}

#-------------------------------------------------------------------------------
# Update menu
#-------------------------------------------------------------------------------
proc ::netinfo::UpdateMenu {} {
	# File menu
	::netinfo::SetMnuTxt $::netinfo::Obj(mnu) $::netinfo::Idx(mnuFile) \
		[ mc "msg &File" ] [ mc "hint File" ]
	::netinfo::SetMnuTxt $::netinfo::Obj(mnuFile) $::netinfo::Idx(mnuFileSaveAs) \
		[ mc "msg &Save as ..." ] [ mc "hint Save as" ]
	::netinfo::SetMnuTxt $::netinfo::Obj(mnuFile) $::netinfo::Idx(mnuFileQuit) \
		[ mc "msg &Quit" ] [ mc "hint Quit" ]

	# Option menu
	::netinfo::SetMnuTxt $::netinfo::Obj(mnu) $::netinfo::Idx(mnuTools) \
		[ mc "msg &Tools" ] [ mc "hint Tools" ]
	::netinfo::SetMnuTxt $::netinfo::Obj(mnuTools) $::netinfo::Idx(mnuRefresh) \
		[ mc "msg &Refresh" ] [ mc "hint Refresh" ]
	::netinfo::SetMnuTxt $::netinfo::Obj(mnuTools) $::netinfo::Idx(mnuToolsLanguage) \
		[ mc "msg &Language" ] [ mc "hint Language" ]

	# Help menu
	::netinfo::SetMnuTxt $::netinfo::Obj(mnu) $::netinfo::Idx(mnuHelp) \
		[ mc "msg &Help" ] [ mc "hint Help" ]
	::netinfo::SetMnuTxt $::netinfo::Obj(mnuHelp) $::netinfo::Idx(mnuHelpAbout) \
		[ mc "msg &About ..." ] [ mc "hint About" ]
	::netinfo::SetMnuTxt $::netinfo::Obj(mnuHelp) $::netinfo::Idx(mnuHelpVersions) \
		[ mc "msg &Versions ..." ] [ mc "hint Versions" ]
}


#-------------------------------------------------------------------------------
# Update contents
#-------------------------------------------------------------------------------
proc ::netinfo::UpdateContents {} {
	::netinfo::BuildTabs
}

#-------------------------------------------------------------------------------
# Update complete GUI
#-------------------------------------------------------------------------------
proc ::netinfo::UpdateAll {} {
	::netinfo::UpdateMainWindow
	::netinfo::UpdateMenu
	::netinfo::UpdateContents
}

#*******************************************************************************
#                               Messages boxes                                 *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Close box
#-------------------------------------------------------------------------------
proc ::netinfo::CloseBox { window } {
	# be sure to get back focus, keyboard and mouse action to main window
	grab release $window
	grab .
	set btnName "tab$::netinfo::CurrentTab"
	focus $::netinfo::Obj($btnName)
	destroy $window
}

#-------------------------------------------------------------------------------
# About box
#-------------------------------------------------------------------------------
proc ::netinfo::About {} {
    global APPNAME
    global VERSION
    global AUTHOR

    set aboutstring [ mc "msg Program: \t%s" "$::APPNAME" ]
	append aboutstring "\n"
    append aboutstring [ mc "msg Version: \t%s" "$::VERSION" ]
	append aboutstring "\n\n"
    append aboutstring "(c) $::AUTHOR 2007 - 2016"
	append aboutstring "\n\n"
    append aboutstring "This software is distributed under the terms of NOL "
	append aboutstring "(No Obligation License) "
    append aboutstring "http://wfr.tcl.tk/nol\n\n"

    if { [ winfo exists .aboutWin ] } {
        focus .aboutWin.close
        return
    }

    toplevel .aboutWin -padx 5 -pady 5 -relief raised -borderwidth 3
    wm overrideredirect .aboutWin 1

    frame .aboutWin.upper
    frame .aboutWin.lower

	label .aboutWin.upper.label -text [ mc "msg About %s" "$APPNAME" ] \
		-relief groove
    text .aboutWin.lower.about -relief groove -width 40 -height 10 \
		-padx 5 -pady 5 -wrap word \
		-background [ .aboutWin.lower cget -background ]
    .aboutWin.lower.about insert 1.0 $aboutstring
    .aboutWin.lower.about configure -state disabled
    button .aboutWin.close -text [ mc "msg Close" ] \
		-command { ::netinfo::CloseBox .aboutWin }

     bind .aboutWin <KeyPress-Return> {::netinfo::CloseBox .aboutWin}

    grid .aboutWin.upper -in .aboutWin -row 1 -column 1 -sticky news \
		-padx 5 -pady 5
    grid .aboutWin.lower -in .aboutWin -row 2 -column 1 -sticky news \
		-padx 5 -pady 5
    grid .aboutWin.close -in .aboutWin -row 3 -column 1 -padx 5 -pady 5

    pack .aboutWin.upper.label -in .aboutWin.upper -expand 1 -fill both
    pack .aboutWin.lower.about -in .aboutWin.lower -expand 1 -fill both

	set deltax [ expr { ( [ winfo width . ] - [ winfo reqwidth .aboutWin ] ) / 2 } ]
	set deltay [ expr { ( [ winfo height . ] - [ winfo reqheight .aboutWin ] ) / 2 } ]
	set xpos [ expr { [ winfo x . ] + $deltax - 44 } ]
    set ypos [ expr { [ winfo y . ] + $deltay - 11 } ]
    wm geometry .aboutWin "+$xpos+$ypos"

    focus .aboutWin.close
    grab .aboutWin
}

#-------------------------------------------------------------------------------
# Versions box
#-------------------------------------------------------------------------------
proc ::netinfo::Versions {} {
    global APPSTRING
	global TCLVERSION
	global TKVERSION
	global MCVERSION

    set versionstring "TCL:\t\t$::TCLVERSION"
	append versionstring "\n"
    append versionstring "TK:\t\t$::TKVERSION"
	append versionstring "\n"
    append versionstring "MSGCAT:\t\t$::MCVERSION"
	append versionstring "\n"

    if { [ winfo exists .versionWin ] } {
        focus .versionWin.close
        return
    }

    toplevel .versionWin -padx 5 -pady 5 -relief raised -borderwidth 3
    wm overrideredirect .versionWin 1

    frame .versionWin.upper
    frame .versionWin.lower

	label .versionWin.upper.label -text "$::APPSTRING" -relief groove
    text .versionWin.lower.version -relief groove -width 30 -height 5 \
		-padx 5 -pady 5 -wrap word \
		-background [ .versionWin.lower cget -background ]
    .versionWin.lower.version insert 1.0 $versionstring
    .versionWin.lower.version configure -state disabled
    button .versionWin.close -text [ mc "msg Close" ] \
		-command { ::netinfo::CloseBox .versionWin }

 	bind .versionWin <KeyPress-Return> {::netinfo::CloseBox .versionWin}

    grid .versionWin.upper -in .versionWin -row 1 -column 1 -sticky news \
		-padx 5 -pady 5
    grid .versionWin.lower -in .versionWin -row 2 -column 1 -sticky news \
		-padx 5 -pady 5
    grid .versionWin.close -in .versionWin -row 3 -column 1 -padx 5 -pady 5

    pack .versionWin.upper.label -in .versionWin.upper -expand 1 -fill both
    pack .versionWin.lower.version -in .versionWin.lower -expand 1 -fill both

	set deltax [ expr { ( [ winfo width . ] - [ winfo reqwidth .versionWin ] ) / 2 } ]
	set deltay [ expr { ( [ winfo height . ] - [ winfo reqheight .versionWin ] ) / 2 } ]
	set xpos [ expr { [ winfo x . ] + $deltax - 22 } ]
    set ypos [ expr { [ winfo y . ] + $deltay } ]
    wm geometry .versionWin "+$xpos+$ypos"

    focus .versionWin.close
    grab .versionWin
}

#*******************************************************************************
#                           Event management                                   *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Save collected data in a file
#-------------------------------------------------------------------------------
proc ::netinfo::FileSaveAs {} {
    global APPSTRING

	set types [list \
				   [list [mc "msg Log files"] "*.log"] \
				   [list [mc "msg All Files"] ".*"] \
				  ]
	set filename "netinfo.log"
	set filename [tk_getSaveFile -filetypes $types -initialfile $filename]

	if {$filename != ""} {
		set defaultCursor [. cget -cursor]
		. configure -cursor watch
		update idletasks
		set title [format [mc "msg Data collected by %s"] $APPSTRING]
		::net_info::saveToFile $filename $title
		. configure -cursor $defaultCursor
	}
}

#-------------------------------------------------------------------------------
# Exit application
#-------------------------------------------------------------------------------
proc ::netinfo::FileQuit {} {
	exit
}

#-------------------------------------------------------------------------------
# Add short cut binding
#-------------------------------------------------------------------------------
proc ::netinfo::BindShortCut {} {
	bind . <F5> ::netinfo::BuildTabs
}

#-------------------------------------------------------------------------------
# Add needed binding for menu hints
#-------------------------------------------------------------------------------
proc ::netinfo::BindHint {} {
	bind Menu <<MenuSelect>> {
		if { [ catch { %W index active } idx ] } {
			set idx "none"
		}
		if { $idx != "none" } {
			set pos [ expr { [ string last "." %W ] + 1 } ]
			set id [ string range %W $pos end ]-$idx
			set id [ string map {"#" "."} $id ]
			if { [ catch { set label $::netinfo::Hints($id) } label ] } {
				set label "    "
			}
		} else {
			set label "    "
		}
		set ::netinfo::StatusText $label
		update idletasks
	}
}

#-------------------------------------------------------------------------------
# Command line help
#-------------------------------------------------------------------------------
proc ::netinfo::Help {} {
	puts "Syntax for $::argv0 :"
	puts "  $::argv0        : start it in graphic mode"
	puts "  $::argv0 -h     : display that help screen"
	puts "  $::argv0 --help : display that help screen"
	puts "  $::argv0 /?     : display that help screen"
	puts "  $::argv0 parameters"
	puts "  With parameters as:"
	puts "    -v level    : specify verbosity level for debugging"
	puts ""

}

#-------------------------------------------------------------------------------
# Main process
#-------------------------------------------------------------------------------
proc ::netinfo::Main {} {
	set ::TKVERSION [ package require Tk ]

	wm withdraw .

	::netinfo::BuildGUI
	::netinfo::LoadLanguage {}
	set ::netinfo::IfQty 0
	::netinfo::UpdateAll
	::netinfo::BindHint
	::netinfo::BindShortCut
	update idletasks

	wm deiconify .
	wm protocol . WM_DELETE_WINDOW exit
	raise .
	focus -force .
	wm geometry . 450x270
	wm minsize . [ winfo width . ] [ expr { [ winfo height . ] + 10 } ]
}


#-------------------------------------------------------------------------------
# Initialisation process
#-------------------------------------------------------------------------------
proc ::netinfo::Init {} {

	# Set default debug mode
	set ::debug::state 0
	set ::debug::level 1

	set mode "gui"
	if { $::argc > 0 } {
		# Read, check, store arguments
		set count 0
		while { $count < $::argc } {
			set arg [ lindex $::argv $count ]
			switch -exact -- $arg \
				{
					"-h"     -
					"--help" -
					"/?"     {
						set mode "help"
					}
					"--" { }
					"-v" {
						incr count
						if { $count < $::argc } {
							set ::debug::state 1
							set ::debug::level [ lindex $::argv $count ]
						}
					}
					default {
						set mode "help"
					}
				}
			incr count
		}
		set ::argv {}
	}

	# get proper packages now that we have set debug level
	package require netinfo_common

	# run it
	switch -exact -- $mode \
		{
			"help" { ::netinfo::Help }
			"gui" { ::netinfo::Main }
			default { ::netinfo::Help }
		}
}


# Run application
::netinfo::Init

