program Lineftp;
{$APPTYPE CONSOLE}

/////////////////////////////////////////////////////////////
//                                                         //
//                      Lineftp                            //
//             Command line ftp client                     //
//         Copyright (C) 2002 Alain JAFFRE                 //
//               http://jack.r.free.fr                     //
/////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////
//                Update history                           //
//                                                         //
//  V0.9.0  First version                                  //
//                                                         //
/////////////////////////////////////////////////////////////

/////////////////////////////////////////////////////////////
//                                                         //
//              Other components used                      //
//                                                         //
//  FtpClient: F.PIETTE                                    //
//    http://www.overbyte.be                               //
//                                                         //
/////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

{$DEFINE NOFORMS}

uses
  sysutils, classes, shellapi, wintypes,
  FtpCli;

type
  TSyncCmd = function : boolean of object;
  TClient = class
    TmpFile: string;
    Args: TStringList;     // command line argument list
    CmdProcessor: pchar;   // which comand processor is used

    FtpClient: TFtpClient;
    BufferSize: longint;   // ftp client buffer size
    SendSize: longint;     // how long is already send
    LastCmd: string;       // last ftp command
    // Client status
    Silent: boolean;       // silent mode, nothing is displayed
    Debug: boolean;        // display client dialogue
    Verbose: boolean;      // display server dialogue
    Bell: boolean;         // beep when command completed
    Prompting: boolean;    // prompt on multiple command
    Globbing: boolean;     // metacharacter expansion
    Hash: boolean;         // display # for each send buffer
    Anonymous: boolean;    // log in as anonymous

    constructor Create;
    destructor Destroy; override;
    { display management }
    procedure ShowMessage(MsgNo: integer; Text: string);
    procedure ShowAnswer(Text: string);
    procedure ShowError(ErrNo: integer;Text: string);
    { argument list management }
    procedure SplitArgs(Text: string);
    { command line management }
    function IsValidParameter(Param: string): boolean;
    function IsValidCommandLine: boolean;
    { ftp client management }
    function ExecuteCmd(Cmd: TSyncCmd):boolean;
    procedure FtpClientCommand(Sender: TObject; var Cmd: String);
    procedure FtpClientDisplay(Sender: TObject; var Msg: String);
    procedure FtpClientProgress(Sender: TObject; Count: Integer;
      var Abort: Boolean);
    procedure FtpClientRequestDone(Sender: TObject;
      RqType: TFtpRequest; Error: Word);
    procedure FtpClientSessionClosed(Sender: TObject;
        Error: Word);
    procedure FtpClientSessionConnected(Sender: TObject; Error: Word);
    { command }
    procedure CmdAppend(Src, Dest: string);
    procedure CmdBell;
    procedure CmdCd(Directory: string);
    procedure CmdClose;
    procedure CmdDelete(Name: string);
    procedure CmdDebug;
    procedure CmdDir(Mask: string);
    procedure CmdGet(Name: string);
    procedure CmdGlob;
    procedure CmdHash;
    procedure CmdHelp(Cmd: string);
    procedure CmdLcd(Directory: string);
    procedure CmdLs(Mask: string);
    procedure CmdMdelete(Name: string);
    procedure CmdMdir;
    procedure CmdMget(Name: string);
    procedure CmdMkdir(Name: string);
    procedure CmdMls;
    procedure CmdMput(Name: string);
    procedure CmdOpen(Host, Port: string);
    procedure CmdPass(Pass: string);
    procedure CmdPrompt;
    procedure CmdPut(Name: string);
    procedure CmdPwd;
    procedure CmdQuit;
    procedure CmdQuote(Cmd: string);
    procedure CmdRename(OldName, NewName: string);
    procedure CmdRmdir(Name: string);
    procedure CmdStatus;
    procedure CmdTrace;
    procedure CmdType(Typ: string);
    procedure CmdUser(User, Pass: string);
    procedure CmdVerbose;
    procedure ProcessCommand(Command: string);
    procedure ProcessCommandFile;
  end;

const
  Version = '0.90';

  DefaultBufferSize = 4096;
  // Basic prompt
  BasePrompt = 'LineFtp> ';
  UserPrompt = 'User (none): ';
  PassPrompt = 'Password (none): ';
  BasePort = '21';
  // Command line error message
  Err_InvalidArgument  = 1;
  Err_EmptyFilename    = 2;
  Err_FilenameNotExist = 3;

  // Message during running
  Msg_ConnectOK    = 101;
  Msg_Connected    = 102;
  Msg_NotConnected = 103;
  Msg_ConnectFail  = 104;
  Msg_Disconnected = 105;
  Msg_Quit         = 106;
  Msg_Error        = 107;
  Msg_CmdFail      = 108;
  Msg_HostOk       = 109;
  Msg_InvalidHost  = 110;
  Msg_BeginData    = 111;
  Msg_EndData      = 112;
  Msg_Close        = 113;
  Msg_CmdUnknown   = 114;
  Msg_Ascii        = 115;
  Msg_Binary       = 116;
  Msg_DebugOn      = 117;
  Msg_DebugOff     = 118;
  Msg_VerboseOn    = 119;
  Msg_VerboseOff   = 120;
  Msg_BellOn       = 121;
  Msg_BellOff      = 122;
  Msg_PromptOn     = 123;
  Msg_PromptOff    = 124;
  Msg_GlobbingOn   = 125;
  Msg_GlobbingOff  = 126;
  Msg_HashOn       = 127;
  Msg_HashOff      = 128;
  Msg_LocalDir     = 129;
var
  Client: TClient;
  Command: string;
  CommandFilename: string;
  CommandLineHost: string;
  MustClose: boolean;
  Prompt: string;
  LocalDirectory: string;

{*****************************************************************************}
{ Utilities                                                                   }
{*****************************************************************************}

function AddWindowsBackSlash(AString: string): string;
// Add windows backslash if needed
begin
  if length(AString) > 1 then
    if AString[length(AString)] <> '\' then AString:= AString + '\';
  Result := AString;
end;

function AddUnixSlash(AString: string): string;
// Add Unix slash if needed
begin
  if length(AString) > 1 then
    if AString[length(AString)] <> '/' then AString:= AString + '/';
  Result := AString;
end;

function AdaptHost(Host: string): string;
begin
  if lowercase(copy(Host,1,6)) = 'ftp://' then delete(Host,1,6);
  result:= host;
end;

function GetCommandProcessor: pchar;
// Which command processor is available
var
  Size: longint;
  CommandProcessor: pchar;
begin
  Size:= GetEnvironmentVariable('Comspec',nil,0);
  CommandProcessor:= StrAlloc(Size);
  GetEnvironmentVariable('Comspec',CommandProcessor,Size);
  result:= CommandProcessor;
end;

function ExecuteAndWait(ExeName: string; Parameters: string): boolean;
// Execute program and wait it end's
// Adapted from answer found on Borland newsgroup archive
var
  StartInfo: TStartupInfo;
  ProcessInfo: TProcessInformation;
begin
  // Fill with known state
  fillchar(StartInfo,sizeof(TStartupInfo),#0);
  fillchar(ProcessInfo,sizeof(TProcessInformation),#0);
  // StartInfo set up
  StartInfo.cb:= sizeof(TStartupInfo);
  StartInfo.wShowWindow:= SW_SHOWNORMAL;
  StartInfo.dwFlags:= STARTF_USESHOWWINDOW;
  // Process
  result:= CreateProcess(nil, pchar(ExeName + ' ' + Parameters),
             nil, nil, true,
             NORMAL_PRIORITY_CLASS,
             nil, nil, StartInfo, ProcessInfo);
  // Wait until finished
  if result then
  begin
    WaitForSingleObject(ProcessInfo.hProcess, INFINITE);
    CloseHandle(ProcessInfo.hProcess);
    CloseHandle(ProcessInfo.hThread);
  end;
end;

{*****************************************************************************}
{ Generic display management                                                  }
{*****************************************************************************}

procedure ShowUse;
begin
  writeln;
  writeln('LINEFTP transfers files to and from a computer running an FTP server');
  writeln('Version: ',Version,' (see http://jack.r.free.fr)');
  writeln;
  writeln('This program is free software under GNU Public License.');
  writeln('LINEFTP -L to display the license.');
  writeln;
  writeln('Usage: LINEFTP [-v] [-d] [-i] [-g] [-s:filename] [-w:buffersize] [host]');
  writeln;
  writeln('  -v             Suppresses display of remote server responses.');
  writeln('  -d             Turn on debugging mode.');
  writeln('  -i             Turn off prompting for multiple command.');
  writeln('  -g             Turn off globbing mode (see glob command).');
  writeln('  -s:filename    Specifies a text file containing FTP commands.');
  writeln('                 Commands will automatically run after FTP starts.');
  writeln('  host           Specifies the host name or IP address of the remote');
  writeln('                 host to connect to.');
end;

procedure ShowLicense;
begin
  writeln;
  writeln('LINEFTP license:');
  writeln;
  writeln('This program is free software. You can redistribute it and/or modify');
  writeln('it under the terms of the GNU Public License as published by the');
  writeln('Free Software Foundation, either version 2 of the license, or');
  writeln('(at your option) any laterversion.');
  writeln;
  writeln('This program is distributed in the hope it will be useful,');
  writeln('but WITHOUT ANYWARRANTY, without even the implied warranty of');
  writeln('MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.');
  writeln('See the GNU General Public License for more details.');
  writeln;
  writeln('You should have received a copy of the GNU General Public License');
  writeln('along with this program, if not, write to the');
  writeln('Free Software Foundation, Inc., 59 Temple Place - Suite 330');
  writeln('Boston, MA 02111-1307, USA.');
end;

procedure ShowHelp;
begin
  writeln ('LINEFTP version: ',Version,' (see http://jack.r.free.fr)');
  writeln('Following command are available:');
  writeln;
  writeln('!            delete         literal        prompt          send');
  writeln('?            debug          ls             put             status');
  writeln('append       dir            mdelete        pwd             type');
  writeln('ascii        disconnect                    quit            user');
  writeln('bell         get            mget           quote           verbose');
  writeln('binary       glob           mkdir          recv');
  writeln('bye          hash                          remotehelp');
  writeln('cd           help           mput           rename');
  writeln('close        lcd            open           rmdir');

(*
  writeln('!            delete         literal        prompt          send');
  writeln('?            debug          ls             put             status');
  writeln('append       dir            mdelete        pwd             type');
  writeln('ascii        disconnect     mdir           quit            user');
  writeln('bell         get            mget           quote           verbose');
  writeln('binary       glob           mkdir          recv');
  writeln('bye          hash           mls            remotehelp');
  writeln('cd           help           mput           rename');
  writeln('close        lcd            open           rmdir');
*)
end;

{*****************************************************************************}
{ Program initialisation                                                      }
{*****************************************************************************}

procedure Init;
begin
  CommandFilename:= '';
  CommandLineHost:= '';
  Prompt:= BasePrompt;
  LocalDirectory:= AddWindowsBackSlash(getcurrentdir);
end;

{*****************************************************************************}
{ Class management                                                            }
{*****************************************************************************}

constructor TClient.Create;
begin
  inherited Create;
  // Ftp client
  FtpClient:= TFtpClient.Create(nil);
  FtpClient.ConnectionType:= ftpDirect;
  FtpClient.DisplayFileFlag:= false;
  FtpClient.MultiThreaded:= false;
  FtpClient.Options:= [ftpAcceptLF,ftpNoAutoResumeAt];
  FtpClient.Port:= BasePort;
  FtpClient.ShareMode:= ftpShareExclusive;
  FtpClient.Timeout:= 180;

  FtpClient.OnCommand:= FtpClientCommand;
  FtpClient.OnDisplay:= FtpClientDisplay;
  FtpClient.OnProgress:= FtpClientProgress;
  FtpClient.OnRequestDone:= FtpClientRequestDone;
  FtpClient.OnSessionClosed:= FtpClientSessionClosed;
  FtpClient.OnSessionConnected:= FtpClientSessionConnected;

  // Argument list
  Args:= TStringList.Create;
  // Command processor
  CmdProcessor:= GetCommandProcessor;
  // Temporary file for directory list
  TmpFile:= ExtractFilePath(paramstr(0));
  TmpFile:= TmpFile + 'LineFtpDir.Txt';
  // Status
  Silent:= false;
  Debug:= false;
  Verbose:= true;
  Bell:= false;
  Prompting:= true;
  Globbing:= true;
  Hash:= false;
  // Other internal var
  Anonymous:= false;
  BufferSize:= DefaultBufferSize;
  SendSize:= 0;
  LastCmd:= '';
  MustClose:= false;
end;

destructor TClient.Destroy;
begin
  if FtpClient.Connected then ExecuteCmd(FtpClient.Quit);
  FtpClient.Destroy;
  Args.Free;
  inherited Destroy;
end;

{*****************************************************************************}
{ Display management                                                          }
{*****************************************************************************}

procedure TClient.ShowMessage(MsgNo: integer; Text: string);
begin
  if not Silent then
  begin
    case MsgNo of
      Msg_ConnectOK    : writeln('Connection establish with: ',Text);
      Msg_Connected    : writeln('Connected.',Text);
      Msg_NotConnected : writeln('Not connected. Command unavailable.',Text);
      Msg_ConnectFail  : writeln('Connection failed with: ',Text);
      Msg_Disconnected : writeln('Disconnected.',Text);
      Msg_Quit         : writeln('Disconnected from: ',Text);
      Msg_Error        : writeln('Error: ',Text);
      Msg_CmdFail      : writeln('Command failed: : ',Text);
      Msg_HostOk       : writeln('Host resolved: ',Text);
      Msg_InvalidHost  : writeln('Invalid host: ',Text);
      Msg_BeginData    : writeln('Beginning Data Transfer.',Text);
      Msg_EndData      : writeln('Data Transfer Complete.',Text);
      Msg_Close        : writeln('Distant machine has close connection.',Text);
      Msg_CmdUnknown   : writeln('Unknown ftp command.',Text);
      Msg_Ascii        : writeln('Using ascii mode to transfer files.');
      Msg_Binary       : writeln('Using binary mode to transfer files.');
      Msg_DebugOn      : writeln('Debug mode is ON.');
      Msg_DebugOff     : writeln('Debug mode is OFF.');
      Msg_VerboseOn    : writeln('Verbose mode is ON.');
      Msg_VerboseOff   : writeln('Verbose mode is OFF.');
      Msg_BellOn       : writeln('Bell mode is ON.');
      Msg_BellOff      : writeln('Bell mode is OFF.');
      Msg_PromptOn     : writeln('Prompt mode is ON.');
      Msg_PromptOff    : writeln('Prompt mode is OFF.');
      Msg_GlobbingOn   : writeln('Glob mode is ON.');
      Msg_GlobbingOff  : writeln('Glob mode is OFF.');
      Msg_HashOn       : writeln('Hash mode is ON.');
      Msg_HashOff      : writeln('Hash mode is OFF.');
      Msg_LocalDir     : writeln('Local directory is now: ', Text);
    else
      writeln('Unknown message :',MsgNo);
    end
  end;
end;

procedure TClient.ShowAnswer(Text: string);
var
  MsgType: char;
begin
  if not Silent then
  begin
    MsgType:= Text[1];
    case MsgType of
      '>' : if Debug then writeln(Text);
      '<' : if Verbose then writeln(Text);
    else
      writeln(Text);
    end;
  end;
end;

procedure TClient.ShowError(ErrNo: integer;Text: string);
begin
  if not Silent then
  begin
    case ErrNo of
      Err_InvalidArgument  : writeln('Invalid argument: ',Text);
      Err_EmptyFilename    : writeln('Missing filename for -s option');
      Err_FilenameNotExist : writeln('File "',Text,'" didn''t exist.');
    else
      writeln('Unknown error :',ErrNo);
    end;
  end;
end;

{*****************************************************************************}
{ Argument list management                                                    }
{*****************************************************************************}

procedure TClient.SplitArgs(Text: string);
var
  SplitterPos: longint;
  Splitter: char;
begin
  Args.Clear;
  while length(Text) > 0 do
  begin
    // Remove trailing space
    while ((Text[1] = ' ') and (length(Text) > 0)) do delete(Text,1,1);
    if length(Text) > 0 then
    begin
      // Quoted arguments
      if Text[1] = '"' then
      begin
        delete(Text,1,1);
        Splitter:= '"';
      end
      else Splitter:= ' ';
      // Split
      SplitterPos:= pos(Splitter,Text);
      if SplitterPos = 0 then
      begin
        Args.Add(Text);
        Text:= '';
      end
      else
      begin
        Args.Add(copy(Text,1,SplitterPos-1));
        delete(Text,1,SplitterPos);
      end;
    end;
  end;
  if Args.Count = 0 then Args.Add(' '); // to prevent error
end;

{*****************************************************************************}
{ Command line management                                                     }
{*****************************************************************************}

function TClient.IsValidParameter(Param: string): boolean;
// Check if Param is a valid command line parameter
// and do the corresponding setup
var
  Arg: char;
begin
  if Param[1]= '-' then
  begin
    // command line argument
    Arg:= Param[2];
    delete(Param,1,2);
    case Arg of
      'v': begin
           // don't display remote server answer
           Verbose:= false;
           result:= true;
         end;
      's': begin
           // text file containing ftp command
           delete(Param,1,1); // delete :
           CommandFilename:= Param;
           if CommandFilename = '' then
           begin
             ShowError(Err_EmptyFilename,'');
             result:= false;
           end
           else
           begin
             if not fileexists(CommandFilename) then
             begin
               ShowError(Err_FilenameNotExist,CommandFilename);
               result:= false;
             end
             else result:= true;
           end;
         end;
      'L': begin
             // license
             ShowLicense;
             result:= true;
           end;
      'd': begin
             // debug
             Debug := true;
             result:= true;
           end;
      'i': begin
             // none interative multi-file transfer
             Prompting:= false;
             result:= true;
           end;
// ignored argument corresponding to windows ftp client
      'n': result:= true; // no automatic connection
      'a': result:= true; // use any local interface to do data connection
///////////////////////////////////////////////////////
      'g': begin
             // toggle metacharecter expansion on local filenames
             Globbing:= false;
             result:= true;
           end;
      'w': begin
             // specify buffer size
             delete(Param,1,1); // delete :
             try
               BufferSize:= StrToInt(Param);
             except
               BufferSize:= DefaultBufferSize;
             end;
             result:= true;
           end;
      'A': begin
             // anonymous connection
             Anonymous:= true;
             result:= true;
           end;
    else
      ShowError(Err_InvalidArgument,'-'+Arg);
      result:= false;
    end;
  end
  else
  begin
    // host
    CommandLineHost:= AdaptHost(Param);
    result:= true;
  end;
end;

function TClient.IsValidCommandLine: boolean;
// Check if the command line argument are valid
var
  N: longint;
  Max: longint;
begin
  result:= true;
  Max:= Paramcount;
  N:= 1;
  while ((N <= Max) and result) do
  begin
    result:= IsValidParameter(paramstr(N));
    inc(N);
  end;
end;

{*****************************************************************************}
{ Ftp client management                                                       }
{*****************************************************************************}

function TClient.ExecuteCmd(Cmd: TSyncCmd):boolean;
begin
  if Cmd then
  begin
    if LastCmd='QUIT' then
    begin
      ShowMessage(Msg_Quit,FtpClient.HostName);
      LastCmd:= '';
    end;
    Result:= true;
  end
  else
  begin
    if LastCmd<>'' then ShowMessage(Msg_CmdFail,LastCmd);
    Result:= false;
  end;
  if Bell then sysutils.beep;
  FtpClient.HostFileName:= '';
end;

procedure TClient.FtpClientCommand(Sender: TObject; var Cmd: String);
begin
  LastCmd:= Cmd;
  if (Cmd='NLST') or (Cmd='LIST') then FtpClient.LocalFileName:= TmpFile;
  if copy(Cmd,1,4)= 'PASS' then LastCmd:= 'PASS';
end;

procedure TClient.FtpClientDisplay(Sender: TObject; var Msg: String);
var
  Tmp: string;
begin
  // Replace password by *****
  if copy(Msg,1,6)='> PASS' then Msg:= '> PASS ******';
  if Hash then
  begin
    Tmp:= copy(LastCmd,1,4);
    if (Tmp = 'STOR') and (copy(Msg,1,5)='< 226') then writeln;
  end;
  ShowAnswer(Msg)
end;

procedure TClient.FtpClientProgress(Sender: TObject; Count: Integer;
  var Abort: Boolean);
begin
  if not Silent then
  begin
    if Hash then
    begin
      if (Count >= (SendSize + BufferSize)) then
      begin
        write('#');
        inc(SendSize,BufferSize);
      end;
    end;
  end;
end;

procedure TClient.FtpClientRequestDone(Sender: TObject;
  RqType: TFtpRequest; Error: Word);
begin
  if Error = 0  then
  begin
    case RqType of
      ftpNone            : ;

      ftpOpenAsync,
      ftpConnectAsync    : if not FtpClient.Connected then
                              ShowMessage(Msg_ConnectFail,FtpClient.HostName);
      ftpUserAsync,
      ftpPassAsync,
      ftpPortAsync       : ;

      ftpDirAsync,
      ftpDirectoryAsync  : ;

      ftpLsAsync,
      ftpListAsync       : ;

      ftpCwdAsync,
      ftpCDupAsync       : ;

      ftpPwdAsync        : ShowAnswer(FtpClient.DirResult);

      ftpGetAsync,
      ftpReceiveAsync    : ;

      ftpRestGetAsync,
      ftpRestartGetAsync : ;

      ftpPutAsync,
      ftpTransmitAsync   : ;

      ftpRestPutAsync,
      ftpRestartPutAsync : ;

      ftpSystAsync,
      ftpSystemAsync     : ;

      ftpAppendAsync     : ;

      ftpSizeAsync,
      ftpFileSizeAsync   : ;

      ftpRqAbort,
      ftpQuitAsync       : ;

      ftpMkdAsync,
      ftpMkdirAsync      : ;

      ftpRmdAsync,
      ftpRmdirAsync      : ;

      ftpRenAsync,
      ftpRenameAsync     : ;

      ftpDeleAsync,
      ftpDeleteAsync     : ;

      ftpRenToAsync,
      ftpRenFromAsync    : ;

      ftpQuoteAsync,
      ftpDoQuoteAsync    : ;

      ftpTypeSetAsync,
      ftpRestAsync       : ;
    end;
  end
  else
  begin
    ShowMessage(Msg_Error,
      IntToStr(Error)+' '+FtpClient.ErrorMessage +'(RequestDone)');
  end;
end;

procedure TClient.FtpClientSessionClosed(Sender: TObject;
  Error: Word);
begin
//
end;

procedure TClient.FtpClientSessionConnected(Sender: TObject; Error: Word);
begin
  ShowMessage(Msg_ConnectOk,FtpClient.HostName);
end;

{*****************************************************************************}
{ Command                                                                     }
{*****************************************************************************}

procedure TClient.CmdAppend(Src, Dest: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostFileName:= Dest;
    FtpClient.LocalFileName:= LocalDirectory + Src;
    ExecuteCmd(FtpClient.Append);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdBell;
begin
  Bell:= not Bell;
  if Bell then ShowMessage(Msg_BellOn,'')
          else ShowMessage(Msg_BellOff,'');
end;

procedure TClient.CmdCd(Directory: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostDirName:= Directory;
    ExecuteCmd(FtpClient.Cwd);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdClose;
begin
  if FtpClient.Connected then ExecuteCmd(FtpClient.Quit);
end;

procedure TClient.CmdDelete(Name: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostFileName:= Name;
    ExecuteCmd(FtpClient.Dele);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdDebug;
begin
  Debug:= not Debug;
  if Debug then ShowMessage(Msg_DebugOn,'')
           else ShowMessage(Msg_DebugOff,'');
end;

procedure TClient.CmdDir(Mask: string);
var
  List: TStringList;
  N: longint;
begin
  if FtpClient.Connected then
  begin
    FtpClient.LocalFileName:= TmpFile;
    FtpClient.HostFileName:= Mask;
    if fileexists(TmpFile) then sysutils.DeleteFile(TmpFile);
    if ExecuteCmd(FtpClient.Dir) then
      if fileexists(TmpFile) then
      begin
        List:= TStringList.Create;
        List.LoadFromFile(TmpFile);
        for N:= 0 to (List.Count-1) do writeln(List[N]);
        List.free;
      end;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdGet(Name: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostFileName:= Name;
    FtpClient.LocalFileName:= LocalDirectory + Name;
    ExecuteCmd(FtpClient.Get);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdGlob;
begin
  Globbing:= not Globbing;
  if Globbing then ShowMessage(Msg_GlobbingOn,'')
              else ShowMessage(Msg_GlobbingOff,'');
end;

procedure TClient.CmdHash;
begin
  Hash:= not Hash;
  if Hash then ShowMessage(Msg_HashOn,'')
          else ShowMessage(Msg_HashOff,'');
end;

procedure TClient.CmdHelp(Cmd: string);
begin
  if Cmd = ''           then ShowHelp;
  if Cmd = '!'          then
    writeln('!                       escape to the shell');
  if Cmd = '?'          then
    writeln('?                       display local help');
  if Cmd = 'append'     then
    writeln('append [src] [dest]     append ''src'' to ''dest''');
  if Cmd = 'ascii'      then
   writeln('ascii                   set ascii transfer mode');
  if Cmd = 'bell'       then
   writeln('bell                    beep when command is completed');
  if Cmd = 'binary'     then
    writeln('binary                  set binary transfer mode');
  if Cmd = 'bye'        then
    writeln('bye                     terminate ftp session and exit');
  if Cmd = 'cd'         then
    writeln('cd [dir]                change remote directory');
  if Cmd = 'close'      then
    writeln('close                   terminate ftp session');
  if Cmd = 'delete'     then
    writeln('delete [file]           delete remote ''file''');
  if Cmd = 'debug'      then
    writeln('debug                   toggle debugging mode');
  if Cmd = 'dir'        then
    writeln('dir                     list contents of remote directory');
  if Cmd = 'disconnect' then
    writeln('disconnect              terminate ftp session');
  if Cmd = 'get'        then
    writeln('get [file]              download ''file''');
  if Cmd = 'glob'       then
    writeln('glob                    toggle globbing mode');
  if Cmd = 'hash'       then
    writeln('hash                    toggle printing ''#'' for each buffer transferred');
  if Cmd = 'help'       then
    writeln('help                    display local help');
  if Cmd = 'lcd'        then
    writeln('lcd                     change local directory');
  if Cmd = 'literal'    then
    writeln('literal                 send abitrary ftp command');
  if Cmd = 'ls'         then
    writeln('ls                      nlist contents of remote directory');
  if Cmd = 'mdelete'    then
    writeln('mdelete                 delete multiple files');
  if Cmd = 'mdir'       then
    writeln('mdir                    list contents of multiple remote directory');
  if Cmd = 'mget'       then
    writeln('mget                    download multiple file');
  if Cmd = 'mkdir'      then
    writeln('mkdir                   create a remote directory');
  if Cmd = 'mls'        then
    writeln('mls                     nlist contents of multiple remote directory');
  if Cmd = 'mput'       then
    writeln('mput                    upload multiple file');
  if Cmd = 'open'       then
    writeln('open [host]             connect to remote ftp server');
  if Cmd = 'prompt'     then
    writeln('prompt                  toggle prompt for multiple actions');
  if Cmd = 'put'        then
    writeln('put [file]              upload ''file''');
  if Cmd = 'pwd'        then
    writeln('pwd                     display remote directory name');
  if Cmd = 'quit'       then
    writeln('quit                    terminate ftp session and exit');
  if Cmd = 'quote'      then
    writeln('quote                   send abitrary ftp command');
  if Cmd = 'recv'       then
    writeln('recv [file]             download ''file''');
  if Cmd = 'remotehelp' then
    writeln('remotehelp              display remote server help');
  if Cmd = 'rename'     then
    writeln('rename [old] [new]      rename ''old'' file with new ''name''');
  if Cmd = 'rmdir'      then
    writeln('rmdir [dir]             remove remote directory');
  if Cmd = 'send'       then
    writeln('send [file]             upload ''file''');
  if Cmd = 'status'     then
    writeln('status                  show current status');
  if Cmd = 'type'       then
    writeln('type                    change transfer mode');
  if Cmd = 'user'       then
    writeln('user                    send new user information');
  if Cmd = 'verbose'    then
    writeln('verbose                 toggle verbose mode');
  writeln;
end;

procedure TClient.CmdLcd(Directory: string);
begin
  Directory:= AddWindowsBackSlash(Directory);
  if setcurrentdir(Directory) then
    LocalDirectory:= AddWindowsBackSlash(getcurrentdir);
  ShowMessage(Msg_LocalDir,LocalDirectory);  
end;

procedure TClient.CmdLs;
var
  List: TStringList;
  N: longint;
begin
  if FtpClient.Connected then
  begin
    FtpClient.LocalFileName:= TmpFile;
    FtpClient.HostFileName:= Mask;
    if fileexists(TmpFile) then sysutils.DeleteFile(TmpFile);
    if ExecuteCmd(FtpClient.Ls) then
      if fileexists(TmpFile) then
      begin
        List:= TStringList.Create;
        List.LoadFromFile(TmpFile);
        for N:= 0 to (List.Count-1) do writeln(List[N]);
        List.free;
      end;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdMdelete(Name: string);
var
  SilentMode: boolean;
  List: TStringList;
  N: longint;
  Answer: string;
begin
  if FtpClient.Connected then
  begin
    // Change to silent mode
    SilentMode:= Silent;
    Silent:= true;
    // Get all matching files
    FtpClient.LocalFileName:= TmpFile;
    FtpClient.HostFileName:= Name;
    if fileexists(TmpFile) then sysutils.DeleteFile(TmpFile);
    if ExecuteCmd(FtpClient.Ls) then
    begin
      if fileexists(TmpFile) then
      begin
        // Back to normal mode
        Silent:= SilentMode;
        // Do all found files
        List:= TStringList.Create;
        List.LoadFromFile(TmpFile);
        for N:= 0 to (List.Count-1) do
        begin
          if (List[N] <> '.') and (List[N] <> '..') then
          begin
            if Prompting then
            begin
              write('mdelete ',List[N],' (y/n)?');
              readln(Answer);
            end
            else Answer:= 'y';
            if Answer[1] in ['y','Y','o','O'] then CmdDelete(List[N]);
          end;
        end;
        List.free;
      end;
    end;
    // Back to normal mode
    Silent:= SilentMode;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdMdir;
begin
  writeln('Command not yet implemented.');
end;

procedure TClient.CmdMget(Name: string);
var
  SilentMode: boolean;
  List: TStringList;
  N: longint;
  Answer: string;
begin
  if FtpClient.Connected then
  begin
    // Change to silent mode
    SilentMode:= Silent;
    Silent:= true;
    // Get all matching files
    FtpClient.LocalFileName:= TmpFile;
    FtpClient.HostFileName:= Name;
    if fileexists(TmpFile) then sysutils.DeleteFile(TmpFile);
    if ExecuteCmd(FtpClient.Ls) then
    begin
      if fileexists(TmpFile) then
      begin
        // Back to normal mode
        Silent:= SilentMode;
        // Do all found files
        List:= TStringList.Create;
        List.LoadFromFile(TmpFile);
        for N:= 0 to (List.Count-1) do
        begin
          if (List[N] <> '.') and (List[N] <> '..') then
          begin
            if Prompting then
            begin
              write('mget ',List[N],' (y/n)?');
              readln(Answer);
            end
            else Answer:= 'y';
            if Answer[1] in ['y','Y','o','O'] then CmdGet(List[N]);
          end;
        end;
        List.free;
      end;
    end;
    // Back to normal mode
    Silent:= SilentMode;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdMkdir(Name: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostFileName:= Name;
    ExecuteCmd(FtpClient.Mkd);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdMls;
begin
  writeln('Command not yet implemented.');
end;

procedure TClient.CmdMput(Name: string);
var
  SilentMode: boolean;
  List: TStringList;
  N: longint;
  Answer: string;
begin
  if FtpClient.Connected then
  begin
    // Change to silent mode
    SilentMode:= Silent;
    Silent:= true;
    // Get all matching files
    if fileexists(TmpFile) then sysutils.DeleteFile(TmpFile);
    if ExecuteAndWait(CmdProcessor + ' /c ','dir ' + Name +' /b > ' + TmpFile) then
    begin
      if fileexists(TmpFile) then
      begin
        // Back to normal mode
        Silent:= SilentMode;
        // Do all found files
        List:= TStringList.Create;
        List.LoadFromFile(TmpFile);
        for N:= 0 to (List.Count-1) do
        begin
          if (List[N] <> '.') and (List[N] <> '..') then
          begin
            if Prompting then
            begin
              write('mput ',List[N],' (y/n)?');
              readln(Answer);
            end
            else Answer:= 'y';
            if Answer[1] in ['y','Y','o','O'] then CmdPut(List[N]);
          end;
        end;
        List.free;
      end;
    end;
    // Back to normal mode
    Silent:= SilentMode;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdOpen(Host, Port: string);
begin
  // Close if already open
  if FtpClient.Connected then ExecuteCmd(FtpClient.Quit);
  // Set up
  if Port = '' then Port:= BasePort;
  FtpClient.HostName:= Host;
  FtpClient.Port:= Port;
  FtpClient.LocalFileName:= TmpFile;
  FtpClient.HostFileName:= '';
  FtpClient.Timeout:= 180;
  ExecuteCmd(FtpClient.Open);
  Prompt:= UserPrompt;
  if Anonymous then
  begin
    CmdUser('anonymous','anonymous@anonymous.org');
    Anonymous:= false;
  end;
end;

procedure TClient.CmdPass(Pass: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.Password:= Pass;
    ExecuteCmd(FtpClient.Pass);
    Prompt:= BasePrompt;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdPrompt;
begin
  Prompting:= not Prompting;
  if Prompting then ShowMessage(Msg_PromptOn,'')
               else ShowMessage(Msg_PromptOff,'');
end;

procedure TClient.CmdPut(Name: string);
begin
  if FtpClient.Connected then
  begin
    SendSize:= 0;
    FtpClient.HostFileName:= Name;
    FtpClient.LocalFileName:= LocalDirectory + Name;
    ExecuteCmd(FtpClient.Put);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdPwd;
begin
  if FtpClient.Connected then ExecuteCmd(FtpClient.Pwd)
                         else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdQuit;
begin
  if FtpClient.Connected then ExecuteCmd(FtpClient.Quit);
  MustClose:= true;
end;

procedure TClient.CmdQuote(Cmd: string);
begin
  FtpClient.LocalFileName:= Cmd;
  ExecuteCmd(FtpClient.Quote);
end;

procedure TClient.CmdRename(OldName, NewName: string);
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostFileName:= OldName;
    FtpClient.LocalFileName:= NewName;
    ExecuteCmd(FtpClient.Ren);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdRmdir;
begin
  if FtpClient.Connected then
  begin
    FtpClient.HostFileName:= Name;
    ExecuteCmd(FtpClient.Rmd);
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdStatus;
  function ModeStatus(Mode: boolean): string;
  begin
    if Mode then result:= 'On'
            else result:= 'Off';
  end;

begin
  if not Silent then
  begin
    if FtpClient.Binary then write('Type: binary; ')
                        else write('Type: ascii; ');
    write('Verbose: ',ModeStatus(Verbose),'; ');
    write('Bell: ',ModeStatus(Bell),'; ');
    write('Prompting: ',ModeStatus(Prompting),'; ');
    writeln('Globbing: ',ModeStatus(Globbing));
    write('Debugging: ',ModeStatus(Debug),'; ');
    writeln('Hash mark printing: ',ModeStatus(Hash));
  end;
end;

procedure TClient.CmdTrace;
begin
  writeln('Command not yet implemented.');
end;

procedure TClient.CmdType(Typ: string);
begin
  if FtpClient.Connected then
  begin
    Typ:= lowercase(Typ);
    if Typ = 'ascii' then FtpClient.Binary:= false
    else
    if Typ = 'binary' then FtpClient.Binary:= true
    else Typ:= '';
    ExecuteCmd(FtpClient.TypeSet);
    if FtpClient.Binary then ShowMessage(Msg_Binary,'')
                        else ShowMessage(Msg_Ascii,'');
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdUser(User, Pass: string);
var
  Code: string;
begin
  if FtpClient.Connected then
  begin
    FtpClient.UserName:= User;
    if Pass <> '' then FtpClient.PassWord:= Pass;
    FtpClient.LocalFileName:= TmpFile;
    FtpClient.HostFileName:= '';
    FtpClient.Timeout:= 180;
    if fileexists(TmpFile) then sysutils.DeleteFile(TmpFile);
    if ExecuteCmd(FtpClient.User) then
    begin
      if (Pass <> '') then
      begin
        FtpClient.Timeout:= 5;
        ExecuteCmd(FtpClient.Pass);
        Prompt:= BasePrompt;
      end
      else
      begin
        Code:= copy(FtpClient.LastResponse,1,3);
        if Code = '331' then Prompt:= PassPrompt;
      end;
    end;
  end
  else ShowMessage(Msg_NotConnected,'');
end;

procedure TClient.CmdVerbose;
begin
  Verbose:= not Verbose;
  if Verbose then ShowMessage(Msg_VerboseOn,'')
             else ShowMessage(Msg_VerboseOff,'')
end;

procedure TClient.ProcessCommand(Command: string);
var
  //List: TStringList;
  //N: longint;
  Cmd: string;
begin
  // help
  if Command[1]='?' then
  begin
    delete(Command,1,1);
    Command:= 'help' + Command;
  end;

  // shell command
  if Command[1]='!' then
  begin
    delete(Command,1,1);
    ExecuteAndWait(CmdProcessor + ' /c ',Command);
  end
  else
  // ftp command
  begin
    SplitArgs(Command);
    if Args.Count > 0 then Cmd:= lowercase(Args[0])
                      else Cmd:= '';
    if Cmd = 'append' then
    begin
      if Args.Count > 2 then CmdAppend(Args[1],Args[2])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'ascii' then CmdType('ascii')
    else
    if Cmd = 'bell' then CmdBell
    else
    if Cmd = 'binary' then CmdType('binary')
    else
    if Cmd = 'bye' then CmdQuit
    else
    if Cmd = 'cd' then
    begin
      if Args.Count > 1 then CmdCd(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'close' then CmdClose
    else
    if Cmd = 'delete' then
    begin
      if Args.Count > 1 then CmdDelete(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'debug' then CmdDebug
    else
    if Cmd = 'dir' then
    begin
      if Args.Count > 1 then CmdDir(Args[1])
                        else CmdDir('')
    end
    else
    if Cmd = 'disconnect' then CmdClose
    else
    if Cmd = 'get' then
    begin
      if Args.Count > 1 then CmdGet(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'glob' then CmdGlob
    else
    if Cmd = 'hash' then CmdHash
    else
    if Cmd = 'help' then
    begin
      if Args.Count > 1 then CmdHelp(Args[1])
                        else CmdHelp('');
    end
    else
    if Cmd = 'lcd' then
    begin
      if Args.Count > 1 then CmdLcd(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'literal' then
    begin
      if Args.Count > 1 then CmdQuote(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'ls' then
    begin
      if Args.Count > 1 then CmdLs(Args[1])
                        else CmdLs('')
    end
    else
    if Cmd = 'mdelete' then
    begin
      if Args.Count > 1 then
      begin
        if Globbing then CmdMdelete(Args[1])
                    else CmdDelete(Args[1]);
      end
      else CmdHelp(Cmd);
    end
    else
    if Cmd = 'mdir' then CmdMdir
    else
    if Cmd = 'mget' then
    begin
      if Args.Count > 1 then
      begin
        if Globbing then CmdMget(Args[1])
                    else CmdGet(Args[1]);
      end
      else CmdHelp(Cmd);
    end
    else
    if Cmd = 'mkdir' then
    begin
      if Args.Count > 1 then CmdMkdir(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'mls' then CmdMls
    else
    if Cmd = 'mput' then
    begin
      if Args.Count > 1 then
      begin
        if Globbing then CmdMput(Args[1])
                    else CmdPut(Args[1]);
      end
      else CmdHelp(Cmd);
    end
    else
    if Cmd = 'open' then
    begin
      if Args.Count > 2 then CmdOpen(Args[1],Args[2])
      else
      if Args.Count > 1 then CmdOpen(Args[1],'')
      else
        CmdHelp(Cmd);
    end
    else
    if Cmd = 'pass' then
    begin
      if Args.Count > 1 then CmdPass(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'prompt' then CmdPrompt
    else
    if Cmd = 'put' then
    begin
      if Args.Count > 1 then CmdPut(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'pwd' then CmdPwd
    else
    if Cmd = 'quit' then CmdQuit
    else
    if Cmd = 'quote' then
    begin
      if Args.Count > 1 then CmdQuote(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'recv' then
    begin
      if Args.Count > 1 then CmdGet(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'remotehelp' then CmdQuote('HELP')
    else
    if Cmd = 'rename' then
    begin
      if Args.Count > 2 then CmdRename(Args[1],Args[2])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'rmdir' then
    begin
      if Args.Count > 1 then CmdRmdir(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'send' then
    begin
      if Args.Count > 1 then CmdPut(Args[1])
                        else CmdHelp(Cmd);
    end
    else
    if Cmd = 'status' then CmdStatus
    else
    if Cmd = 'trace' then CmdTrace
    else
    if Cmd = 'type' then
    begin
      if Args.Count > 1 then CmdType(Args[1])
                        else CmdType('');
    end
    else
    if Cmd = 'user' then
    begin
      if Args.Count > 2 then CmdUser(Args[1],Args[2])
      else
      if Args.Count > 1 then CmdUser(Args[1],'')
      else
        CmdHelp(Cmd);
    end
    else
    if Cmd = 'verbose' then CmdVerbose
    else
    ShowMessage(Msg_CmdUnknown,' (' + Cmd + ')');
  end;
end;

{*****************************************************************************}
{ Command file management                                                     }
{*****************************************************************************}

procedure TClient.ProcessCommandFile;
var
  CommandList: TStringList;
  N: longint;
begin
  CommandList:= TStringList.Create;
  CommandList.LoadFromFile(CommandFilename);
  for N:= 0 to (CommandList.Count - 1) do
  begin
    if CommandList[N] <> '' then ProcessCommand(CommandList[N]);
  end;
  CommandList.Free;
end;

{*****************************************************************************}
{ Main program                                                                }
{*****************************************************************************}

begin
  Client:= TClient.Create;
  Init;
  if Client.IsValidCommandLine then
  begin
    if CommandLineHost <> '' then
      Client.ProcessCommand('open ' + CommandLineHost);
    if CommandFilename <> '' then
    begin
      Client.ProcessCommandFile;
      Client.ProcessCommand('quit');
    end
    else
    begin
      while not MustClose do
      begin
        write(Prompt);
        readln(Command);
        if Prompt = UserPrompt then
        begin
          if Command = '' then Command:= '(none)';
          Command:= 'user ' + Command;
        end;
        if Prompt = PassPrompt then
        begin
          if Command = '' then Command:= '(none)';
          Command:= 'pass ' + Command;
        end;
        if Command <> '' then Client.ProcessCommand(Command);
      end;
    end;
  end
  else ShowUse;
  Client.Destroy;
end.
