unit JVDwutil;

/////////////////////////////////////////////////////////////////////////////
//                     Part of JVDesk project                              //
//                     Multiple desktop management                         //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Windows management utilities for that project                           //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

interface

uses
  Windows, Classes, Types, TlHelp32, Forms, SysUtils,
  JclSysInfo;

var
  ProgManHandle: THandle;        // program manager handle
  WindowsDeskHandle: THandle;    // Windows desktop handle
  SysTrayHandle: THandle;        // system tray handle
  MainFormHandle: THandle;       // main form handle
  WindowShifting: integer;       // windows shifting value

{*****************************************************************************}
{ Some general window routine used mainly internally                          }
{*****************************************************************************}

  function GetDeskArea: TRect;
  function GetProgManHandle: THandle;
  function GetWinDeskHandle: THandle;
  function GetTaskbarHandle: THandle;
  function GetStartButtonHandle: THandle;
  function GetSysTrayHandle: THandle;
  function GetReBarHandle: THandle;
  function GetMSTaskBarHandle: THandle;
  function GetMSSystemTabHandle: THandle;

  function GetWindowTitle(AHandle: THandle): string;
  function GetWindowPos(AHandle: THandle): TRect;
  function GetWindowClass(AHandle: THandle): string;
  function GetWindowStatus(AHandle: THandle): byte;
  function GetWindowProcessId(AHandle: THandle): cardinal;
  function GetWindowModule(AHandle: THandle): string;
  function GetWindowParent(AHandle: THandle): THandle;
  function GetWindowMainParent(AHandle: THandle): THandle;

  procedure ShiftAWindow(AHandle: THandle; DeltaX, DeltaY: integer);
  procedure HideAWindowFromTaskBar(AHandle: THandle; Hide: boolean);
  procedure MoveAWindowOut(AHandle: THandle);
  procedure MoveAWindowIn(AHandle: THandle);
  procedure HideAWindow(AHandle: THandle);
  procedure ShowAWindow(AHandle: THandle);

  function IsValideWindow(AHandle: THandle): boolean;
  function IsMouseOnWindow(AHandle: THandle): boolean;
  procedure RecoverLostWindows;
  procedure FlashWindow(AHandle: THandle);

  { from news groups }

  function ForceForegroundWindow(hwnd: THandle): boolean;

implementation

{*****************************************************************************}
{ Windows main elements management routine                                    }
{*****************************************************************************}

function GetDeskArea: TRect;
// Get available area on the windows desk without task bar
var
  DeskRect: TRect;
begin
  SystemParametersInfo(SPI_GETWORKAREA,0,@DeskRect,0);
  Result:= DeskRect;
end;

{------------------------------------------------------------------------------}

function GetProgManHandle: THandle;
// Return the program manager handle
begin
  Result:= FindWindow('ProgMan',nil);
end;

{------------------------------------------------------------------------------}

function GetWinDeskHandle: THandle;
// Return Windows desk handle, in fact it is a ListView
var
  S: string;
begin
  Result:= FindWindow('ProgMan',nil);
  Result:= GetWindow(result,GW_CHILD);
  Result:= GetWindow(result,GW_CHILD);
  SetLength(S,40);
  GetClassName(result,PChar(S),39);
  if PChar(S)<>'SysListView32' then result:= 0;
end;

{------------------------------------------------------------------------------}

function GetTaskbarHandle: THandle;
// Return Windows task bar handle
begin
  Result:= FindWindow('Shell_TrayWnd', nil);
end;

{------------------------------------------------------------------------------}

function GetStartButtonHandle: THandle;
// Return Windows start button handle
begin
  Result:= FindWindowEx(GetTaskbarHandle,0,'Button', nil);
end;

{------------------------------------------------------------------------------}

function GetSysTrayHandle: THandle;
// Return Windows system tray handle
begin
  Result:= FindWindowEx(GetTaskbarHandle,0,'TrayNotifyWnd', nil);
end;

{------------------------------------------------------------------------------}

function GetReBarHandle: THandle;
// Return Windows ressource bar handle
// That is the one which contain taskbar toolbars
// and all open application button
begin
  Result:= FindWindowEx(GetTaskbarHandle,0,'ReBarWindow32', nil);
end;

{------------------------------------------------------------------------------}

function GetMSTaskBarHandle: THandle;
// Return Microsoft taskbar handle
// That is the taskbar part which contain the container of
// all open application button
begin
  Result:= FindWindowEx(GetReBarHandle,0,'ToolbarWindow32', nil);
end;

{------------------------------------------------------------------------------}

function GetMSSystemTabHandle: THandle;
// Return Microsoft systemtab handle
// That is the one which contain all open application button
begin
  Result:= FindWindowEx(GetMSTaskBarHandle,0,'SysTabControl32', nil);
end;

{*****************************************************************************}
{ Window management routine                                                   }
{*****************************************************************************}

function GetWindowTitle(AHandle: THandle): string;
// Return the window title (text of title bar)
var
  St: array [0..256] of char;
begin
  GetWindowText(AHandle, St, SizeOf(St));
  Result:= St;
end;

{------------------------------------------------------------------------------}

function GetWindowPos(AHandle: THandle): TRect;
// Return the window position
var
  Rect: TRect;
begin
  GetWindowRect(AHandle,Rect);
  Result:= Rect;
end;

{------------------------------------------------------------------------------}

function GetWindowClass(AHandle: THandle): string;
// Return the window class
var
  St: array [0..256] of char;
begin
  GetClassName(AHandle, St, SizeOf(St));
  Result:= St;
end;

{------------------------------------------------------------------------------}

function GetWindowStatus(AHandle: THandle): byte;
// return the window status (Normal, maximize, minimize, ...)
var
  WindowPlacement: TWindowPlacement;
begin
  // Init window placement structure
  fillchar(WindowPlacement,sizeof(WindowPlacement),0);
  WindowPlacement.length:= sizeof(WindowPlacement);
  // Get window placement
  GetWindowPlacement(AHandle,@WindowPlacement);
  Result:= WindowPlacement.showCmd;
end;

{------------------------------------------------------------------------------}

function GetWindowProcessId(AHandle: THandle): cardinal;
// Return the process ID attached to that window
var
  ProcessId: cardinal;
begin
  GetWindowThreadProcessId(AHandle,@ProcessId);
  Result:= ProcessId;
end;

{------------------------------------------------------------------------------}

function GetWindowModule(AHandle: THandle): string;
// Return the module which create that window
var
  Snapshot: Integer;
  ProcessId: cardinal;
  ProcessEntry: TProcessEntry32;
  Found: boolean;
begin
  ProcessId:= GetWindowProcessId(AHandle);
  Snapshot:= CreateToolhelp32Snapshot(TH32CS_SNAPPROCESS, 0);
  try
    ProcessEntry.dwSize := SizeOf(ProcessEntry);
    Found:= false;
    if(Process32First(Snapshot,ProcessEntry))then
    begin
      repeat
        Found:= ProcessEntry.th32ProcessID = ProcessID
      until (Found or (not(Process32Next(Snapshot,ProcessEntry))));
    end;
  finally
    CloseHandle(Snapshot);
  end;
  if Found then Result:= ProcessEntry.szExeFile
           else Result:= '';
end;

{------------------------------------------------------------------------------}

function GetWindowParent(AHandle: THandle): THandle;
// Return the handle of its parent window
begin
  Result:= GetWindowLong(AHandle, GWL_HWNDPARENT);
end;

{------------------------------------------------------------------------------}

function GetWindowMainParent(AHandle: THandle): THandle;
// Return the handle of its main window parent
// If already a main window, return the same handle
var
  Found: boolean;
begin
  if IsMainAppWindow(AHandle) then Result:= AHandle
  else
  begin
    repeat
      Result:= GetWindowLong(AHandle, GWL_HWNDPARENT);
      Found:= (IsMainAppWindow(Result)) or (Result = AHandle) or (Result = 0);
      if not Found then AHandle:= Result;
    until Found;
  end;
end;

{------------------------------------------------------------------------------}

procedure ShiftAWindow(AHandle: THandle; DeltaX, DeltaY: integer);
// Shift a window
var
  Rect: TRect;
begin
  Rect:= GetWindowPos(AHandle);
  with Rect do
  SetWindowPos(AHandle,HWND_NOTOPMOST,Left + DeltaX, Top + DeltaY, Right - Left,
    Bottom - Top,SWP_NOSIZE + SWP_NOZORDER + SWP_NOSENDCHANGING);
end;

{------------------------------------------------------------------------------}

procedure HideAWindowFromTaskBar(AHandle: THandle; Hide: boolean);
// Hide or show the specified taskbar button
begin
  if Hide then
    SetWindowPos(AHandle,HWND_NOTOPMOST,0,0,0,0,
      SWP_HIDEWINDOW+SWP_NOSIZE+SWP_NOMOVE+SWP_NOZORDER+SWP_NOSENDCHANGING)
  else
    SetWindowPos(AHandle,HWND_NOTOPMOST,0,0,0,0,
      SWP_SHOWWINDOW+SWP_NOSIZE+SWP_NOMOVE+SWP_NOZORDER+SWP_NOSENDCHANGING);
end;

{------------------------------------------------------------------------------}

procedure MoveAWindowOut(AHandle: THandle);
// Move a window out of the screen area by shifting it to the right
begin
  ShiftAWindow(AHandle, WindowShifting,0);
end;

{------------------------------------------------------------------------------}

procedure MoveAWindowIn(AHandle: THandle);
// Move back in the screen area, a window which have previously been moved out
var
  Rect: TRect;
  Delta: integer;
begin
  Rect:= GetWindowPos(AHandle);
  //Delta:= (Rect.Left div WindowShifting) * WindowShifting;
  Delta:= 0;
  while (Rect.Left - Delta) > Screen.Width do Delta:= Delta + WindowShifting;
  while (Rect.Right - Delta) < 0 do Delta:= Delta - Screen.Width;
  if Delta <> 0 then ShiftAWindow(AHandle, -Delta,0);
end;

{------------------------------------------------------------------------------}

procedure HideAWindow(AHandle: THandle);
// Hide a window and move it out of the screen area by shifting it to the right
var
  Rect: TRect;
begin
  Rect:= GetWindowPos(AHandle);
  with Rect do
  SetWindowPos(AHandle,HWND_NOTOPMOST,Left + WindowShifting, Top, Right - Left,
    Bottom - Top,SWP_HIDEWINDOW + SWP_NOSIZE + SWP_NOZORDER + SWP_NOSENDCHANGING);
end;

{------------------------------------------------------------------------------}

procedure ShowAWindow(AHandle: THandle);
// Show a window previously hide and move out of the screen area
var
  Rect: TRect;
  Delta: integer;
begin
  Rect:= GetWindowPos(AHandle);
  Delta:= 0;
  while (Rect.Left - Delta) > Screen.Width do Delta:= Delta + WindowShifting;
  while (Rect.Right - Delta) < 0 do Delta:= Delta - Screen.Width;
  if Delta <> 0 then
  begin
    with Rect do
      SetWindowPos(AHandle,HWND_NOTOPMOST,Left - Delta, Top, Right - Left,
        Bottom - Top,SWP_SHOWWINDOW + SWP_NOSIZE + SWP_NOZORDER
          + SWP_NOSENDCHANGING);
  end
  else
    if not IsWindowVisible(AHandle) then
      with Rect do
        SetWindowPos(AHandle,HWND_NOTOPMOST,Left, Top, Right - Left,
          Bottom - Top,SWP_SHOWWINDOW + SWP_NOSIZE + SWP_NOZORDER
            + SWP_NOSENDCHANGING);
end;

{------------------------------------------------------------------------------}

function IsValideWindow(AHandle: THandle): boolean;
// Return true if the window is a valide one
var
  Title: string;
  MainParent: THandle;
  IsMain: boolean;
begin
  Result:= false;
  Title:= GetWindowTitle(AHandle);
  MainParent := GetWindowMainParent(AHandle);
  IsMain:=  IsMainAppWindow(AHandle);
  if (AHandle <> MainFormHandle)       // Not our software
    and (MainParent <> MainFormHandle)     // Not a child of our software
      and (IsMain or (MainParent <> 0))    // Main window or child from main
       and (AHandle <> ProgManHandle)  // Not Program Manager
          then Result:= true;
end;

{------------------------------------------------------------------------------}

function IsMouseOnWindow(AHandle: THandle): boolean;
// Return true if the mouse is on the widow with the specified handle
var
  MousePos: TPoint;
  WindowPos: TRect;
begin
  GetCursorPos(MousePos);
  WindowPos:= GetWindowPos(AHandle);
  if (MousePos.X >= WindowPos.Left) and (MousePos.X <= WindowPos.Right)
    and (MousePos.Y >= WindowPos.Top) and (MousePos.Y <= WindowPos.Bottom) then
    Result:= true
  else
    Result:= false;
end;

{------------------------------------------------------------------------------}

procedure RecoverLostWindows;

function EnumWindowsCallback(AHandle: HWND; Param: LPARAM): BOOL; stdcall;
var
  Rect: TRect;
begin
  Rect:= GetWindowPos(AHandle);
  if Rect.Left > Screen.Width then
  begin
    if GetWindowTitle(AHandle) <> '' then ShowAWindow(AHandle);
  end;
  Result := True;
end;

begin
  EnumWindows(@EnumWindowsCallback,0);
end;

{------------------------------------------------------------------------------}

procedure FlashWindow(AHandle: THandle);
var
  N: integer;
begin
  for N:= 1 to 5 do
  begin
    HideAWindow(AHandle);
    sleep(300);
    ShowAWindow(AHandle);
    sleep(300);
  end;
end;

{*****************************************************************************}
{ From news groups                                                            }
{*****************************************************************************}

function ForceForegroundWindow(hwnd: THandle): boolean;
const
  SPI_GETFOREGROUNDLOCKTIMEOUT = $2000;
  SPI_SETFOREGROUNDLOCKTIMEOUT = $2001;
var
  ForegroundThreadID: DWORD;
  ThisThreadID : DWORD;
  timeout : DWORD;


begin
  if IsIconic(hwnd) then ShowWindow(hwnd, SW_RESTORE);
  if GetForegroundWindow = hwnd then Result := true
  else begin

// Windows 98/2000 doesn't want to foreground a window when some other
// window has keyboard focus

  if ((Win32Platform = VER_PLATFORM_WIN32_NT) and (Win32MajorVersion >4))
or
  ((Win32Platform = VER_PLATFORM_WIN32_WINDOWS) and
  ((Win32MajorVersion >4) or ((Win32MajorVersion = 4) and
  (Win32MinorVersion >0)))) then
   begin

// Code from Karl E. Peterson, www.mvps.org/vb/sample.htm
// Converted to Delphi by Ray Lischner
// Published in The Delphi Magazine 55, page 16

    Result := false;
    ForegroundThreadID :=
    GetWindowThreadProcessID(GetForegroundWindow,nil);
    ThisThreadID := GetWindowThreadPRocessId(hwnd,nil);
    if AttachThreadInput(ThisThreadID, ForegroundThreadID, true) then
    begin
    BringWindowToTop(hwnd); // IE 5.5 related hack
    SetForegroundWindow(hwnd);
    AttachThreadInput(ThisThreadID, ForegroundThreadID, false);
    Result := (GetForegroundWindow = hwnd);
    end;
    if not Result then begin

// Code by Daniel P. Stasinski

     SystemParametersInfo(SPI_GETFOREGROUNDLOCKTIMEOUT, 0, @timeout, 0);
     SystemParametersInfo(SPI_SETFOREGROUNDLOCKTIMEOUT, 0, TObject(0),
     SPIF_SENDCHANGE);
     BringWindowToTop(hwnd); // IE 5.5 related hack
     SetForegroundWindow(hWnd);
     SystemParametersInfo(SPI_SETFOREGROUNDLOCKTIMEOUT, 0,
     TObject(timeout), SPIF_SENDCHANGE);
   end;
  end
  else begin
    BringWindowToTop(hwnd); // IE 5.5 related hack
  SetForegroundWindow(hwnd);
  end;

  Result := (GetForegroundWindow = hwnd);
  end;
end; { ForceForegroundWindow }

{*****************************************************************************}
{                                                                             }
{*****************************************************************************}

begin
  // Store Program Manager handle
  ProgManHandle:= GetProgManHandle;
  // Store windows desk handle
  WindowsDeskHandle:= GetWinDeskHandle;
  // Store system tray handle
  SysTrayHandle:= GetSysTrayHandle;
  // Default windows shifting value
  WindowShifting:= 10000;
end.
