unit JVDwmng;

/////////////////////////////////////////////////////////////////////////////
//                     Part of JVDesk project                              //
//                     Multiple desktop management                         //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Windows management for that project                                     //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

// Desktop number is 0 based

interface

uses
  Windows, Classes, Contnrs, SysUtils, JclSysInfo,
  JVDwutil, JVDutil;

type

  TWindow = class(TObject)   // information regarding a window
  private
    FHandle: THandle;        // windows handle
    FMainParentHwd: THandle; // parent windows handle
    FMain: boolean;          // is a main window
    FStatus: byte;           // windows status (minimize, maximize, ...)
    FDesktop: byte;          // on which desktop number it is
    FActive: boolean;        // is it the active window
    FSticky: boolean;        // is it a sticky window
    procedure SetHandle(AHandle: THandle);
    procedure SetMainParentHwd(AHandle: THandle);
    procedure SetMain(IsMain: boolean);
    procedure SetStatus(AStatus: byte);
    procedure SetDesktop(ADesktop: byte);
    procedure SetActive(IsActive: boolean);
    procedure SetSticky(IsSticky: boolean);
  public
    Constructor Create;
    Destructor Destroy; Override;
  published
    property Handle: THandle read FHandle write SetHandle;
    property MainParentHwd: THandle read FMainParentHwd write SetMainParentHwd;
    property Main: boolean read FMain write SetMain;
    property Status: byte read FStatus write SetStatus;
    property Desktop: byte read FDesktop write SetDesktop;
    property Active: boolean read FActive write SetActive;
    property Sticky: boolean read FSticky write SetSticky;
  end;

  TWindowsList = class(TObjectList)
  private
    function GetItem(Index: integer): TWindow;
    procedure SetItem(Index: integer; const Value: TWindow);
  public
    property Items[Index: integer]: TWindow read GetItem write SetItem;
    function Add(AWindow: TWindow): Integer;
    function Remove(AWindow: TWindow): Integer;
    procedure Insert(Index: Integer; AWindow: TWindow);
    procedure Sort;
    procedure RemoveFromDesktop(ADesktopNumber: byte);
  end;

  TWindowsManager = class
  private
    { Dclarations prives }
    FMainWindowsList: TWindowsList;      // list of all managed main windows
    FOtherWindowsList: TWindowsList;     // list of all managed other windows
    FMainFormHandle: THandle;            // main form handle
    FActiveHandle: THandle;              // current desktop active window handle
    FMaxDesktop: byte;                   // maximum number of desktop (preset to 12)
    FCurrDesktop: byte;                  // current desktop (0 based)
    FHideFromTaskBar: boolean;           // hide from task bar window not on
                                         // current desktop
    procedure SetMainFormHandle(AHandle: THandle);
    procedure SetActiveHandle(AHandle: THandle);
    procedure SetMaxDesktop(ADesktopNumber: byte);
    procedure SetCurrDesktop(ADesktopNumber: byte);
    procedure SetHideFromTaskBar(Hide: boolean);
    function GetMainWindowCount: integer;
    function GetOtherWindowCount: integer;
    function GetMainWindowIndex(AHandle: THandle): integer;
    function GetWindowByHandle(AHandle: THandle; var AWindow: TWindow): boolean;
    procedure AddWindow(AWindow: TWindow);
    procedure UpdateWindowsList;
    procedure UpdateTaskBar;
    procedure MakeWindowVisible(AHandle: THandle; Visible: boolean);
    procedure ForceShowAllWindows;
    procedure HideWindowsOnDesktop;
    procedure ShowWindowsOnDesktop;
    procedure RemoveFromDesktop(ADesktopNumber: byte);
  public
    { Dclarations publiques }
    Constructor Create;
    Destructor Destroy; Override;
    procedure UpdateActiveHandle;
    procedure ChangeActiveHandle(AHandle: THandle);
    procedure RefreshWindowList;
    procedure ShowActiveWindow;
    procedure MoveToDesktop(ADesktopNumber: byte);
    function GetMainWindow(Index: integer): TWindow;
    function GetOtherWindow(Index: integer): TWindow;
    procedure BringToDesk(AHandle: THandle; ADesktopNumber: byte);
    procedure BringHigherDeskTo(ADesktopNumber: byte);
    procedure ToggleStickyWindow;
    procedure LoadLocation(AFilename: TFilename);
  published
    property ActiveHandle: THandle read FActiveHandle;
    property MaxDesktop: byte read FMaxDesktop write SetMaxDesktop;
    property CurrDesktop: byte read FCurrDesktop write SetCurrDesktop;
    property HideFromTaskBar: boolean read FHideFromTaskBar
      write setHideFromTaskBar;
    property MainWindowCount: integer read GetMainWindowCount;
    property OtherWindowCount: integer read GetOtherWindowCount;
  end;

implementation

{*****************************************************************************}
{ TWindow                                                                     }
{*****************************************************************************}

constructor TWindow.Create;
begin
  inherited;
  SetHandle(0);
  SetMainParentHwd(0);
  SetMain(false);
  SetStatus(0);
  SetDesktop(0);
  SetActive(false);
  SetSticky(false);
end;

{------------------------------------------------------------------------------}

destructor TWindow.Destroy;
begin

  inherited;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetActive(IsActive: boolean);
begin
  FActive:= IsActive;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetSticky(IsSticky: boolean);
begin
  FSticky:= IsSticky;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetDesktop(ADesktop: byte);
begin
  FDesktop:= ADesktop;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetHandle(AHandle: THandle);
begin
  FHandle:= AHandle;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetMain(IsMain: boolean);
begin
  FMain:= IsMain;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetMainParentHwd(AHandle: THandle);
begin
  FMainParentHwd:= AHandle;
end;

{------------------------------------------------------------------------------}

procedure TWindow.SetStatus(AStatus: byte);
begin
  FStatus:= AStatus;
end;

{*****************************************************************************}
{ TWindowList                                                                 }
{*****************************************************************************}

function CompareDesktop(Item1, Item2: Pointer): Integer;
// Compare destop number, main parent handle, handle
var
  Window1: TWindow;
  Window2: TWindow;
  Value1: byte;
  Value2: byte;
begin
  Window1:= TWindow(Item1);
  Window2:= TWindow(Item2);
  Value1:= Window1.Desktop;
  Value2:= Window2.Desktop;
  if Value1 = Value2 then
  begin
    // Same desktop
    Value1:= Window1.MainParentHwd;
    Value2:= Window2.MainParentHwd;
    if Value1 = Value2 then
    begin
      // Same desktop, same main parent
      Value1:= Window1.Handle;
      Value2:= Window2.Handle;
      if Value1 = Value2 then Result:= 0
      else
        if Value1 < Value2 then Result:= -1
                           else Result:= 1;
    end
    else
    begin
      // Same desktop but main parent differ
      if Value1 < Value2 then Result:= -1
                         else Result:= 1;
    end;
  end
  else
  begin
    if Value1 < Value2 then Result:= -1
                       else Result:= 1;
  end;
end;

{------------------------------------------------------------------------------}

function TWindowsList.GetItem(Index: integer): TWindow;
begin
  Result:= TWindow(inherited Items[Index]);
end;

{------------------------------------------------------------------------------}

procedure TWindowsList.SetItem(Index: integer; const Value: TWindow);
begin
  inherited Items[Index]:= Value;
end;

{------------------------------------------------------------------------------}

function TWindowsList.Add(AWindow: TWindow): Integer;
begin
  Result:= inherited Add(AWindow);
end;

{------------------------------------------------------------------------------}

function TWindowsList.Remove(AWindow: TWindow): Integer;
begin
  Result:= inherited Remove(AWindow);
end;

{------------------------------------------------------------------------------}

procedure TWindowsList.Insert(Index: Integer; AWindow: TWindow);
begin
  inherited Insert(Index,AWindow);
end;

{------------------------------------------------------------------------------}

procedure TWindowsList.Sort;
begin
  inherited Sort(@CompareDesktop);
end;

{------------------------------------------------------------------------------}

procedure TWindowsList.RemoveFromDesktop(ADesktopNumber: byte);
var
  Index: integer;
  AWindow: TWindow;
begin
  for Index:= Count-1 downto 0 do
  begin
    AWindow:= Items[Index];
    if (AWindow.Desktop = ADesktopNumber)
     and (not AWindow.Sticky) then Delete(Index);
  end;
end;

{*****************************************************************************}
{ TWindowsManager                                                             }
{*****************************************************************************}

procedure TWindowsManager.SetMainFormHandle(AHandle: THandle);
begin
  FMainFormHandle:= AHandle;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.SetActiveHandle(AHandle: THandle);
begin
  if (AHandle <> FActiveHandle)
   then
   begin
     FActiveHandle:= AHandle;
   end;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.SetMaxDesktop(ADesktopNumber: byte);
begin
  FMaxDesktop:= ADesktopNumber;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.SetCurrDesktop(ADesktopNumber: byte);
begin
  if ADesktopNumber > FMaxDesktop then ADesktopNumber:= FMaxDesktop;
  FCurrDesktop:= ADesktopNumber;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.SetHideFromTaskBar(Hide: boolean);
begin
  FHideFromTaskBar:= Hide;
  UpdateTaskBar;
end;

{------------------------------------------------------------------------------}

function TWindowsManager.GetMainWindowCount: integer;
begin
  Result:= FMainWindowsList.Count;
end;

{------------------------------------------------------------------------------}

function TWindowsManager.GetOtherWindowCount: integer;
begin
  Result:= FOtherWindowsList.Count;
end;

{------------------------------------------------------------------------------}

function TWindowsManager.GetMainWindowIndex(AHandle: THandle): integer;
var
  Index: integer;
  AWindow: TWindow;
  Found: boolean;
begin
  Found:= false;
  Index:= 0;
  while (not Found) and (Index < FMainWindowsList.Count) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    Found:= AWindow.Handle = AHandle;
    inc(Index);
  end;
  if Found then Result:= Index-1
           else Result:= -1;
end;

{------------------------------------------------------------------------------}

function TWindowsManager.GetWindowByHandle(AHandle: THandle;
  var AWindow: TWindow): boolean;
var
  Index: integer;
  Found: boolean;
begin
  Found:= false;
  if IsMainAppWindow(AHandle) then
  begin
    Index:= 0;
    while (not Found) and (Index < FMainWindowsList.Count) do
    begin
      Found:= FMainWindowsList.Items[Index].Handle = AHandle;
      if Found then AWindow:= FMainWindowsList.Items[Index];
      inc(Index);
    end;
  end
  else
  begin
    Index:= 0;
    while (not Found) and (Index < FOtherWindowsList.Count) do
    begin
      Found:= FOtherWindowsList.Items[Index].Handle = AHandle;
      inc(Index);
    end;
  end;
  Result:= Found;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.AddWindow(AWindow: TWindow);
begin
  if AWindow.Main then FMainWindowsList.Add(AWindow)
                  else FOtherWindowsList.Add(AWindow);
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.UpdateWindowsList;

function EnumWindowsCallback(AHandle: HWND; Param: LPARAM): BOOL; stdcall;
var
  AWindow: TWindow;
begin
  if IsWindowVisible(AHandle)
    and IsValideWindow(AHandle) then
    begin
      if TWindowsManager(Param).GetWindowByHandle(AHandle,AWindow) then
      begin
        // Existing one so update if same desktop and sticky
        if (AWindow.Desktop = TWindowsManager(Param).CurrDesktop)
          and AWindow.Sticky then
            AWindow.Active:= AHandle = TWindowsManager(Param).ActiveHandle;
      end
      else
      begin
        // New one so add it
        AWindow:= TWindow.Create;
        AWindow.Handle:= AHandle;
        AWindow.MainParentHwd:= GetWindowMainParent(AHandle);
        AWindow.Main:= IsMainAppWindow(AHandle);
        AWindow.Desktop:= TWindowsManager(Param).CurrDesktop;
        AWindow.Status:= GetWindowStatus(AHandle);
        AWindow.Active:= AHandle = TWindowsManager(Param).ActiveHandle;
        // Store information
        with TWindowsManager(Param) do AddWindow(AWindow);
      end;
    end;
  Result := True;
end;

begin
  UpdateActiveHandle;
  EnumWindows(@EnumWindowsCallback,LongInt(Self));
  FMainWindowsList.Sort;
  FOtherWindowsList.Sort;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.UpdateTaskBar;
var
  Index: integer;
  AWindow: TWindow;
begin
  for Index:= 0 to (FOtherWindowsList.Count-1) do
  begin
    AWindow:= FOtherWindowsList.Items[Index];
    if AWindow.Desktop <> CurrDesktop then
      HideAWindowFromTaskBar(AWindow.Handle,FHideFromTaskBar);
  end;
  for Index:= 0 to (FMainWindowsList.Count-1) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    if AWindow.Desktop <> CurrDesktop then
      HideAWindowFromTaskBar(AWindow.Handle,FHideFromTaskBar);
  end;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.MakeWindowVisible(AHandle: THandle;
  Visible: boolean);
begin
  if Visible then
  begin
    if FHideFromTaskBar then ShowAWindow(AHandle)
                        else MoveAWindowIn(AHandle);
  end
  else
  begin
    if FHideFromTaskBar then HideAWindow(AHandle)
                        else MoveAWindowOut(AHandle);
  end;

end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.ForceShowAllWindows;
var
  Index: integer;
  AWindow: TWindow;
begin
  UpdateActiveHandle;
  for Index:= 0 to (FOtherWindowsList.Count-1) do
  begin
    AWindow:= FOtherWindowsList.Items[Index];
    ShowAWindow(AWindow.Handle);
  end;
  for Index:= 0 to (FMainWindowsList.Count-1) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    ShowAWindow(AWindow.Handle);
  end;
  ShowActiveWindow;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.HideWindowsOnDesktop;
var
  Index: integer;
  AWindow: TWindow;
begin
  for Index:= 0 to (FOtherWindowsList.Count-1) do
  begin
    AWindow:= FOtherWindowsList.Items[Index];
    if (AWindow.Desktop = CurrDesktop)
      and (not AWindow.Sticky) then
        MakeWindowVisible(AWindow.Handle, false);
  end;
  for Index:= 0 to (FMainWindowsList.Count-1) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    if (AWindow.Desktop = CurrDesktop)
      and (not AWindow.Sticky) then
        MakeWindowVisible(AWindow.Handle, false);
  end;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.ShowWindowsOnDesktop;
var
  Index: integer;
  AWindow: TWindow;
begin
  FActiveHandle:= WindowsDeskHandle;  // reset active handle to get the proper one
  for Index:= 0 to (FOtherWindowsList.Count-1) do
  begin
    AWindow:= FOtherWindowsList.Items[Index];
    if AWindow.Desktop = CurrDesktop then
    begin
      MakeWindowVisible(AWindow.Handle,true);
      if AWindow.Active then SetActiveHandle(AWindow.Handle);
    end;
  end;

  for Index:= 0 to (FMainWindowsList.Count-1) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    if AWindow.Desktop = CurrDesktop then
    begin
      MakeWindowVisible(AWindow.Handle,true);
      if AWindow.Active then SetActiveHandle(AWindow.Handle);
    end;
  end;
  ShowActiveWindow;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.RemoveFromDesktop(ADesktopNumber: byte);
begin
  FMainWindowsList.RemoveFromDesktop(ADesktopNumber);
  FOtherWindowsList.RemoveFromDesktop(ADesktopNumber);
end;

{------------------------------------------------------------------------------}

constructor TWindowsManager.Create;
begin
  inherited;
  FMainWindowsList:= TWindowsList.Create;
  FOtherWindowsList:= TWindowsList.Create;
  // Default values
  SetMainFormHandle(0);
  SetMaxDesktop(12);
  SetCurrDesktop(0);
  FActiveHandle:= WindowsDeskHandle;
  FHideFromTaskBar:= true;
  // Get all currently visible windows
  UpdateWindowsList;
end;

{------------------------------------------------------------------------------}

destructor TWindowsManager.Destroy;
begin
  ForceShowAllWindows;
  FOtherWindowsList.Free;
  FMainWindowsList.Free;
  inherited;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.UpdateActiveHandle;
var
  AHandle: THandle;
begin
  AHandle:= GetForegroundWindow;
  if IsValideWindow(AHandle) then SetActiveHandle(AHandle);
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.ChangeActiveHandle(AHandle: THandle);
begin
  if IsValideWindow(AHandle) then SetActiveHandle(AHandle);
  ShowActiveWindow;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.RefreshWindowList;
begin
  RemoveFromDesktop(CurrDesktop);
  UpdateWindowsList;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.ShowActiveWindow;
begin
  ForceForegroundWindow(FActiveHandle);
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.MoveToDesktop(ADesktopNumber: byte);
begin
  UpdateActiveHandle;
  RefreshWindowList;
  HideWindowsOnDesktop;
  CurrDesktop:= ADesktopNumber;
  ShowWindowsOnDesktop;
end;

{------------------------------------------------------------------------------}

function TWindowsManager.GetMainWindow(Index: integer): TWindow;
begin
  Result:= FMainWindowsList.Items[Index];
end;

{------------------------------------------------------------------------------}

function TWindowsManager.GetOtherWindow(Index: integer): TWindow;
begin
  Result:= FOtherWindowsList.Items[Index];
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.BringToDesk(AHandle: THandle; ADesktopNumber: byte);
// Bring the corresponding window and its child to the specified desktop
var
  Index: integer;
  AWindow: TWindow;
begin
  // Refresh list to prevent trying moving an unknown window
  if ADesktopNumber <> CurrDesktop then RefreshWindowList;
  Index:= GetMainWindowIndex(AHandle);
  if Index > -1 then
  begin
    // Bring main window
    AWindow:= GetMainWindow(Index);
    AWindow.Desktop:= ADesktopNumber;
    ADesktopNumber:= AWindow.Desktop; // get the corrected number
    if ADesktopNumber = CurrDesktop then
    begin
      SetActiveHandle(AWindow.Handle);
      MakeWindowVisible(AWindow.Handle,true);
    end
    else
    begin
      MakeWindowVisible(AWindow.Handle,false);
    end;
    FMainWindowsList.Sort;
    // Bring child window
    for Index:= 0 to (FOtherWindowsList.Count-1) do
    begin
      AWindow:= FOtherWindowsList.Items[Index];
      if AWindow.MainParentHwd = AHandle then
      begin
        AWindow.Desktop:= ADesktopNumber;
        if ADesktopNumber = CurrDesktop then
          MakeWindowVisible(AWindow.Handle,true)
        else
          MakeWindowVisible(AWindow.Handle,false);
      end;
    end;
    FOtherWindowsList.Sort;
    if ADesktopNumber = CurrDesktop then
      ShowActiveWindow
    else
      FActiveHandle:= WindowsDeskHandle;
  end;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.BringHigherDeskTo(ADesktopNumber: byte);
// Bring all windows from desk higher than ADeskNumber to that one
var
  Index: integer;
  AWindow: TWindow;
begin
  // Bring child window
  for Index:= 0 to (FOtherWindowsList.Count-1) do
  begin
    AWindow:= FOtherWindowsList.Items[Index];
    if AWindow.Desktop > ADesktopNumber then
    begin
      AWindow.Desktop:= ADesktopNumber;
      if ADesktopNumber = CurrDesktop then
        MakeWindowVisible(AWindow.Handle,true)
      else
        MakeWindowVisible(AWindow.Handle,false);
    end;
  end;
  FOtherWindowsList.Sort;
  // Bring main window
  for Index:= 0 to (FMainWindowsList.Count-1) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    if AWindow.Desktop > ADesktopNumber then
    begin
      AWindow.Desktop:= ADesktopNumber;
      AWindow.Active:= false;
      if ADesktopNumber = CurrDesktop then
        MakeWindowVisible(AWindow.Handle,true)
      else
        MakeWindowVisible(AWindow.Handle,false);
    end;
  end;
  FMainWindowsList.Sort;
  if ADesktopNumber = CurrDesktop then ShowActiveWindow;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.ToggleStickyWindow;
var
  Index: integer;
  AWindow: TWindow;
begin
  // Child windows
  for Index:= 0 to (OtherWindowCount - 1) do
  begin
    AWindow:= FOtherWindowsList.Items[Index];
    if AWindow.Handle = FActiveHandle then
    begin
      AWindow.Sticky:= not AWindow.Sticky;
      if not AWindow.Sticky then AWindow.Desktop:= CurrDesktop;
    end;
  end;
  // Main windows
  for Index:= 0 to (MainWindowCount - 1) do
  begin
    AWindow:= FMainWindowsList.Items[Index];
    if AWindow.Handle = FActiveHandle then
    begin
      AWindow.Sticky:= not AWindow.Sticky;
      if not AWindow.Sticky then AWindow.Desktop:= CurrDesktop;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TWindowsManager.LoadLocation(AFilename: TFilename);
// Load the application name and desk number which must be restored
var
  AStringList: TStringList;
  AnExeName: string;
  ADeskNumber: string;
  AStickyStatus: string;
  N: integer;
  AWindow: TWindow;
  AHandle: THandle;
begin
  if FileExists(AFilename) then
  begin
    // Process file
    AStringList:= TStringList.Create;
    AStringList.LoadFromFile(AFilename);
    for N:= 1 to AStringList.Count do
    begin
      ExtractPrgString(AStringList.Strings[N-1],AnExeName,ADeskNumber,AStickyStatus);
      // Start application
      AHandle:= ExecuteFile(MainFormHandle,AnExeName,'','',SW_SHOWDEFAULT);
      // Get window information
      AWindow:= TWindow.Create;
      AWindow.Handle:= AHandle;
      AWindow.MainParentHwd:= GetWindowMainParent(AHandle);
      AWindow.Main:= IsMainAppWindow(AHandle);
      AWindow.Desktop:= StrToInt(ADeskNumber)-1;
      AWindow.Status:= GetWindowStatus(AHandle);
      AWindow.Sticky:= AStickyStatus = '1';
      AWindow.Active:= false;
      // Store information
      AddWindow(AWindow);
      // Hide it if not on that desktop
      if not AWindow.Sticky then
      begin
        if AWindow.Desktop <> CurrDesktop then
          MakeWindowVisible(AWindow.Handle,false);
      end;
    end;
    AStringList.Free;
    // Bring back the active form on that desk
    ShowActiveWindow;
  end;
end;

end.
