unit JVDdata;

/////////////////////////////////////////////////////////////////////////////
//                     Part of JVDesk project                              //
//                     Multiple desktop management                         //
//  2003  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Data management for that project                                        //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 2 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, sinon, veuillez crire  la Free Software Foundation, Inc., }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 2 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, write to the Free Software Foundation, Inc.,   }
{ 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.                  }
{***************************************************************************}

interface

uses
  Windows, Classes, Types, Menus, SysUtils, IniFiles, Graphics, Forms,
  JVDwmng;

const
  SoftVersion = '0.7.2';    // software version number

  MaxDesk = 12;             // maximum number of desktop
                            // cannot be more than 255 as we use byte in code
                            // desk number start from 0

  General   = 'General';    // general section in configuration file
  HotKey    = 'Hotkey';     // hotkey section in configuration file
  Pager     = 'Pager';      // pager section in configuration file
  Option    = 'Option';     // option section in configuration file
  Switching = 'Switching';  // desk border switching section in config. file

  ToggleStickyHKId  = 995;  // handle for toggling sticky status
  PrevDeskHKId      = 996;  // handle for previous desk hot key
  NextDeskHKId      = 997;  // handle for next desk hot key
  DeskUpHKId        = 998;  // handle for desk up hot key
  DeskDownHKId      = 999;  // handle for desk down hot key

  GotoAppliMnuIndex   = 0;  // popupmenu item index for goto application
  BringHereMnuIndex   = 1;  // popupmenu item index for bring application here
  SendToMnuIndex      = 2;  // popupmenu item index for send application to
  WindowsListMnuIndex = 4;  // popupmenu item index for windows list
  UnmanagedMnuIndex   = 5;  // popupmenu item index for unmanaged windows
  LanguageMnuIndex    = 7;  // popupmenu item index for language
  ConfigMnuIndex      = 8;  // popupmenu item index for configuration
  HelpMnuIndex        = 9;  // popupmenu item index for help
  AboutMnuIndex       = 10; // popupmenu item index for about
  ExitMnuIndex        = 12; // popupmenu item index for exit

type
  TLaunchedPrg = array [0..2] of string;

  TCfgHotKey = record           // store hotkey configuration
    PrevDeskHotKey: TShortCut;                    // hotkey to go to previous
    NextDeskHotKey: TShortCut;                    // hotkey to go to next
    DeskUpHotKey: TShortCut;                      // hotkey to go one line up
    DeskDownHotKey: TShortCut;                    // hotkey to go one line down
    DeskName: array [0..MaxDesk] of string;       // desktop name
    DeskHotKey: array [0..MaxDesk] of TShortCut;  // hotkey for each desktop
    ToggleStickyHotKey: TShortCut;                // hotkey for toggling sticky
  end;

  TConfig = class           // store and manage software configuration
  private
    FLanguage: string;          // currently used language
    FNbHDesk: byte;             // number of horizontal desk
    FNbVDesk: byte;             // number of vertical desk
    FPagerDeskWidth: byte;      // width of a desk on the pager
    FPagerDeskHeight: byte;     // height of a desk on the pager
    FRememberPos: boolean;      // remember pager position
    FHidePager: boolean;        // hide pager
    FLastDesk: byte;            // last available desk H x V - 1 (0 based)
    FCycling: byte;             // cycling between desk
                                // (0: none, 1: line or column, 2: full)
    FHotKeys: TCfgHotKey;       // hot key configuration
    FRestoreAppli: boolean;     // restore application on each desk as saved
    FAskRestore: boolean;       // ask before retoring application on each desk
                                // (laptop not connected to network)
    FHideApplication: boolean;  // hide none visible application from task bar
    FActiveDeskColor: TColor;   // color used to display active desk on pager
    FInactiveDeskColor: TColor; // color used to display inactive desk on pager
    FIconOnPager: boolean;      // display of application icon on pager desk
    FPagerRefresh: boolean;     // constantly update pager display
    FBorderSwitching: boolean;  // allow desk border switching
    FPagerSwitching: boolean;   // allow switching from pager
    procedure SetNbHDesk(ANumber: byte);
    procedure SetNbVDesk(ANumber: byte);
    procedure SetCycling(ANumber: byte);
    procedure InitHotKey;
    procedure ConvertShortCut(AShortCut: TShortCut;
      var AModifier, AVirtualKey: word);
    function AddHotKey(AHandle: THandle; AnId: integer;
      AShortCut: TShortCut): boolean;
    procedure RemoveHotKey(AHandle: THandle; AnId: integer);
    function AddAllHotKeys: boolean;
    procedure RemoveAllHotKeys;
    procedure DefaultCfg;
  public
    Constructor Create;
    Destructor Destroy; Override;
    function LoadFromFile(AFilename: TFilename): boolean;
    function SaveToFile(AFilename: TFilename): boolean;
    function AdaptDeskNumber(ADeskNumber, CurrDesk: integer): byte;
    procedure LoadPagerPos(AFilename: TFilename; var X,Y: integer);
    procedure SavePagerPos(AFilename: TFilename; X,Y: integer);
    procedure SaveLanguage(AFilename: TFilename);
  published
    property Language: string read FLanguage write FLanguage;
    property NbHDesk: byte read FNbHDesk write SetNbHdesk;
    property NbVDesk: byte read FNbVDesk write SetNbVDesk;
    property PagerDeskWidth: byte read FPagerDeskWidth write FPagerDeskWidth;
    property PagerDeskHeight: byte read FPagerDeskHeight write FPagerDeskHeight;
    property RememberPos: boolean read FRememberPos write FRememberPos;
    property HidePager: boolean read FHidePager write FHidePager;
    property LastDesk: byte read FLastDesk write FLastDesk;
    property Cycling: byte read FCycling write SetCycling;
    property HotKeys: TCfgHotKey read FHotKeys write FHotKeys;
    property RestoreAppli: boolean read FRestoreAppli write FRestoreAppli;
    property AskRestore: boolean read FAskRestore write FAskRestore;
    property HideApplication: boolean read FHideApplication write FHideApplication;
    property ActiveDeskColor: TColor read FActiveDeskColor
      write FActiveDeskColor;
    property InactiveDeskColor: TColor read FInactiveDeskColor
      write FInactiveDeskColor;
    property IconOnPager: boolean read FIconOnPager write FIconOnPAger;
    property PagerRefresh: boolean read FPagerRefresh write FPagerRefresh;
    property BorderSwitching: boolean read FBorderSwitching write FBorderSwitching;
    property PagerSwitching: boolean read FPagerSwitching write FPagerSwitching;
  end;

var
  AppliDir: TFilename;             // directory of exe
  WindowsManager: TWindowsManager; // manage all windows

  Config: TConfig;                 // store configuration
  ConfigFilename: TFilename;       // configuration filename

  LocationFilename: TFilename;     // restore application filename

  procedure GetDirectories;

implementation

uses
  JVDutil, JVDwutil, JVDlng, JclSysInfo;

{*****************************************************************************}
{ Directories and filenames                                                   }
{*****************************************************************************}

procedure GetDirectories;
var
  AppliName: TFilename;
  ConfigDir: TFilename;
begin
  // Get application directory
  AppliDir:= ExtractFilePath(Application.ExeName);
  // Get application name
  AppliName:= Application.ExeName;
  delete(AppliName,1,length(ExtractFileDir(Application.ExeName)));
  AppliName:= copy(AppliName,1,length(AppliName)-4);
  // Build configuration directory
  ConfigDir:= GetUserApplicationDataDir;
  if ConfigDir = '' then ConfigDir:= AppliDir
  else
  begin
    ConfigDir:= ConfigDir + AppliName + '\';
    if not DirectoryExists(ConfigDir) then
    begin
      mkdir(ConfigDir);
      if FileExists(AppliDir + AppliName + '.ini') then
        CopyFile(PChar(AppliDir+AppliName+'.ini'),
          PChar(ConfigDir+AppliName+'.ini'),true);
      if FileExists(AppliDir + AppliName + '.loc') then
        CopyFile(PChar(AppliDir+AppliName+'.loc'),
          PChar(ConfigDir+AppliName+'.loc'),true);
    end;
  end;
  // Set location filename
  LocationFilename:= ConfigDir + AppliName + '.loc';
  // Set configuration filename
  ConfigFilename:= ConfigDir + AppliName + '.ini';
end;

{*****************************************************************************}
{ TConfig Private                                                             }
{*****************************************************************************}

procedure TConfig.SetNbHDesk(ANumber: byte);
// Set the number of horizontal desk
begin
  FNbHDesk:= ANumber;
  // Compute last desk number
  LastDesk:= (FNbHDesk * FNbVDesk) -1; // desk number are 0 based
end;

{------------------------------------------------------------------------------}

procedure TConfig.SetNbVDesk(ANumber: byte);
// Set the number of vertical desk
begin
  FNbVDesk:= ANumber;
  // Compute last desk number
  LastDesk:= (FNbHDesk * FNbVDesk) -1; // desk number are 0 based
end;

{------------------------------------------------------------------------------}

procedure TConfig.SetCycling(ANumber: byte);
// Set the cycling style
begin
  if ANumber <=2 then FCycling:= ANumber
                 else FCycling:= 0;
end;

{------------------------------------------------------------------------------}

{*****************************************************************************}
{ HotKey management                                                           }
{*****************************************************************************}

procedure TConfig.InitHotKey;
// Initialize all hotkey to none
var
  N: byte;
begin
  with HotKeys do
  begin
    NextDeskHotKey     := 0;
    PrevDeskHotKey     := 0;
    DeskUpHotKey       := 0;
    DeskDownHotKey     := 0;
    for N:= 0 to MaxDesk do
    begin
      DeskName[N]:= '';
      DeskHotKey[N]:= 0;
    end;
    ToggleStickyHotKey := 0;
  end;
end;

{------------------------------------------------------------------------------}

procedure TConfig.ConvertShortCut(AShortCut: TShortCut;
  var AModifier, AVirtualKey: word);
// Convert a short cut into virtual key and modifier
var
  AShiftState: TShiftState;
begin
  ShortCutToKey(AShortCut, AVirtualKey, AShiftState);
  AModifier:= 0;
  if ssShift in AShiftState then AModifier:= AModifier + MOD_SHIFT;
  if ssCtrl in AShiftState then AModifier:= AModifier + MOD_CONTROL;
  if ssAlt in AShiftState then AModifier:= AModifier + MOD_ALT;
end;

{------------------------------------------------------------------------------}

function TConfig.AddHotKey(AHandle: THandle; AnId: integer;
  AShortCut: TShortCut): boolean;
// Register a hot key for the corresponding application (handle)
var
  AModifier: word;
  AVirtualKey: word;
begin
  ConvertShortCut(AShortCut,AModifier,AVirtualKey);
  result:= RegisterHotKey(AHandle, AnId, AModifier, AVirtualKey)
end;

{------------------------------------------------------------------------------}

procedure TConfig.RemoveHotKey(AHandle: THandle; AnId: integer);
// Cancel the registered hot key
begin
  UnregisterHotKey(AHandle,AnId);
end;

{------------------------------------------------------------------------------}

function TConfig.AddAllHotKeys: boolean;
// Register all configured hot key and return if success for all
var
  N: byte;
begin
  result:= true;
  with HotKeys do
  begin
    if PrevDeskHotKey > 0 then
      result:= result and AddHotKey(MainFormHandle,PrevDeskHKId,PrevDeskHotKey);
    if NextDeskHotKey > 0 then
      result:= result and AddHotKey(MainFormHandle,NextDeskHKId,NextDeskHotKey);
    if DeskUpHotKey > 0 then
      result:= result and AddHotKey(MainFormHandle,DeskUpHKId,DeskUpHotKey);
    if DeskDownHotKey > 0 then
      result:= result and AddHotKey(MainFormHandle,DeskDownHKId,DeskDownHotKey);
    for N := 0 to LastDesk do
    begin
      if DeskHotKey[N] > 0 then
        result:= result and AddHotKey(MainFormHandle,1000+N,DeskHotKey[N]);
    end;
    if ToggleStickyHotKey > 0 then
      result:= result and
        AddHotKey(MainFormHandle,ToggleStickyHKId,ToggleStickyHotKey);
  end;
end;

{------------------------------------------------------------------------------}

procedure TConfig.RemoveAllHotKeys;
// Cancel all registered hot key
var
  N: byte;
begin
  with HotKeys do
  begin
    if PrevDeskHotKey > 0 then
      RemoveHotKey(MainFormHandle,PrevDeskHKId);
    if NextDeskHotKey > 0 then
      RemoveHotKey(MainFormHandle,NextDeskHKId);
    if DeskUpHotKey > 0 then
      RemoveHotKey(MainFormHandle,DeskUpHKId);
    if DeskDownHotKey > 0 then
      RemoveHotKey(MainFormHandle,DeskDownHKId);
    for N := 0 to MaxDesk do
    begin
      if DeskHotKey[N] > 0 then
        RemoveHotKey(MainFormHandle,1000+N);
    end;
    if ToggleStickyHotKey > 0 then
      RemoveHotKey(MainFormHandle,ToggleStickyHKId);
  end;
end;

{*****************************************************************************}
{ Configuration management                                                    }
{*****************************************************************************}

procedure TConfig.DefaultCfg;
// Create a default configuration
begin
  InitHotKey;                    // clear hot keys
  NbHDesk        := 2;
  NbVDesk        := 2;
  LastDesk       := (NbHDesk * NbVDesk) - 1;
  Cycling        := 2;
  PagerDeskWidth := 24;
  PagerDeskHeight:= 18;
  RememberPos    := true;
  HidePager      := false;
  with HotKeys do
  begin
    PrevDeskHotKey     := 49189;  // Ctrl + Alt + left
    NextDeskHotKey     := 49191;  // Ctrl + Alt + right
    DeskUpHotKey       := 49190;  // Ctrl + ALt + up
    DeskDownHotKey     := 49192;  // Ctrl + ALt + down
    ToggleStickyHotKey := 16467;  // Ctrl + S
  end;
  // Pager
  ActiveDeskColor:= StringToColor('clBtnFace');
  InactiveDeskColor:= StringToColor('clBtnShadow');
  IconOnPager:= false;
  PagerRefresh:= false;
  // Switching
  BorderSwitching:= true;
  PagerSwitching:= true;
  // Option
  RestoreAppli:= false;
  AskRestore:= true;
  HideApplication:= true;
  Config.SaveToFile(ConfigFilename);
end;

{------------------------------------------------------------------------------}

{ TConfig Public }
constructor TConfig.Create;
// Create config
begin
  inherited;
  InitHotKey;
end;

{------------------------------------------------------------------------------}

destructor TConfig.Destroy;
// Destroy config
begin
  RemoveAllHotKeys;
  inherited;
end;

{------------------------------------------------------------------------------}

function TConfig.LoadFromFile(AFilename: TFilename): boolean;
// Load configuration from a file
var
  ConfigFile: TIniFile;       // configuration file
  N: byte;
begin
  result:= true;
  if FileExists(AFilename) then
  begin
    // Open configuration file
    ConfigFile:= TIniFile.Create(ConfigFilename);
    // General
    Language:= ConfigFile.ReadString(General,'Language',RefLng);
    NbHDesk:= ConfigFile.ReadInteger(General,'NbHDesk',1);
    NbVDesk:= ConfigFile.ReadInteger(General,'NbVDesk',1);
    PagerDeskWidth:= ConfigFile.ReadInteger(General,'PagerDeskWidth',24);
    PagerDeskHeight:= ConfigFile.ReadInteger(General,'PagerDeskHeight',18);
    RememberPos:= ConfigFile.ReadBool(General,'RememberPos',false);
    HidePager:= ConfigFile.ReadBool(General,'HidePager',false);
    Cycling:= ConfigFile.ReadInteger(General,'Cycling',0);
    // HotKey
    with HotKeys do
    begin
      PrevDeskHotKey:= ConfigFile.ReadInteger(Hotkey,'PrevDeskHotKey',0);
      NextDeskHotKey:= ConfigFile.ReadInteger(Hotkey,'NextDeskHotKey',0);
      DeskUpHotKey:= ConfigFile.ReadInteger(Hotkey,'DeskUpHotKey',0);
      DeskDownHotKey:= ConfigFile.ReadInteger(Hotkey,'DeskDownHotKey',0);
      for N:= 0 to MaxDesk do
      begin
        DeskName[N]:= ConfigFile.ReadString(Hotkey,'DeskName'+IntToStr(N),'');
        DeskHotKey[N]:= ConfigFile.ReadInteger(Hotkey,'DeskHotKey'+IntToStr(N),0);
      end;
      ToggleStickyHotKey:= ConfigFile.ReadInteger(Hotkey,'ToggleStickyHotKey',0);
    end;
    // Pager
    ActiveDeskColor:=
      StringToColor(ConfigFile.ReadString(Pager,'ActiveDeskColor','clBtnFace'));
    InactiveDeskColor:=
      StringToColor(ConfigFile.ReadString(Pager,'InactiveDeskColor','clBtnShadow'));
    IconOnPager:= ConfigFile.ReadBool(Pager,'IconOnPager',false);
    PagerRefresh:= ConfigFile.ReadBool(Pager,'PagerRefresh',false);
    // Option
    RestoreAppli:= ConfigFile.ReadBool(Option,'RestoreAppli',false);
    AskRestore:= ConfigFile.ReadBool(Option,'AskRestore',true);
    HideApplication:= ConfigFile.ReadBool(Option,'HideApplication',true);
    // Switching
    BorderSwitching:= ConfigFile.ReadBool(Switching,'Border',false);
    PagerSwitching:= ConfigFile.ReadBool(Switching,'Pager',false);
    // Close configuration file
    ConfigFile.Free;
    // Register all hot key
    result:= AddAllHotKeys;
  end
  else DefaultCfg;
end;

{------------------------------------------------------------------------------}

function TConfig.SaveToFile(AFilename: TFilename): boolean;
// Save configuration to a file
var
  ConfigFile: TIniFile;       // configuration file
  N: byte;
begin
  // Create, open configuration file
  ConfigFile:= TIniFile.Create(ConfigFilename);
  // General
  ConfigFile.WriteInteger(General,'NbHDesk',NbHDesk);
  ConfigFile.WriteInteger(General,'NbVDesk',NbVDesk);
  ConfigFile.WriteInteger(General,'Cycling',Cycling);
  ConfigFile.WriteInteger(General,'PagerDeskWidth',PagerDeskWidth);
  ConfigFile.WriteInteger(General,'PagerDeskHeight',PagerDeskHeight);
  ConfigFile.WriteBool(General,'RememberPos',RememberPos);
  ConfigFile.WriteBool(General,'HidePager',HidePager);
  // HotKey
  with HotKeys do
  begin
    ConfigFile.WriteInteger(HotKey,'PrevDeskHotKey',PrevDeskHotKey);
    ConfigFile.WriteInteger(HotKey,'NextDeskHotKey',NextDeskHotKey);
    ConfigFile.WriteInteger(HotKey,'DeskUpHotKey',DeskUpHotKey);
    ConfigFile.WriteInteger(HotKey,'DeskDownHotKey',DeskDownHotKey);
    for N:= 0 to MaxDesk do
    begin
      ConfigFile.WriteString(HotKey,'DeskName'+IntToStr(N),DeskName[N]);
      ConfigFile.WriteInteger(HotKey,'DeskHotKey'+IntToStr(N),DeskHotKey[N]);
    end;
    ConfigFile.WriteInteger(HotKey,'ToggleStickyHotKey',ToggleStickyHotKey);
  end;
  // Pager
  ConfigFile.WriteString(Pager,'ActiveDeskColor',ColorToString(ActiveDeskColor));
  ConfigFile.WriteString(Pager,'InactiveDeskColor',
    ColorToString(InactiveDeskColor));
  ConfigFile.WriteBool(Pager,'IconOnPager',IconOnPager);
  ConfigFile.WriteBool(Pager,'PagerRefresh',PagerRefresh);
  // Option
  ConfigFile.WriteBool(Option,'RestoreAppli',RestoreAppli);
  ConfigFile.WriteBool(Option,'AskRestore',AskRestore);
  ConfigFile.WriteBool(Option,'HideApplication',HideApplication);
  // Switching
  ConfigFile.WriteBool(Switching,'Border',BorderSwitching);
  ConfigFile.WriteBool(Switching,'Pager',PagerSwitching);
  // Close configuration file
  ConfigFile.Free;
  // Remove all hot key
  RemoveAllHotKeys;
  // Register all hot key
  result:= AddAllHotKeys;
end;

{------------------------------------------------------------------------------}

function TConfig.AdaptDeskNumber(ADeskNumber, CurrDesk: integer): byte;
// Adapt the given desk number in order to match configuration
var
  Delta: integer;
  CurrHPos: byte;
  CurrVPos: byte;
begin
  Delta:= ADeskNumber - CurrDesk;
  CurrHPos:= (CurrDesk + 1) mod NbHDesk;
  if CurrHPos = 0 then CurrHPos:= NbHDesk;
  CurrVPos:= (CurrDesk div NbHDesk) + 1;
  // No cycling
  if Cycling = 0 then
  begin
    // Start of line
    if ((Delta = -1) and (CurrHPos = 1)) then
      ADeskNumber:= CurrDesk;
    // End of line
    if ((Delta = 1) and (CurrHPos = NbHDesk)) then
      ADeskNumber:= CurrDesk;
    // Top of column
    if ((Delta = -NbHDesk) and (CurrVPos = 1)) then
      ADeskNumber:= CurrDesk;
    // Bottom of column
    if ((Delta = NbHDesk) and (CurrVPos = NbVDesk)) then
      ADeskNumber:= CurrDesk;
  end;
  // Cycling in the same line or column
  if Cycling = 1 then
  begin
    // Start of line
    if ((Delta = -1) and (CurrHPos = 1)) then
      ADeskNumber:= ADeskNumber + NbHDesk;
    // End of line
    if ((Delta = 1) and (CurrHPos = NbHDesk)) then
      ADeskNumber:= ADeskNumber - NbHDesk;
    // Top of column
    if ((Delta = -NbHDesk) and (CurrVPos = 1)) then
      ADeskNumber:= LastDesk - NbHDesk + CurrHPos;
    // Bottom of column
    if ((Delta = NbHDesk) and (CurrVPos = NbVDesk)) then
      ADeskNumber:= CurrHPos - 1;
  end;
  // Full cycling with line or column change
  if Cycling = 2 then
  begin
    // First desk
    if (Delta = -1) and (CurrDesk = 0) then ADeskNumber:= LastDesk;
    if (Delta = -NbHDesk) and (CurrDesk = 0) then ADeskNumber:= LastDesk;
    // Last desk
    if (Delta = 1) and (CurrDesk = LastDesk) then ADeskNumber:= 0;
    if (Delta = NbHDesk) and (CurrDesk = LastDesk) then ADeskNumber:= 0;
    // Top of column
    if ((CurrDesk <> 0) and (Delta = -NbHDesk) and (CurrVPos = 1)) then
      ADeskNumber:= LastDesk - NbHDesk + CurrHPos - 1;
    // Bottom of column
    if ((CurrDesk <> LastDesk) and (Delta = NbHDesk)
      and (CurrVPos = NbVDesk)) then  ADeskNumber:= CurrHPos;
  end;
  result:= ADeskNumber;
end;

{------------------------------------------------------------------------------}

procedure TConfig.LoadPagerPos(AFilename: TFilename; var X, Y: integer);
var
  ConfigFile: TIniFile;       // configuration file
begin
  // Create, open configuration file
  ConfigFile:= TIniFile.Create(ConfigFilename);
  // General
  X:= ConfigFile.ReadInteger(General,'XPager',0);
  Y:= ConfigFile.ReadInteger(General,'YPager',0);
  // Close configuration file
  ConfigFile.Free;
end;

{------------------------------------------------------------------------------}

procedure TConfig.SavePagerPos(AFilename: TFilename; X, Y: integer);
var
  ConfigFile: TIniFile;       // configuration file
begin
  // Create, open configuration file
  ConfigFile:= TIniFile.Create(ConfigFilename);
  // General
  ConfigFile.WriteInteger(General,'XPager',X);
  ConfigFile.WriteInteger(General,'YPager',Y);
  // Close configuration file
  ConfigFile.Free;
end;

{------------------------------------------------------------------------------}

procedure TConfig.SaveLanguage(AFilename: TFilename);
var
  ConfigFile: TIniFile;       // configuration file
begin
  // Create, open configuration file
  ConfigFile:= TIniFile.Create(ConfigFilename);
  // General
  ConfigFile.WriteString(General,'Language',FLanguage);
  // Close configuration file
  ConfigFile.Free;
end;

{------------------------------------------------------------------------------}
{------------------------------------------------------------------------------}

end.
