unit FileListView;

/////////////////////////////////////////////////////////////
//            File list viewer componant                   //
//         Copyright (C) 2001 Alain JAFFRE                 //
//             http://jack.r.free.fr                       //
//                 jack.r@free.fr                          //
/////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////
//                Update history                           //
//                                                         //
//  V0.1.0  First version                                  //
//                                                         //
/////////////////////////////////////////////////////////////

{***************************************************************************}
{ L'usage, la modification, la diffusion de ce code source est libre.       }
{                                                                           }
{ Ce code est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE      }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER.                                      }
{                                                                           }
{***************************************************************************}

{***************************************************************************}
{ Usage, modification, distribution of this source code is free.            }
{                                                                           }
{ This source is distributed in the hope it will be useful, but WITHOUT     }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE.                                         }
{                                                                           }
{***************************************************************************}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ShellApi;

resourcestring
  RsName = 'Nom';
  RsSize = 'Taille';
  RsDate = 'Date';
  RsNewFolder = 'Nouveau dossier';
  RsDelete = 'Suppression de : ';
  RsErrorRemoveDir = 'Impossible de supprimer : ';
  RsErrorDeleteFile = 'Impossible de supprimer : ';

type
  TListItemType = (itFile, itFolder);

  TDirChange = procedure(Sender: TObject) of Object;

  TFileListView = class(TCustomListView)
  private
    { Dclarations prives }
    FCurrentDir: string;               // Current directory displayed
    FColumnToSort: integer;            // Which column did we sort
    FEditing: boolean;                 // Are we editing a caption
    FOldFilename: string;
    FDeleteEnabled: boolean;           // Is deletion available
    ImlLargeIcon: TImageList;
    ImlSmallIcon: TImageList;
    FDisplayed: boolean;               // Have we displayed the list
    FDirChange: TDirChange;            // Did we have change directory
    FOnCompare: TNotifyEvent;
    FOnColumnClick: TNotifyEvent;
    FOnDblClick: TNotifyEvent;
    FOnEdited: TNotifyEvent;
    FOnEditing: TNotifyEvent;
    FOnKeyDown: TNotifyEvent;
    FOnKeyUp: TNotifyEvent;
    procedure SetCurrentDir(Dir: string);
    function GetCurrentDir: string;
    procedure AddNewItem(Item:TListItem;AData: TWin32FindData);
    function GetListItemType(Item: TListItem): TListItemType;
    procedure InternalCompare(Sender: TObject; Item1,
      Item2: TListItem; Data: Integer; var Compare: Integer);
    procedure InternalColumnClick(Sender: TObject; Column: TListColumn);
    procedure InternalDblClick(Sender: TObject);
    procedure InternalEdited(Sender: TObject; Item: TListItem; var S: String);
    procedure InternalEditing(Sender: TObject; Item: TListItem;
      var AllowEdit: Boolean);
    procedure InternalKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure InternalKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
  protected
    { Dclarations protges }
  public
    { Dclarations publiques }
    Constructor Create(AOwner:TComponent); Override;
    Destructor Destroy; Override;
    procedure ClearFileList;
    procedure ReadFileList;
    procedure DoDirUp;
    function CreateNewDir(Dir: string): boolean;
    function GetFileProperties: TSearchRec;
  published
    { Dclarations publies }
    property Align;
    property AllocBy;
    property Anchors;
    property BiDiMode;
    property BorderStyle;
    property BorderWidth;
    property Checkboxes;
    property Color;
    property Constraints;
    property Ctl3D;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property FlatScrollBars;
    property FullDrag;
    property GridLines;
    property HideSelection;
    property HotTrack;
    property HotTrackStyles;
    property HoverTime;
    property IconOptions;
    property Items;
    property LargeImages;
    property MultiSelect;
    property OwnerData;
    property OwnerDraw;
    property ReadOnly default False;
    property RowSelect;
    property ParentBiDiMode;
    property ParentColor default False;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowColumnHeaders;
    property ShowWorkAreas;
    property ShowHint;
    property SmallImages;
    property SortType;
    property StateImages;
    property TabOrder;
    property TabStop default True;
    property ViewStyle;
    property Visible;
// -- Added property compare to TListView -- //
    property CurrentDir: string read GetCurrentDir write SetCurrentDir;
    property DeleteEnabled: boolean read FDeleteEnabled write FDeleteEnabled
      default false;
    property OnDirChange: TDirChange read FDirChange write FDirChange;
// -- End of added property compare to TListView -- //
    property OnAdvancedCustomDraw;
    property OnAdvancedCustomDrawItem;
    property OnAdvancedCustomDrawSubItem;
    property OnChange;
    property OnChanging;
    property OnClick;
    property OnColumnClick: TNotifyEvent read FOnColumnClick write FOnColumnClick;
    property OnColumnDragged;
    property OnColumnRightClick;
    property OnCompare: TNotifyEvent read FOnCompare write FOnCompare;
    property OnContextPopup;
    property OnCustomDraw;
    property OnCustomDrawItem;
    property OnCustomDrawSubItem;
    property OnData;
    property OnDataFind;
    property OnDataHint;
    property OnDataStateChange;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnDeletion;
    property OnDrawItem;
    property OnEdited: TNotifyEvent read FOnEdited write FOnEdited;
    property OnEditing: TNotifyEvent read FOnEditing write FOnEditing;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnGetImageIndex;
    property OnGetSubItemImage;
    property OnDragDrop;
    property OnDragOver;
    property OnInfoTip;
    property OnInsert;
    property OnKeyDown: TNotifyEvent read FOnKeyDown write FOnKeyDown;
    property OnKeyPress;
    property OnKeyUp: TNotifyEvent read FOnKeyUp write FOnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnResize;
    property OnSelectItem;
    property OnStartDock;
    property OnStartDrag;
  end;

procedure Register;

implementation

{*****************************************************************************}
{ Some useful functions                                                       }
{*****************************************************************************}

function AddEndSlashToDir(Dir: string): string;
// Add '\' to Dir if needed
begin
  if length(Dir)>1 then
    if Dir[length(Dir)]<>'\' then Dir:=Dir+'\';
  result:=Dir;
end;

function GetDirUp(Dir: string):string;
// Return the upper directory
var
  DirLength: integer;
begin
  DirLength:= length(Dir);
  if DirLength > 3 then
  begin
    // Suppress char after '\'
    while (copy(Dir,DirLength,1)<>'\') and (DirLength>0) do
    begin
      delete(Dir,DirLength,1);
      dec(DirLength);
    end;
    // Suppress '\'
    if DirLength>0 then
    begin
      delete(Dir,DirLength,1);
      dec(DirLength);
    end;
    // Find previous '\'
    while (copy(Dir,DirLength,1)<>'\') and (DirLength>0) do
    begin
      delete(Dir,DirLength,1);
      dec(DirLength);
    end;
  end;
  result:= Dir;
end;

function FileSizeFormat(Bytes: dword): string;
// Format a file size in bytes
begin
  result:= format('%d %s', [Bytes, '']);
end;

function FileTimeToDateTime(FileTime: TFileTime): TDateTime;
// Convert a file time return by Windows to a TDateTime
{
  Snaffled from UseNet, so don't whine about the hard-coded magic numbers.
  It does what it's supposed to :)
}
begin
  result := (Comp(FileTime) / 8.64E11) - 109205.0
end;

function GetShellInfo(const FileName: string;
  dwFileAttributes: DWORD): TSHFileInfo;
// Return the shell file info of a file
begin
  SHGetFileInfo(PChar(FileName), dwFileAttributes, result, SizeOf(TSHFileInfo),
    SHGFI_SYSICONINDEX or SHGFI_USEFILEATTRIBUTES);
end;

function RecursiveDelete(Origin: string):boolean;
// Recursivly delete directories and contained files without prompt
  function ScanRep(var Path: string):boolean;
  var
    SRec: TSearchRec;
    Pathlen: integer;
    Res: integer;
    Name: string;
  begin
    result:= true;
    Path:= AddEndSlashToDir(Path);
    Pathlen:= length(Path);
    // Find files
    Res:= FindFirst(Path+'*.*',faAnyfile,SRec);
    if Res=0 then
    try
      while Res=0 do
      begin
        if ((SRec.Attr and faDirectory)<>faDirectory)
          and (SRec.Name<>'.') and (SRec.Name<>'..') then
        begin
          Name:= SRec.Name;
          // Delete file
          DeleteFile(Path + Name);
        end;
        Res:= FindNext(SRec);
      end;
    finally
      FindClose(SRec);
    end;
    Application.ProcessMessages;

    // Find directories
    Res:= FindFirst(path+'*.*',faDirectory,SRec);
    if Res=0 then
    try
      while (Res=0) and result do
      begin
        if ((SRec.Attr and faDirectory)=faDirectory)
          and (SRec.Name<>'.') and (SRec.Name<>'..') then
        begin
          Name:= SRec.Name;
          // Recurse
          Path:= Path + Name + '\';
          result:= ScanRep(Path);
          // Go upper directory
          setlength(Path, Pathlen);
          RemoveDir(Path + Name);
        end;
        Res:= FindNext(SRec);
      end;
    finally
      FindClose(SRec);
    end;
  end;

begin
  ScanRep(Origin);
  result:= RemoveDir(Origin);
end;

{*****************************************************************************}
{ TFileListView                                                               }
{*****************************************************************************}

{ TFileListView }

// Private //

procedure TFileListView.SetCurrentDir(Dir: string);
// Store Dir in FCurrentDir
begin
  if (not FDisplayed) or (Dir <> FCurrentDir) then
  begin
    FCurrentDir:= AddEndSlashToDir(Dir);
    ReadFileList;
    if assigned(FDirChange) then FDirChange(Self);
  end;
end;

function TFileListView.GetCurrentDir: string;
// Return FCurrentDir
begin
  result:= FCurrentDir;
end;

procedure TFileListView.AddNewItem(Item: TListItem; AData: TWin32FindData);
// Add a new item to the file list
var
  NewWindowsData: pWin32FindData;
  SFI: TSHFileInfo;
begin
  with Item do
  begin
    new(NewWindowsData);
    move(AData, NewWindowsData^, sizeof(AData));
    Data := NewWindowsData;
    // Filename
    Caption := AData.cFileName;
    // Icon
    SFI := GetShellInfo(FCurrentDir + Caption, AData.dwFileAttributes);
    ImageIndex := SFI.iIcon;

    if (AData.dwFileAttributes and File_Attribute_Directory)
       = File_Attribute_Directory
    then
      // Directory so nothing
      SubItems.Add('')
    else
      // File size
      SubItems.Add(FileSizeFormat(AData.nFileSizeLow));

    // Date
    SubItems.Add(FormatDateTime('dd/mm/yyyy hh:mm:ss',
        FileTimeToDateTime(AData.ftLastWriteTime)));
  end;
end;

function TFileListView.GetListItemType(Item: TListItem): TListItemType;
// Return the item type (itFolder, itFile)
begin
  if not assigned(Item.Data) then result := itFile
  else
  begin
    if (TWin32FindData(Item.Data^).dwFileAttributes and faDirectory <> 0) then
      result := itFolder
    else
      result := itFile;
  end;
end;

procedure TFileListView.InternalCompare(Sender: TObject; Item1,
  Item2: TListItem; Data: Integer; var Compare: Integer);
// Compare routine to sort column
var
  S1: string;
  S2: string;
  Date1: TDateTime;
  Date2: TDateTime;
begin
  case FColumnToSort of
    // Sort by name
    0 : begin
          if GetListItemType(Item1) = GetListItemType(Item2) then
          begin
            Compare := CompareText(Item1.Caption, Item2.Caption);
          end
          else
          begin
            if GetListItemType(Item1) = itFolder then Compare := -1
                                                 else Compare := 1;
          end;
        end;
    // Sort by size
    1 : begin
          S1:= trim(Item1.SubItems[0]);
          S2:= trim(Item2.SubItems[0]);
          if S1 = '' then Compare := -1
          else
          if S2 = '' then Compare := 1
          else
          if GetListItemType(Item1) = GetListItemType(Item2) then
          begin
            if StrToInt(S1) = StrToInt(S2) then
            begin
              Compare := CompareText(Item1.Caption, Item2.Caption);
            end
            else
            begin
              if StrToInt(S1) < StrToInt(S2) then Compare := -1
                                             else Compare := 1;
            end;
          end
          else
          begin
            if GetListItemType(Item1) = itFolder then Compare := -1
                                                 else Compare := 1;
          end;
        end;
    // Sort by date
    2 : begin
          if GetListItemType(Item1) = GetListItemType(Item2) then
          begin
            Date1:= StrToDateTime(Item1.SubItems[1]);
            Date2:= StrToDateTime(Item2.SubItems[1]);
            if Date1=Date2 then
            begin
              Compare := CompareText(Item1.Caption, Item2.Caption);
            end
            else
            begin
              if Date1 < Date2 then Compare:=-1
                               else Compare:= 1;
            end;
          end
          else
          begin
            if GetListItemType(Item1) = itFolder then Compare := -1
                                                 else Compare := 1;
          end;
        end;
  end;
  if Assigned(FOnCompare) then FOnCompare(Sender);
end;

procedure TFileListView.InternalColumnClick(Sender: TObject;
  Column: TListColumn);
// Sort the clicked column
begin
  FColumnToSort := Column.Index;
  (Sender as TCustomListView).AlphaSort;
  if Assigned(FOnColumnClick) then FOnColumnClick(Sender);
end;

procedure TFileListView.InternalDblClick(Sender: TObject);
// Change directory if double click on directory
var
  Name: string;
begin
  if assigned(Selected) then
  begin
    Name:= Selected.Caption;
    case GetListItemType(Selected) of
      itFolder : SetCurrentDir(GetCurrentDir + Name);
      itFile   : // nothing;
    end;
  end;
  if Assigned(FOnDblClick) then FOnDblClick(Sender);
end;

procedure TFileListView.InternalEdited(Sender: TObject; Item: TListItem;
  var S: String);
// Rename edited file or directory
begin
  if S <> FOldFilename then
  begin
    if not renamefile(FCurrentDir + FOldFilename,FCurrentDir + S) then
    begin
      Beep;
      S:= FOldFilename;
    end;
  end;
  if Assigned(FOnEdited) then FOnEdited(Sender);
  FEditing:= false;
end;

procedure TFileListView.InternalEditing(Sender: TObject; Item: TListItem;
  var AllowEdit: Boolean);
// Store the old caption
begin
  FOldFilename:= Selected.Caption;
  FEditing:= true;
  if Assigned(FOnEditing) then FOnEditing(Sender);
end;

procedure TFileListView.InternalKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
// Refresh file list or edit selected caption
begin
  case Key of
    VK_F2 : Selected.EditCaption;
    VK_F5 : ReadFileList;
  end;
  if Assigned(FOnKeyDown) then FOnKeyDown(Sender);
end;

procedure TFileListView.InternalKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
// Manage deletion if FDeleteEnabled = true
var
  CurrentItem: TListItem;
  NextItem: TListItem;
  Answer: word;
begin
  Answer:= mrNo;
  if FEditing and (Key=VK_RETURN) then FEditing:= false;
  if (not FEditing) and FDeleteEnabled and (Key=VK_DELETE) and (SelCount>0) then
  begin
    Screen.Cursor:= crHourGlass;
    Items.BeginUpdate;
    CurrentItem:= Selected;
    while SelCount>0 do
    begin
     if SelCount>0 then
       NextItem:= GetNextItem(CurrentItem,sdAll,[isSelected])
       else NextItem:= nil;
     if Answer<> mrAll then Answer:= MessageDlg(RsDelete+ CurrentItem.Caption,
        mtWarning, [mbYes,mbNo,mbAll], 0);
     if Answer in [mrYes,mrAll] then
     begin
       case GetListItemType(CurrentItem) of
         itFolder : if RecursiveDelete(FCurrentDir + CurrentItem.Caption)
                      then Items.Delete(Items.IndexOf(CurrentItem))
                    else
                    begin
                      Beep;
                      MessageDlg(RsErrorRemoveDir + CurrentItem.Caption,
                        mtWarning, [mbYes,mbNo,mbAll], 0);
                      Items.Item[Items.IndexOf(CurrentItem)].Selected:= false;
                    end;

         itFile  : if DeleteFile(FCurrentDir + CurrentItem.Caption) then
                        Items.Delete(Items.IndexOf(CurrentItem))
                   else
                   begin
                     Beep;
                     MessageDlg(RsErrorDeleteFile + CurrentItem.Caption,
                       mtWarning, [mbYes,mbNo,mbAll], 0);
                     Items.Item[Items.IndexOf(CurrentItem)].Selected:= false;
                   end;
       end;
     end
     else Items.Item[Items.IndexOf(CurrentItem)].Selected:= false;
     if NextItem <> nil then CurrentItem:= NextItem;
    end;
    Items.EndUpdate;
    Update;
    Screen.Cursor:= crDefault;
  end;
  if Assigned(FOnKeyUp) then FOnKeyUp(Sender);
end;

// Protected //

// Public //
constructor TFileListView.Create(AOwner: TComponent);
// Create the componant
var
  NewColumn: TListColumn;
  SFI: TSHFileInfo;
begin
  inherited Create(AOwner);
  // Name column
  NewColumn := Columns.Add;
  NewColumn.Caption := RsName;
  NewColumn.Alignment:= taLeftJustify;
  NewColumn.Width:= 200;
  // Size column
  NewColumn := Columns.Add;
  NewColumn.Caption := RsSize;
  NewColumn.Alignment:= taRightJustify;
  NewColumn.Width:= 90;
  // Date column
  NewColumn := Columns.Add;
  NewColumn.Caption := RsDate;
  NewColumn.Alignment:= taLeftJustify;
  NewColumn.Width:= 120;
  // Large icon image list
  ImlLargeIcon:= TImageList.Create(self);
  ImlLargeIcon.ShareImages:= true;
  ImlLargeIcon.Handle := SHGetFileInfo('', 0, SFI, SizeOf(SFI),
    SHGFI_SYSICONINDEX or SHGFI_LARGEICON);
  LargeImages:= ImlLargeIcon;
  // Small icon image list
  ImlSmallIcon:= TImageList.Create(self);
  ImlSmallIcon.ShareImages:= true;
  ImlSmallIcon.Handle := SHGetFileInfo('', 0, SFI, SizeOf(SFI),
    SHGFI_SYSICONINDEX or SHGFI_SMALLICON);
  SmallImages:= ImlSmallIcon;
  // Internal process
  FOnCompare:= nil;
  inherited OnCompare:= InternalCompare;
  FOnColumnClick:= nil;
  inherited OnColumnClick:= InternalColumnClick;
  FOnDblClick:= nil;
  inherited OnDblClick:= InternalDblClick;
  FOnEdited:= nil;
  inherited OnEdited:= InternalEdited;
  FOnEditing:= nil;
  inherited OnEditing:= InternalEditing;
  FOnKeyDown:= nil;
  inherited OnKeyDown:= InternalKeyDown;
  FOnKeyUp:= nil;
  inherited OnKeyUp:= InternalKeyUp;
  // View style
  ViewStyle:= vsReport;
  // Sort on name column
  FColumnToSort:= 0;
  // Init internal variable
  FCurrentDir:= 'C:\';
  FEditing:= false;
  FOldFilename:= '';
  FDisplayed:= false;
end;

destructor TFileListView.Destroy;
// Destroy the componant
begin
  // Free everything
  ImlSmallIcon.Free;
  ImlLargeIcon.Free;
  inherited Destroy;
end;

procedure TFileListView.ClearFileList;
// Clear the filelist
var
  N: integer;
begin
 if FDisplayed then
 begin
   for N:= 0 to pred(Items.Count) do
     if assigned(Items[N].Data) then dispose(pWin32FindData(Items[N].Data));
 end;
 Items.Clear;
end;

procedure TFileListView.ReadFileList;
// Read and display the file list of FCurrentDir
var
  SrcFile:   TSearchRec;
begin
  Screen.Cursor:= crHourGlass;
  Items.BeginUpdate;
  try
    ClearFileList;
    if FCurrentDir <> '' then
    begin
      try
        if findfirst(FCurrentDir + '*.*', faAnyFile, SrcFile)=0 then
        begin
          try
            repeat
              if not ((SrcFile.Name='..') or (SrcFile.Name='.')) then
                 AddNewItem(Items.Add,SrcFile.FindData);
            until findnext(SrcFile)<>0;
            FDisplayed:= true;
          finally
            Sysutils.findclose(SrcFile);
          end;
        end;
      except
        on exception do;
      end;
    end;
  finally
    AlphaSort;
    Items.EndUpdate;
    Screen.Cursor:= crDefault;
  end;
end;

procedure TFileListView.DoDirUp;
// Go to upper directory
begin
  SetCurrentDir(GetDirUp(FCurrentDir));
end;

function TFileListView.CreateNewDir(Dir: string): boolean;
// Create a new dierctory with Dir as name or with RsNewFolder as name if Dir=''
var
  NewItem: TListItem;
  SrcFile:   TSearchRec;
begin
  result:= true;
  if Dir = '' then Dir:= RsNewFolder;
  if CreateDir(FCurrentDir + Dir) then
  begin
    NewItem:= Items.Add;
    try
      if findfirst(FCurrentDir + Dir, faAnyFile, SrcFile)=0 then
      begin
        try
          if not ((SrcFile.Name='..') or (SrcFile.Name='.')) then
               AddNewItem(NewItem,SrcFile.FindData);
        finally
          Sysutils.findclose(SrcFile);
        end;
      end;
    except
      on exception do;
    end;
    if Dir = RsNewFolder then NewItem.EditCaption
                         else ReadFileList;
  end
  else result:= false;
end;

function TFileListView.GetFileProperties: TSearchRec;
// Return the selected file properties
var
  SrcFile: TSearchRec;
begin
  if findfirst(FCurrentDir + Selected.Caption, faAnyFile, SrcFile)<>0 then
  begin
    SrcFile.Time:= 0;
    SrcFile.Size:= 0;
    SrcFile.Attr:= 0;
    SrcFile.Name:= '';
    SrcFile.ExcludeAttr:= 0;
  end;
  result:= SrcFile;
end;

{*****************************************************************************}
{ Register                                                                    }
{*****************************************************************************}

procedure Register;
// Register the componant
begin
  RegisterComponents('Win32', [TFileListView]);
end;


end.
