#!/usr/bin/env tclsh
# -*- mode: tcl; coding: utf-8-unix; -*-
# Time-stamp: <2008-12-27 14:06:27 alain>%
# $Id: $
#-------------------------------------------------------------------------------
# Created in 11/2008 by A.JAFFRE jack.r@free.fr
# http://jack.r.free.fr
#-------------------------------------------------------------------------------
# 20081201 version 0.5
#   First shown version
# 20081207 version 0.6
#   Addition of configuration backup in a file
# 20081207 version 0.7
#   Addition of multilingual user interface
# 20081213 version 0.8
#   Universal source which can be use by shell, tclkit, starkit, starpack
#    etcl, etcl kit, etcl pack
# 20081215 version 0.8.1
#   Bug correction in ::confdown::buildGui, missing window path in reqwidth and
#    reqheight
# 20081226 version 0.9.0
#   New color selection dialog for Linux platform
#   Open configuration dialog at start up only if it is the first time
#    (no confdown.ini file)
#   Be sure that progress bar is always fully visible in the screen
#   Be sure that configuration dialog is always fully visible in the screen
#   Be sure we can run only one instance
# 20081226 version 1.0.0
#   First stable release
# 20081227 version 1.1.0
#   Improvement in configuration dialog, duration tab to have separator on
#    full width and make more visible switch between end time and duration.
#    Only corresponding spinbox are activated.
#   Addition of management through keyboard shortcut
#-------------------------------------------------------------------------------
# Roadmap
#-------------------------------------------------------------------------------

#-------------------------------------------------------------------------------
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#-------------------------------------------------------------------------------

namespace eval ::confdown:: {
	variable appVersion 1.1.0
	variable appName "Conference count down"

	variable appEngine
	variable appFolder
	variable scriptFolder

	variable widgets
	variable idx

	variable running

	variable cfgFilename
	variable cfg
	variable tmpCfg
# These are arrays with:
#   normal_color   : color use normally
#   warning_color  : color used in warning area
#   critical_color : color used in critical area
# 	end_type       : 0 end time, 1 duration
# 	start_time     : time at which we start in seconds
# 	end_time       : time at which we stop in seconds
# 	duration       : duration value in seconds
# 	warning        : warning duration in seconds
# 	critical       : critical duration in seconds
#   save_config    : save configuration on exit
#   save_position  : seve position on exit
#   x              : position on x axis
#   y              : position on y axis
#   save_lng       : seve user interface language on exit
#   lng            : user interface language
# In case of tmpCfg, following additionnal informations will
#  be computed
# 	end_hour       : hour at which we stop (0-23)
# 	end_minute     : minute at which we stop (0-59)
# 	end_second     : second at which we stop (0-59)
# 	duration_hour  : hour for duration
# 	duration_minute: minute for duration
# 	duration_second: second for duration
# 	warning_hour   : hour for warning delay
# 	warning_minute : minute for warning delay
# 	warning_second : second for warning delay
# 	critical_hour  : hour for critical delay
# 	critical_minute: minute for critical delay
# 	critical_second: second for critical delay

	variable bNormal
	variable bWarning
	variable bCritical

	variable firstTime 1
	variable wmBorder 23

	variable debug
}

# Adjustment to be compatible with tclsh, TclKit, eTcl
set ::confdown::appFolder [file dirname [file normalize $::argv0]]
set ::confdown::scriptFolder [file dirname [file normalize [info script]]]

if { [info commands ::etcl::run] eq "::etcl::run" } {
	set ::confdown::appEngine "etcl"
} else {
	if {![catch {package require starkit}]} {
		set ::confdown::appEngine "tclkit"
		set ::confdown::appFolder [file dirname $::confdown::appFolder]
	} else {
		set ::confdown::appEngine "shell"
	}
}


# Adjust auto_path to contain script folder and application folder
lappend auto_path $::confdown::appFolder
if {$::confdown::scriptFolder != $::confdown::appFolder} {
	lappend auto_path $::confdown::scriptFolder
}
# Adjust auto_path to contain lib folder if exist
if {[file isdirectory [file join $::confdown::scriptFolder lib]]} {
	set ::auto_path [linsert $::auto_path 0 [file join $::confdown::scriptFolder lib]]
}

# Specify required packages
package require Tcl 8.5
package require Tk 8.5

package require msgcat
namespace import ::msgcat::*

package require inifile
package require lngcode
package require lock

if {$::tcl_platform(os) eq {Linux}} {
	package require colorDlg
	rename ::tk_chooseColor ::_tk_chooseColor
	proc ::tk_chooseColor {args} {
		return [eval colorDlg $args]
	}
}


set ::confdown::debug 0

#*******************************************************************************
#                  Program utilities and widgets  from others                  *
#*******************************************************************************

#---------------
# progress.tcl
#---------------
# Created by William J Giddings, 2006
#
# Simple progress bar with minimal options
#
# Notes: Value sent to bar must be within range 0.0 - 1.0
#
#---------------
# Credits: http://wiki.tcl.tk/1146
#---------------

#---------------
# simple progress bar
#---------------
proc progress {w args} {
	eval frame $w $args ;# create the "base" thing
	frame $w.bar -relief raised -borderwidth 3
	place $w.bar -relwidth .3 -relheight 1.0

	rename $w _$w      ;# keep the original widget command
	# Here comes the overloaded widget proc:
	proc $w {args} {
		set self [lindex [info level 0] 0] ;# get name I was called with
		foreach {opt val} $args {
			switch -- $opt {
				-val   {eval "place $self.bar -relwidth $val"}
				-bg    {eval "$self.bar configure -background $val"}
				default {uplevel 1 _$self $args}
			}
		}
	}
	return $w ;# like the original "text" command
}

#*******************************************************************************
#                         Program utilities and widgets                        *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Convert seconds into hour, minute and second values
#-------------------------------------------------------------------------------
proc ::confdown::seconds2hms {seconds} {
	set h [ expr {$seconds / 3600} ]
	set seconds [ expr {$seconds % 3600} ]
	set m [ expr {$seconds / 60} ]
	set s [ expr {$seconds % 60} ]
	return [format "%02d %02d %02d" $h $m $s]
}

#-------------------------------------------------------------------------------
# Convert hour, minute and second values into seconds
#-------------------------------------------------------------------------------
proc ::confdown::hms2seconds {h m s} {
	set seconds $s
	incr seconds [expr { $m * 60}]
	incr seconds [expr { $h * 3600}]
	return $seconds
}

#*******************************************************************************
#                                 Configuration                                *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Program initialisation
#-------------------------------------------------------------------------------
proc ::confdown::init {} {
	set filename [lindex [file split [info script]] end]
	set filename [string replace $filename end-3 end ".ini"]
	set ::confdown::cfgFilename [file join $::confdown::appFolder $filename]

	set ::confdown::running 0

	set ::confdown::cfg(end_type) 1
	set ::confdown::cfg(duration) [expr {45 * 60}]
	set ::confdown::cfg(warning)  [expr {15 * 60}]
	set ::confdown::cfg(critical) [expr {5 * 60}]
	set ::confdown::cfg(end_time) [ clock seconds ]

	set ::confdown::cfg(normal_color)   #00ff00
	set ::confdown::cfg(warning_color)  #ffa500
	set ::confdown::cfg(critical_color) #ff0000

	set ::confdown::cfg(save_config) 0
	set ::confdown::cfg(save_position) 0
	set ::confdown::cfg(x) 0
	set ::confdown::cfg(y) 0
	set ::confdown::cfg(save_lng) 0
	set ::confdown::cfg(lng) {}
}

#-------------------------------------------------------------------------------
# Save configuration in a file
#-------------------------------------------------------------------------------
proc ::confdown::saveConfig {} {
	set iniFile [::ini::open $::confdown::cfgFilename w+]
	# times
	::ini::set $iniFile "time" "end_type"  $::confdown::cfg(end_type)
	::ini::set $iniFile "time" "duration"  $::confdown::cfg(duration)
	::ini::set $iniFile "time" "warning"  $::confdown::cfg(warning)
	::ini::set $iniFile "time" "critical"  $::confdown::cfg(critical)
	::ini::set $iniFile "time" "end_time"  $::confdown::cfg(end_time)
	# colors
	::ini::set $iniFile "color" "normal" $::confdown::cfg(normal_color)
	::ini::set $iniFile "color" "warning" $::confdown::cfg(warning_color)
	::ini::set $iniFile "color" "critical" $::confdown::cfg(critical_color)
	# save
	::ini::set $iniFile "save" "config" $::confdown::cfg(save_config)
	::ini::set $iniFile "save" "position" $::confdown::cfg(save_position)
	if {$::confdown::cfg(save_position)} {
		::ini::set $iniFile "save" "x" [ winfo rootx .main ]
		::ini::set $iniFile "save" "y" [ winfo rooty .main ]
	}
	::ini::set $iniFile "save" "lng" $::confdown::cfg(save_lng)
	if {$::confdown::cfg(save_lng)} {
		::ini::set $iniFile "save" "lng_code" $::confdown::cfg(lng)
	}

	catch {::ini::commit $iniFile}
	::ini::close $iniFile
}

#-------------------------------------------------------------------------------
# Load configuration from a file
#-------------------------------------------------------------------------------
proc ::confdown::loadConfig {} {
	if { [file exists $::confdown::cfgFilename] } {
		set iniFile [ ::ini::open $::confdown::cfgFilename r ]

		# times
		foreach {key value} [::ini::get $iniFile "time" ] {
			switch -exact $key \
				end_type { if {[string is boolean $value]} {
					set ::confdown::cfg(end_type) $value} } \
				duration { if {[string is integer $value]} {
					set ::confdown::cfg(duration) $value} } \
				warning  { if {[string is integer $value]} {
					set ::confdown::cfg(warning) $value} } \
				critical { if {[string is integer $value]} {
					set ::confdown::cfg(critical) $value} } \
				end_time { if {[string is integer $value]} {
					set ::confdown::cfg(end_time) $value} } \
		}
		# colors
		foreach {key value} [::ini::get $iniFile "color" ] {
			if {($value >= "#000000") && ($value <= "#ffffff")} {
				switch -exact $key \
					normal   {set ::confdown::cfg(normal_color) $value} \
					warning  {set ::confdown::cfg(warning_color) $value} \
					critical {set ::confdown::cfg(critical_color) $value}
			}
		}
		# save
		foreach {key value} [::ini::get $iniFile "save" ] {
			switch -exact $key \
				config   { if {[string is boolean $value]} {
					set ::confdown::cfg(save_config) $value} } \
				position { if {[string is boolean $value]} {
					set ::confdown::cfg(save_position) $value} } \
				x        { if {[string is integer $value]} {
					set ::confdown::cfg(x) $value} } \
				y        { if {[string is integer $value]} {
					set ::confdown::cfg(y) $value} } \
				lng      { if {[string is boolean $value]} {
					set ::confdown::cfg(save_lng) $value} } \
			    lng_code { set ::confdown::cfg(lng) $value }
		}

		::ini::close $iniFile
		set ::confdown::firstTime 0
	}
}

#*******************************************************************************
#                                    GUI                                       *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Build popup menu
#-------------------------------------------------------------------------------
proc ::confdown::buildPopupMenu {f} {
	set index 0
	set ::confdown::widgets(mnu) [ menu $f.menu -tearoff 0 ]
	$f.menu add command -label [mc "Configuration"] -command ::confdown::cfgDlg
	set ::confdown::idx(mnuConfig) $index
	incr index

	# Language
	if { [ ::lngcode::getCodeList "$::confdown::scriptFolder/msg" ] } {
		set ::confdown::widgets(mnuLng) [menu $f.menu.lng -tearoff 0]
		$f.menu add cascade -label [mc "Language"] -menu $f.menu.lng
		set ::confdown::idx(mnuLng) $index
		incr index
		foreach code $::lngcode::CodeList {
			$f.menu.lng add check \
				-label [ ::lngcode::getLngName $code 0 ] \
				-command [ list ::confdown::switchLanguage $code ]
		}
	}

	$f.menu add command -label [mc "Quit"] -command ::confdown::quit
	set ::confdown::idx(mnuQuit) $index
	incr index

	if {[tk windowingsystem]=="aqua"} {
		bind $f <2> "tk_popup $f.menu %X %Y"
		bind $f <Control-1> "tk_popup $f.menu %X %Y"
	} else {
		bind $f <3> "tk_popup $f.menu %X %Y"
}

}

#-------------------------------------------------------------------------------
# Toggle end type
#-------------------------------------------------------------------------------
proc ::confdown::toggleEndType {} {
	if {$::confdown::tmpCfg(end_type) eq 0} {
		$::confdown::widgets(sEndHour) configure -state normal
		$::confdown::widgets(sEndMinute) configure -state normal
		$::confdown::widgets(sEndSecond) configure -state normal
		$::confdown::widgets(sDurationHour) configure -state disabled
		$::confdown::widgets(sDurationMinute) configure -state disabled
		$::confdown::widgets(sDurationSecond) configure -state disabled
	} else {
		$::confdown::widgets(sEndHour) configure -state disabled
		$::confdown::widgets(sEndMinute) configure -state disabled
		$::confdown::widgets(sEndSecond) configure -state disabled
		$::confdown::widgets(sDurationHour) configure -state normal
		$::confdown::widgets(sDurationMinute) configure -state normal
		$::confdown::widgets(sDurationSecond) configure -state normal
	}
}

#-------------------------------------------------------------------------------
# Build time page
#-------------------------------------------------------------------------------
proc ::confdown::buildTimePage {f} {
	# End time
	set row 1
	set rEndTime [ttk::radiobutton $f.radEndTime \
					  -text [ mc "End time" ] \
					  -command ::confdown::toggleEndType \
					  -variable ::confdown::tmpCfg(end_type) \
					  -value 0 ]

	set ::confdown::widgets(sEndHour) \
		[spinbox $f.sbEndHour \
			 -from 0 \
			 -to 23 \
			 -increment 1.0 \
			 -format %2.0f \
			 -width 2 \
			 -textvariable ::confdown::tmpCfg(end_hour) ]
	set lEndHour [ttk::label $f.lblEndHour \
					   -text [ mc "h" ] ]

	set ::confdown::widgets(sEndMinute) \
		[spinbox $f.sbEndMinute \
			 -from 0 \
			 -to 59 \
			 -increment 1.0 \
			 -format %2.0f \
			 -width 2 \
			 -textvariable ::confdown::tmpCfg(end_minute) ]
	set lEndMinute [ttk::label $f.lblEndMinute \
					   -text [ mc "m" ] ]

	set ::confdown::widgets(sEndSecond) \
		[spinbox $f.sbEndSecond \
			 -from 0 \
			 -to 59 \
			 -increment 1.0 \
			 -format %2.0f \
			 -width 2 \
			 -textvariable ::confdown::tmpCfg(end_second) ]
	set lEndSecond [ttk::label $f.lblEndSecond \
					   -text [ mc "s" ] ]

 	grid $rEndTime -column 1 -row $row -sticky w -padx 4
 	grid $::confdown::widgets(sEndHour) -column 2 -row $row -sticky w
 	grid $lEndHour -column 3 -row $row -sticky w -padx 4
 	grid $::confdown::widgets(sEndMinute) -column 4 -row $row -sticky w
 	grid $lEndMinute -column 5 -row $row -sticky w -padx 4
 	grid $::confdown::widgets(sEndSecond) -column 6 -row $row -sticky w
 	grid $lEndSecond -column 7 -row $row -sticky w -padx 4

	# Duration
	incr row
	set rDuration [ttk::radiobutton $f.radDuration \
					   -text [ mc "Duration" ] \
					   -command ::confdown::toggleEndType \
					   -variable ::confdown::tmpCfg(end_type) \
					   -value 1 ]
	set ::confdown::widgets(sDurationHour) \
		[spinbox $f.sbDurationHour \
			 -from 0 \
			 -to 23 \
			 -increment 1.0 \
			 -format %2.0f \
			 -width 2 \
			 -textvariable ::confdown::tmpCfg(duration_hour) ]
	set lDurationHour [ttk::label $f.lblDurationHour \
					   -text [ mc "h" ] ]

	set ::confdown::widgets(sDurationMinute) \
		[spinbox $f.sbDurationMinute \
			 -from 0 \
			 -to 59 \
			 -increment 1.0 \
			 -format %2.0f \
			 -width 2 \
			 -textvariable ::confdown::tmpCfg(duration_minute) ]
	set lDurationMinute [ttk::label $f.lblDurationMinute \
					   -text [ mc "m" ] ]

	set ::confdown::widgets(sDurationSecond) \
		[spinbox $f.sbDurationSecond \
			 -from 0 \
			 -to 59 \
			 -increment 1.0 \
			 -format %2.0f \
			 -width 2 \
			 -textvariable ::confdown::tmpCfg(duration_second) ]
	set lDurationSecond [ttk::label $f.lblDurationSecond \
					   -text [ mc "s" ] ]

 	grid $rDuration -column 1 -row $row -sticky w -padx 4
 	grid $::confdown::widgets(sDurationHour) -column 2 -row $row -sticky w
 	grid $lDurationHour -column 3 -row $row -sticky w -padx 4
 	grid $::confdown::widgets(sDurationMinute) -column 4 -row $row -sticky w
 	grid $lDurationMinute -column 5 -row $row -sticky w -padx 4
 	grid $::confdown::widgets(sDurationSecond) -column 6 -row $row -sticky w
 	grid $lDurationSecond -column 7 -row $row -sticky w -padx 4

	# Separator
	incr row 2
	set sep1 [ttk::separator $f.sep1 -orient horizontal]
 	grid $sep1 -column 1 -row $row -columnspan 7 -sticky ew -padx 4 -pady 4

	# Warning delay
	incr row
	set lWarningTime [ttk::label $f.lblWarningTime \
					   -text [ mc "Warning time" ] ]
	set sWarningHour [spinbox $f.sbWarningHour \
					  -from 0 \
					  -to 23 \
					  -increment 1.0 \
					  -format %2.0f \
					  -width 2 \
					  -textvariable ::confdown::tmpCfg(warning_hour) ]
	set lWarningHour [ttk::label $f.lblWarningHour \
					   -text [ mc "h" ] ]

	set sWarningMinute [spinbox $f.sbWarningMinute \
					  -from 0 \
					  -to 59 \
					  -increment 1.0 \
					  -format %2.0f \
					  -width 2 \
					  -textvariable ::confdown::tmpCfg(warning_minute) ]
	set lWarningMinute [ttk::label $f.lblWarningMinute \
					   -text [ mc "m" ] ]

	set sWarningSecond [spinbox $f.sbWarningSecond \
					  -from 0 \
					  -to 59 \
					  -increment 1.0 \
					  -format %2.0f \
					  -width 2 \
					  -textvariable ::confdown::tmpCfg(warning_second) ]
	set lWarningSecond [ttk::label $f.lblWarningSecond \
					   -text [ mc "s" ] ]

 	grid $lWarningTime -column 1 -row $row -sticky w -padx 4
 	grid $sWarningHour -column 2 -row $row -sticky w
 	grid $lWarningHour -column 3 -row $row -sticky w -padx 4
 	grid $sWarningMinute -column 4 -row $row -sticky w
 	grid $lWarningMinute -column 5 -row $row -sticky w -padx 4
 	grid $sWarningSecond -column 6 -row $row -sticky w
 	grid $lWarningSecond -column 7 -row $row -sticky w -padx 4

	# Critical delay
	incr row
	set lCriticalTime [ttk::label $f.lblCriticalTime \
					   -text [ mc "Critical time" ] ]
	set sCriticalHour [spinbox $f.sbCriticalHour \
					  -from 0 \
					  -to 23 \
					  -increment 1.0 \
					  -format %2.0f \
					  -width 2 \
					  -textvariable ::confdown::tmpCfg(critical_hour) ]
	set lCriticalHour [ttk::label $f.lblCriticalHour \
					   -text [ mc "h" ] ]

	set sCriticalMinute [spinbox $f.sbCriticalMinute \
					  -from 0 \
					  -to 59 \
					  -increment 1.0 \
					  -format %2.0f \
					  -width 2 \
					  -textvariable ::confdown::tmpCfg(critical_minute) ]
	set lCriticalMinute [ttk::label $f.lblCriticalMinute \
					   -text [ mc "m" ] ]

	set sCriticalSecond [spinbox $f.sbCriticalSecond \
					  -from 0 \
					  -to 59 \
					  -increment 1.0 \
					  -format %2.0f \
					  -width 2 \
					  -textvariable ::confdown::tmpCfg(critical_second) ]
	set lCriticalSecond [ttk::label $f.lblCriticalSecond \
					   -text [ mc "s" ] ]

 	grid $lCriticalTime -column 1 -row $row -sticky w -padx 4
 	grid $sCriticalHour -column 2 -row $row -sticky w
 	grid $lCriticalHour -column 3 -row $row -sticky w -padx 4
 	grid $sCriticalMinute -column 4 -row $row -sticky w
 	grid $lCriticalMinute -column 5 -row $row -sticky w -padx 4
 	grid $sCriticalSecond -column 6 -row $row -sticky w
 	grid $lCriticalSecond -column 7 -row $row -sticky w -padx 4

	grid columnconfigure $f 1 -weight 1
	grid columnconfigure $f 2 -weight 1
	grid columnconfigure $f 3 -weight 1
	grid columnconfigure $f 4 -weight 1
	grid columnconfigure $f 5 -weight 1
	grid columnconfigure $f 6 -weight 1
	grid columnconfigure $f 7 -weight 1

	::confdown::toggleEndType
}


#-------------------------------------------------------------------------------
# Set a color
#-------------------------------------------------------------------------------
proc ::confdown::setColor {n} {
	switch -exact $n {
		normal   { set c $::confdown::tmpCfg(normal_color) }
		warning  { set c $::confdown::tmpCfg(warning_color) }
		critical { set c $::confdown::tmpCfg(critical_color) }
	}
	set c [ tk_chooseColor -initialcolor $c -title [ mc "Choose color" ] ]
	if { $c != ""} {
		switch -exact $n {
			normal   {
				set ::confdown::tmpCfg(normal_color)   $c
				$::confdown::bNormal configure -background $c
			}
			warning  {
				set ::confdown::tmpCfg(warning_color)  $c
				$::confdown::bWarning configure -background $c
			}
			critical {
				set ::confdown::tmpCfg(critical_color) $c
				$::confdown::bCritical configure -background $c
			}
		}
	}

}

#-------------------------------------------------------------------------------
# Build color page
#-------------------------------------------------------------------------------
proc ::confdown::buildColorPage {f} {
	# Normal color
	set row 1
	set lNormalColor [ttk::label $f.lblNormalColor \
					   -text [ mc "Normal color" ] ]
	set ::confdown::bNormal [button $f.bNormalColor \
								 -background $::confdown::tmpCfg(normal_color) \
								 -width 4 \
								 -command [list ::confdown::setColor normal ] ]
 	grid $lNormalColor -column 1 -row $row -sticky w -padx 4
 	grid $::confdown::bNormal -column 2 -row $row -sticky w -padx 4

	incr row
	set lWarningColor [ttk::label $f.lblWarningColor \
					   -text [ mc "Warning color" ] ]
	set ::confdown::bWarning [button $f.bWarningColor \
								  -background $::confdown::tmpCfg(warning_color) \
								  -width 4 \
								  -command [list ::confdown::setColor warning ] ]
 	grid $lWarningColor -column 1 -row $row -sticky w -padx 4
 	grid $::confdown::bWarning -column 2 -row $row -sticky w -padx 4

	incr row
	set lCriticalColor [ttk::label $f.lblCriticalColor \
					   -text [ mc "Critical color" ] ]
	set ::confdown::bCritical [button $f.bCriticalColor \
								   -background $::confdown::tmpCfg(critical_color) \
								   -width 4 \
								   -command [list ::confdown::setColor critical ] ]
 	grid $lCriticalColor -column 1 -row $row -sticky w -padx 4
 	grid $::confdown::bCritical -column 2 -row $row -sticky w -padx 4


}

#-------------------------------------------------------------------------------
# Build save page
#-------------------------------------------------------------------------------
proc ::confdown::buildSavePage {f} {
	set row 1
	set cbConfig [ttk::checkbutton $f.cb1 -text [mc "Save configuration on exit"] \
					  -variable ::confdown::tmpCfg(save_config) ]
	grid $cbConfig -column 1 -row $row -sticky w -padx 4

	incr row
	set cbPosition [ttk::checkbutton $f.cb2 -text [mc "Save position on exit"] \
						-variable ::confdown::tmpCfg(save_position) ]
	grid $cbPosition -column 1 -row $row -sticky w -padx 4

	incr row
	set cbLanguage [ttk::checkbutton $f.cb3 -text [mc "Save language on exit"] \
						-variable ::confdown::tmpCfg(save_lng) ]
	grid $cbLanguage -column 1 -row $row -sticky w -padx 4

}

#-------------------------------------------------------------------------------
# Build help page
#-------------------------------------------------------------------------------
proc ::confdown::buildHelpPage {f} {
	set help_text ""
	lappend help_text "$::confdown::appName, version $::confdown::appVersion"
	lappend help_text ""
	lappend help_text [ mc "Management with mouse" ]
	lappend help_text [ mc "  - Left click on the progress bar to start / stop count down" ]
	lappend help_text [ mc "  - Right click on the progresss bar display menu" ]
	lappend help_text [ mc "  - Middle click or Shift + left click on the progress bar to quit" ]
	lappend help_text ""
	lappend help_text [ mc "Management with keyboard" ]
	lappend help_text [ mc "  - Space bar to start / stop count down" ]
	lappend help_text [ mc "  - Alt+arrow to move it in arrow direction"]
	lappend help_text [ mc "  - Alt+c to show configuration dialog" ]
	lappend help_text [ mc "  - Alt+m to show popup menu" ]
	lappend help_text [ mc "  - Alt+q to quit" ]
	lappend help_text ""
	lappend help_text " A.JAFFRE 2008"
	lappend help_text " http://jack.r.free.fr"

	set t [ text $f.t -yscrollcommand [list $f.sb set] -height 8 -width 40 ]
	set sb [scrollbar $f.sb -orient vertical -command [list $t yview]]

	$t insert end [ join $help_text \n ]
	$t configure -wrap word
	$t configure -state disabled

 	grid $t -column 0 -row 0 -sticky nsew -padx 4
	grid $sb -sticky ns -column 1 -row 0

	grid rowconfigure $f 0 -weight 1
	grid columnconfigure $f 0 -weight 1

}

#-------------------------------------------------------------------------------
# Build license page
#-------------------------------------------------------------------------------
proc ::confdown::buildLicensePage {f} {
	set license_text ""
	lappend license_text "$::confdown::appName, version $::confdown::appVersion"
	lappend license_text ""
	lappend license_text \
"This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program.  If not, see <http://www.gnu.org/licenses/>."

	set t [ text $f.t -yscrollcommand [list $f.sb set] -height 8 -width 40 ]
	set sb [scrollbar $f.sb -orient vertical -command [list $t yview]]

	$t insert end [ join $license_text \n ]
	$t configure -wrap word
	$t configure -state disabled

 	grid $t -column 0 -row 0 -sticky nsew -padx 4
	grid $sb -sticky ns -column 1 -row 0

	grid rowconfigure $f 0 -weight 1
	grid columnconfigure $f 0 -weight 1
}

#-------------------------------------------------------------------------------
# Set configuration values in configuration dialog
#-------------------------------------------------------------------------------
proc ::confdown::cfgDlgSet {} {
	array set ::confdown::tmpCfg [ array get ::confdown::cfg ]

	lassign [clock format $::confdown::tmpCfg(end_time) \
				 -format "%H %M %S" ] \
		::confdown::tmpCfg(end_hour) \
		::confdown::tmpCfg(end_minute) \
		::confdown::tmpCfg(end_second)

	# adapt number of digit for spinbox bug in 8.5
	if { $::confdown::tmpCfg(end_hour) < 10} {
		set ::confdown::tmpCfg(end_hour) [ string index $::confdown::tmpCfg(end_hour) end ]
	}
	if { $::confdown::tmpCfg(end_minute) < 10} {
		set ::confdown::tmpCfg(end_minute) [ string index $::confdown::tmpCfg(end_minute) end ]
	}
	if { $::confdown::tmpCfg(end_second) < 10} {
		set ::confdown::tmpCfg(end_second) [ string index $::confdown::tmpCfg(end_second) end ]
	}

	lassign [ ::confdown::seconds2hms $::confdown::tmpCfg(duration) ] \
		::confdown::tmpCfg(duration_hour) \
		::confdown::tmpCfg(duration_minute) \
		::confdown::tmpCfg(duration_second)

	lassign [ ::confdown::seconds2hms $::confdown::tmpCfg(warning) ] \
		::confdown::tmpCfg(warning_hour) \
		::confdown::tmpCfg(warning_minute) \
		::confdown::tmpCfg(warning_second)

	lassign [ ::confdown::seconds2hms $::confdown::tmpCfg(critical) ] \
		::confdown::tmpCfg(critical_hour) \
		::confdown::tmpCfg(critical_minute) \
		::confdown::tmpCfg(critical_second)

}

#-------------------------------------------------------------------------------
# Get configuration values from configuration dialog
#-------------------------------------------------------------------------------
proc ::confdown::cfgDlgGet {} {
	# Be sure to have 2 digit for spinbox bug in 8.5
	set ::confdown::tmpCfg(end_hour) [format "%02d" $::confdown::tmpCfg(end_hour)]
	set ::confdown::tmpCfg(end_minute) [format "%02d" $::confdown::tmpCfg(end_minute)]
	set ::confdown::tmpCfg(end_second) [format "%02d" $::confdown::tmpCfg(end_second)]

	# Get values
	set ::confdown::tmpCfg(end_time) \
		[clock scan \
			 [ list $::confdown::tmpCfg(end_hour) \
				   $::confdown::tmpCfg(end_minute) \
				   $::confdown::tmpCfg(end_second) ] \
			 -format "%H %M %S" ]

	set	::confdown::tmpCfg(duration) [ ::confdown::hms2seconds \
										   $::confdown::tmpCfg(duration_hour) \
										   $::confdown::tmpCfg(duration_minute) \
										   $::confdown::tmpCfg(duration_second)]

	set	::confdown::tmpCfg(warning) [ ::confdown::hms2seconds \
										   $::confdown::tmpCfg(warning_hour) \
										   $::confdown::tmpCfg(warning_minute) \
										   $::confdown::tmpCfg(warning_second)]

	set	::confdown::tmpCfg(critical) [ ::confdown::hms2seconds \
										   $::confdown::tmpCfg(critical_hour) \
										   $::confdown::tmpCfg(critical_minute) \
										   $::confdown::tmpCfg(critical_second)]

	array set ::confdown::cfg [ array get ::confdown::tmpCfg ]
}

#-------------------------------------------------------------------------------
# Configuration dialog actions
#-------------------------------------------------------------------------------
proc ::confdown::cfgDlgAction  {action} {
	if {$action eq "ok"} {
		::confdown::cfgDlgGet
	}
	destroy .cfgDlg
	set x [winfo rootx .]
	raise .
	wm geometry . "-$x+0"
	wm geometry . "-10000+0"
}

#-------------------------------------------------------------------------------
# Configuration dialog
#-------------------------------------------------------------------------------
proc ::confdown::cfgDlg {} {

	if { $::confdown::running eq 1 } {
		# Do not show configuration dialog if running
		return
	}

    if { [ winfo exists .cfgDlg ] } {
		set xpos [winfo rootx .main]
		set xmax [expr {[winfo screenwidth .] - [winfo reqwidth .cfgDlg]}]
		if {$xpos > $xmax} {
			set xpos [expr {[winfo rootx .main] + [winfo reqwidth .main] \
								- [winfo reqwidth .cfgDlg]}]
		}
		set ypos [expr {[winfo rooty .main] + [winfo reqheight .main]}]
		set ymax [expr {[winfo screenheight .] - \
							([winfo reqheight .cfgDlg] + $::confdown::wmBorder)}]
		if {$ypos > $ymax} {
			set ypos [expr {[winfo rooty .main] - \
								([winfo reqheight .cfgDlg] + $::confdown::wmBorder)} ]
		}
		wm geometry .cfgDlg "+$xpos+$ypos"
        focus .cfgDlg.bCancel
        return
    }

	set cfgDlg [ toplevel .cfgDlg ]

    wm withdraw .cfgDlg

	# Initialize it
	::confdown::cfgDlgSet

	# Contents
	# --------
	set nb [ ttk::notebook .cfgDlg.nb ]

	set time_page [ ttk::frame $nb.time]
	::confdown::buildTimePage $time_page
	.cfgDlg.nb add $time_page -underline 0 -padding 2
	.cfgDlg.nb tab 0 -text [mc "Time"]

	set color_page [ ttk::frame $nb.color ]
	::confdown::buildColorPage $color_page
	.cfgDlg.nb add $color_page -underline 0 -padding 2
	.cfgDlg.nb tab 1 -text [mc "Color"]

	set save_page [ ttk::frame $nb.save ]
	::confdown::buildSavePage $save_page
	.cfgDlg.nb add $save_page -underline 0 -padding 2
	.cfgDlg.nb tab 2 -text [mc "Save"]

	set help_page [ ttk::frame $nb.help ]
	::confdown::buildHelpPage $help_page
	.cfgDlg.nb add $help_page -underline 0 -padding 2
	.cfgDlg.nb tab 3 -text [mc "Help"]

	set license_page [ ttk::frame $nb.license ]
	::confdown::buildLicensePage $license_page
	.cfgDlg.nb add $license_page -underline 0 -padding 2
	.cfgDlg.nb tab 4 -text [mc "License"]

	# Actions
	# -------
    set faction [ ttk::frame .cfgDlg.fAction -borderwidth 2 -relief groove ]
	set bOk [ ttk::button $faction.bOk \
				  -text [ mc "Ok" ] \
				  -command {::confdown::cfgDlgAction "ok"} ]
	set bCancel [ ttk::button $faction.bCancel \
					  -text [ mc "Cancel" ] \
					  -command {::confdown::cfgDlgAction "cancel"} ]
	grid $bOk -column 0 -row 1 -padx 10 -pady 5
	grid $bCancel -column 1 -row 1 -padx 10 -pady 5
	grid columnconfigure $faction 0 -weight 1
	grid columnconfigure $faction 1 -weight 1

	pack $nb -padx 5 -pady 5 -fill both -expand 1
    pack $faction -fill x -expand 1 -anchor s

	update

	# Display it
	# ----------
	wm title .cfgDlg [ mc "Configuration" ]

	set xpos [winfo rootx .main]
	set xmax [expr {[winfo screenwidth .] - [winfo reqwidth .cfgDlg]}]
	if {$xpos > $xmax} {
		set xpos [expr {[winfo rootx .main] + [winfo reqwidth .main] \
							- [winfo reqwidth .cfgDlg]}]
	}
	set ypos [expr {[winfo rooty .main] + [winfo reqheight .main]}]
	set ymax [expr {[winfo screenheight .] - \
						([winfo reqheight .cfgDlg] + $::confdown::wmBorder)}]
	if {$ypos > $ymax} {
		set ypos [expr {[winfo rooty .main] - \
							([winfo reqheight .cfgDlg] + $::confdown::wmBorder)} ]
	}

    wm geometry .cfgDlg "+$xpos+$ypos"

    wm deiconify .cfgDlg
    raise .cfgDlg

    focus $bCancel
    grab .cfgDlg
}

#-------------------------------------------------------------------------------
# Move GUI
#-------------------------------------------------------------------------------
proc ::confdown::moveGui {key} {
	set x [winfo rootx .main]
	set y [winfo rooty .main]
	set w [winfo reqwidth .main]
	set h [winfo reqheight .main]
	switch -exact -- $key {
		Right {set x [expr {$x + $h}]}
		Left  {set x [expr {$x - $h}]}
		Up    {set y [expr {$y - $h}]}
		Down  {set y [expr {$y + $h}]}
	}
	if {$x < 0} {
		set x 0
	} elseif {[expr {$x + $w}] > [winfo screenwidth .]} {
		set x [expr {[winfo screenwidth .] - $w}]
	}
	if {$y < 0} {
		set y 0
	} elseif {[expr {$y + $h}] > [winfo screenheight .]} {
		set y [expr {[winfo screenheight .] - $h}]
	}
	wm geometry .main "+$x+$y"
}

#-------------------------------------------------------------------------------
# Build GUI
#-------------------------------------------------------------------------------
proc ::confdown::buildGui {} {
	wm geometry . 1x1-10000+0
	wm withdraw .

	toplevel .main
	progress .main.pg -height 8 -width 120 -relief sunken
	pack .main.pg
	::confdown::buildPopupMenu .main.pg

	.main.pg -val 0.0

	wm overrideredirect .main true
	wm attributes .main -topmost true
	update idletasks

	if {$::confdown::cfg(save_position)} {
		# must always be fully visible in the screen
		if {$::confdown::cfg(x) < 0 } {
			set ::confdown::cfg(x) 0
		} else {
			set xmax [expr {[winfo screenwidth .] - [winfo reqwidth .main]}]
			if {$::confdown::cfg(x) > $xmax } {
				set ::confdown::cfg(x) $xmax
			}
		}
		if {$::confdown::cfg(y) < 0 } {
			set ::confdown::cfg(y) 0
		} else {
			set ymax [expr {[winfo screenheight .] - [winfo reqheight .main]}]
			if {$::confdown::cfg(y) > $ymax } {
				set ::confdown::cfg(y) $ymax
			}
		}
		wm geometry .main "+$::confdown::cfg(x)+$::confdown::cfg(y)"
	}

	# Mouse button for moving window
	# From: http://wiki.tcl.tk/9406
	bind .main.pg <ButtonPress-1> {
		set iX0 [expr {%X - [winfo rootx .main]}]
		set iY0 [expr {%Y - [winfo rooty .main]}]
		set pgMoved 0
	}

	bind .main.pg <Button1-Motion> {
		if {%X < $iX0} {
			set iX0 %X
		} else {
			set xmax [expr {[winfo screenwidth .] - [winfo reqwidth .main] + $iX0}]
			if {%X > $xmax} {
				set iX0 [expr {%X - ([winfo screenwidth .] - [winfo reqwidth .main])}]
			}
		}
		if { %Y < $iY0 } {
			set iY0 %Y
		} else {
			set ymax [expr {[winfo screenheight .] - [winfo reqheight .main] + $iY0}]
			if {%Y > $ymax} {
				set iY0 [expr {%Y - ([winfo screenheight .] - [winfo reqheight .main])}]
			}
		}
		wm geometry .main +[expr {%X - $iX0}]+[expr {%Y - $iY0}]
		set pgMoved 1
	}

	bind .main.pg <ButtonRelease-1> {
		if { $pgMoved } break
	}

	# Mouse buttons actions in main toplevel
	bind .main <ButtonRelease-1>       {::confdown::start}
	bind .main <ButtonRelease-2>       {::confdown::quit}
	bind .main <Shift-ButtonRelease-1> {::confdown::quit}

	# Keyboard binding in .
	bind . <KeyPress-space>            {::confdown::start}
	bind . <Alt-KeyPress-c>            {::confdown::cfgDlg}
	bind . <Alt-KeyPress-m>            {
		tk_popup .main.pg.menu [winfo rootx .main] [winfo rooty .main]
	}
	bind . <Alt-KeyPress-q>            {::confdown::quit}
	bind . <Alt-KeyPress-Right>        {::confdown::moveGui %K}
	bind . <Alt-KeyPress-Left>         {::confdown::moveGui %K}
	bind . <Alt-KeyPress-Up>           {::confdown::moveGui %K}
	bind . <Alt-KeyPress-Down>         {::confdown::moveGui %K}

	# Adjust . which is keep for keyboard access out of the screen
	wm deiconify .
	raise .
	wm geometry . -10000+0
}

#-------------------------------------------------------------------------------
# Update a menu item
#-------------------------------------------------------------------------------
proc ::confdown::setMnuTxt {menu index name} {
	set underline [ string first & $name ]
	if { $underline < 0 } {
		set label $name
	} else {
		if { $underline == 0 } {
			set label [ string range $name 1 end ]
		} else {
			set label [ string range $name 0 [ expr { $underline - 1 } ] ]
			append label [ string range $name [ expr { $underline + 1 } ] end ]
		}
	}

	$menu entryconfigure $index -label $label
	if { $underline >= 0 } {
		$menu entryconfigure $index -underline $underline
	}
}

#-------------------------------------------------------------------------------
# Update GUI
#-------------------------------------------------------------------------------
proc ::confdown::updateGui {} {
	::confdown::setMnuTxt $::confdown::widgets(mnu) $::confdown::idx(mnuConfig) \
 		[mc "Configuration"]
	::confdown::setMnuTxt $::confdown::widgets(mnu) $::confdown::idx(mnuLng) \
 		[mc "Language"]
	::confdown::setMnuTxt $::confdown::widgets(mnu) $::confdown::idx(mnuQuit) \
 		[mc "Quit"]
}

#*******************************************************************************
#                                 Languages                                    *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Load a new language file
#-------------------------------------------------------------------------------
proc ::confdown::loadLanguage {code} {
	if {$code == {} } {
		# get user default language
		set code [ eval "::lngcode::getValidCode [ mcpreferences ]" ]
	} else {
		# be sure supplyed code is a valid language code
		set code [ ::lngcode::getValidCode $code ]
	}
	mclocale $code
	# load language file stored in "msg" folder
	mcload "$::confdown::scriptFolder/msg"
	set ::confdown::cfg(lng) $code
}

#-------------------------------------------------------------------------------
# Change language
#-------------------------------------------------------------------------------
proc ::confdown::switchLanguage {code} {
	::confdown::loadLanguage $code
	::confdown::updateGui
}

#*******************************************************************************
#                                 Running                                      *
#*******************************************************************************

#-------------------------------------------------------------------------------
# Run count down
#-------------------------------------------------------------------------------
proc ::confdown::run {} {
	update idletasks
	if {$::confdown::running eq 0} {
		set remaining 0
	} else {
		set remaining [expr { $::confdown::cfg(end_time) - [clock seconds] } ]
	}

	if { $remaining <= $::confdown::cfg(warning) } {
		if { $remaining <= $::confdown::cfg(critical) } {
			.main.pg -bg $::confdown::cfg(critical_color)
		} else {
			.main.pg -bg $::confdown::cfg(warning_color)
		}
	}
	set ratio [expr { ($remaining * 1.0) / $::confdown::cfg(duration) }]

	.main.pg -val $ratio

	if { $remaining > 0 } {
		after 1000 ::confdown::run
	} else {
		.main.pg -val 0.0
		set ::confdown::running 0

		if {$::confdown::debug} {
			puts "Finished"
			puts "-------------------"
		}
	}
}

#-------------------------------------------------------------------------------
# Initialize and start count down
#-------------------------------------------------------------------------------
proc ::confdown::start {} {

	if {$::confdown::running} {
		set ::confdown::running 0
		.main.pg -val 0.0
		update idletasks
	} else {
		.main.pg -bg $::confdown::cfg(normal_color)
		.main.pg -val 1.0

		set ::confdown::cfg(start_time) [clock seconds]

		if {$::confdown::cfg(end_type) eq 0} {
			# end time
			set ::confdown::cfg(duration) \
				[expr { $::confdown::cfg(end_time) - $::confdown::cfg(start_time) } ]
		}

		set ::confdown::cfg(end_time) \
			[expr { $::confdown::cfg(start_time) + $::confdown::cfg(duration) } ]

		if {$::confdown::debug} {
			puts "start_time: $::confdown::cfg(start_time)"
			puts "end_time:   $::confdown::cfg(end_time)"
			puts "duration:   $::confdown::cfg(duration)"
		}

		set ::confdown::running 1
		::confdown::run
	}
}

#-------------------------------------------------------------------------------
# Exit application
#-------------------------------------------------------------------------------
proc ::confdown::quit {} {
	if { $::confdown::cfg(save_config) eq 1 } {
		::confdown::saveConfig
	}
	exit
}

#-------------------------------------------------------------------------------
# Main process
#-------------------------------------------------------------------------------
proc ::confdown::main {} {
	if {[::lock::aquireLock confdown]} {
		::confdown::init
		::confdown::loadConfig
		::confdown::buildGui
		::confdown::loadLanguage $::confdown::cfg(lng)
		set index [ lsearch $::lngcode::CodeList [mclocale] ]
		$::confdown::widgets(mnuLng) invoke $index
		update idletasks
		if {$::confdown::firstTime} {
			::confdown::cfgDlg
		}
	} else {
		puts "Already running"
		exit
	}
}

::confdown::main
