unit MCMain;

/////////////////////////////////////////////////////////////////////////////
//                       Part of MD5Checker project                        //
//                     Compute and compare MD5 file hash                   //
//  2007  Main developper Alain JAFFRE         http://jack.r.free.fr       //
//                                                                         //
// Main part of that project                                               //
/////////////////////////////////////////////////////////////////////////////
//  13/11/2008 Version 1.0.1                                               //
//   Change end of line from LF to CR + LF when saving md5sums             //
/////////////////////////////////////////////////////////////////////////////

{***************************************************************************}
{ Ce logiciel est un logiciel libre. Vous pouvez le diffuser et/ou le       }
{ modifier suivant les termes de la GNU General Public License telle que    }
{ publie par la Free Software Foundation, soit la version 3 de cette        }
{ license, soit ( votre convenance) une version ultrieure.                }
{                                                                           }
{ Ce programme est diffus dans l'espoir qu'il sera utile, mais SANS AUCUNE }
{ GARANTIE, sans mme une garantie implicite de COMMERCIALISABILITE ou      }
{ d'ADEQUATION A UN USAGE PARTICULIER. Voyez la GNU General Public License  }
{ pour plus de dtails.                                                     }
{                                                                           }
{ Vous devriez avoir reu une copie de la GNU General Public License avec   }
{ ce programme, veuillez consulter <http://www.gnu.org/licenses/>           }
{***************************************************************************}

{***************************************************************************}
{ This program is free software. You can redistribute it and/or modify it   }
{ under the terms of the GNU Public License as published by the             }
{ Free Software Foundation, either version 3 of the license, or             }
{ (at your option) any later version.                                       }
{                                                                           }
{ This program is distributed in the hope it will be useful, but WITHOUT    }
{ ANY WARRANTY, without even the implied warranty of MERCHANTABILITY or     }
{ FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for  }
{ more details.                                                             }
{                                                                           }
{ You should have received a copy of the GNU General Public License along   }
{ with this program, if not, see <http://www.gnu.org/licenses/>.            }
{***************************************************************************}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, ComCtrls, ImgList, ToolWin, Menus, ActnList,
  StdCtrls, Printers, XPMenu, DCPcrypt2, DCPmd5;

type
  TFrmMain = class(TForm)
    MainMnu: TMainMenu;
    MnuFile: TMenuItem;
    MnuTools: TMenuItem;
    MnuHelp: TMenuItem;
    MnuFileOpen: TMenuItem;
    MnuFileSave: TMenuItem;
    MnuFileSpacer1: TMenuItem;
    MnuFileQuit: TMenuItem;
    MnuToolsLanguage: TMenuItem;
    MnuToolsFrench: TMenuItem;
    MnuToolsEnglish: TMenuItem;
    MnuHelpAbout: TMenuItem;
    TbrAction: TToolBar;
    TBtnOpen: TToolButton;
    TBSpacer1: TToolButton;
    TBtnSave: TToolButton;
    TBSpacer2: TToolButton;
    TBtnPrint: TToolButton;
    MnuFilePrint: TMenuItem;
    ImgLst: TImageList;
    SBrMain: TStatusBar;
    OpenDlg: TOpenDialog;
    ActionList: TActionList;
    ActFileOpen: TAction;
    ActFileSave: TAction;
    ActFilePrint: TAction;
    ActFileQuit: TAction;
    ActHelpAbout: TAction;
    TbrMenu: TToolBar;
    ListView: TListView;
    PgrBr: TProgressBar;
    DCP_md5: TDCP_md5;
    LblCurrentPath: TLabel;
    ActStop: TAction;
    TBSpacer3: TToolButton;
    TBtnStop: TToolButton;
    PrintDlg: TPrintDialog;
    procedure FormCreate(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure ActFileOpenExecute(Sender: TObject);
    procedure ActFileSaveExecute(Sender: TObject);
    procedure ActFilePrintExecute(Sender: TObject);
    procedure ActFileQuitExecute(Sender: TObject);
    procedure ActHelpAboutExecute(Sender: TObject);
    procedure ActStopExecute(Sender: TObject);
    procedure MnuToolsLanguageClick(Sender: TObject);
  private
    { Dclarations prives }
    XPMenu: TXPMenu;
    FileList: TStrings;
    CurrentPath: string;
    HashList: TStringList;
    Stopped: boolean;
    function ComputeMD5(filename: string): string;
    procedure SaveMd5(Hash, Filename: string);
    procedure SaveMd5sumsFile(Filename: string);
    procedure SplitLine(aLine: string; var Hash, Filename: string);
    procedure GetMd5FromFile(MD5Filename: string; var aList: TStringList);
    procedure GetMd5sumsFileContents;
    procedure GetCompareHash(Filename: string; var CompareHash, CompareFilename: string);
    procedure UpdateUI;
    procedure LockUI;
    procedure ResetUI;
  public
    { Dclarations publiques }
  end;

const
  Version= '1.0.1';
var
  FrmMain: TFrmMain;

implementation

{$R *.dfm}

uses
  MCLng, MCAbout, MCSaveFormat;

{------------------------------------------------------------------------------}

{*****************************************************************************}
{ Main form                                                                   }
{*****************************************************************************}

procedure TFrmMain.FormCreate(Sender: TObject);
// Create the main form of that application
var
  N: integer;
  Locale: string;
begin
  // XP menu style
  XPMenu := TXPMenu.Create(Self);
  with XPMenu do
  begin
    Name := 'XPMenu';
    DimLevel := 30;
    GrayLevel := 10;
    Font.Charset := ANSI_CHARSET;
    Font.Color := clMenuText;
    Font.Height := -11;
    Font.Name := 'Microsoft Sans Serif';
    Font.Style := [];
    Color := clBtnFace;
    DrawMenuBar := False;
    IconBackColor := clBtnFace;
    MenuBarColor := clBtnFace;
    SelectColor := clHighlight;
    SelectBorderColor := clHighlight;
    SelectFontColor := clMenuText;
    DisabledColor := clInactiveCaption;
    SeparatorColor := clBtnFace;
    CheckedColor := clHighlight;
    IconWidth := 24;
    DrawSelect := True;
    UseSystemColors := True;
    UseDimColor := False;
    OverrideOwnerDraw := False;
    Gradient := False;
    FlatMenu := True;
    AutoDetect := True;
    Active := True;
    Left := 56;
    Top := 120;
  end;

  TBSpacer3.Visible:= false;
  TBtnStop.Visible:= false;
  PgrBr.Parent:= SBrMain;
  PgrBr.Top:= 2;
  PgrBr.Left:= 1;
  PgrBr.Visible:= false;
  SBrMain.SimplePanel:= true;
  ListView.Align:= alClient;
  ListView.Columns[1].MinWidth:= 200;

  ActFileSave.Enabled:= false;
  ActFilePrint.Enabled:= false;

  FileList:= TStringList.Create;
  CurrentPath:= GetCurrentDir;
  HashList:= TStringList.Create;
  Stopped:= false;

  // Manage languages
  BuildMsgList;
  N:= Languages.IndexOf(SysLocale.DefaultLCID);
  Locale:= Languages.Name[N];
  if Pos('Franais',Locale) > 0 then CurrentLanguage:= fr
                                else CurrentLanguage:= en;
  MnuToolsFrench.Checked:= true;
  UpdateUi;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.FormShow(Sender: TObject);
begin
  FrmSaveFormat.SaveFormat.Clear;
  FrmSaveFormat.SaveFormat.Add('md5sums');
  FrmSaveFormat.SaveFormat.Add('md5sums.txt');
  FrmSaveFormat.SaveFormat.Add('_md5sum.txt');

  FrmSaveFormat.SelectedFormat:= 0;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.FormDestroy(Sender: TObject);
// Destroy the main form of that application
var
  N: integer;
begin
  for N:= 0 to HashList.Count-1 do
  begin
    if HashList.Objects[N] <> nil then HashList.Objects[N].Free;
  end;
  HashList.Free;
  FileList.Free;
end;

{*****************************************************************************}
{ Private                                                                      }
{*****************************************************************************}

function TFrmMain.ComputeMD5(Filename: string): string;
// Compute MD5 hash for specified file
var
  Buffer: array[0..65535] of byte;
  ReadBytes: cardinal;
  InputStream: TFileStream;
  Digest: array of byte;
  Hash: string;
  N: longint;
  F: file of Byte;
  MaxSize: Cardinal;
  CurrentSize: Cardinal;
  Coef: double;
  Iteration: cardinal;
begin
  if FileExists(Filename) then
  begin
    // Get file size
    AssignFile(F, Filename);
    Reset(F);
    try
      MaxSize := FileSize(F);
     finally
      CloseFile(F);
    end;
    if MaxSize <> 0 then Coef:= 100 / MaxSize
                    else Coef:= 1;
    CurrentSize:= 0;
    PgrBr.Max:= 100;
    PgrBr.Position:= 0;
    // Compute MD5
    try
      InputStream := TFileStream.Create(Filename,fmOpenRead);
      DCP_md5.Init;
      Iteration:= 0;
      repeat
        // read into the buffer
        ReadBytes := InputStream.Read(Buffer,Sizeof(Buffer));
        // hash the buffer
        DCP_md5.Update(Buffer,ReadBytes);
        // Display
        CurrentSize:= CurrentSize + ReadBytes;
        PgrBr.Position:= round(CurrentSize * Coef);
        Iteration:= Iteration + 1;
        if (MaxSize > 1000000000) and (Iteration mod 160 = 0) then
          SBrMain.Panels[2].Text:= IntToStr(CurrentSize div 1048576) + ' / ' + IntToStr(MaxSize div 1048576) + ' Mb';
        Update;
        Application.ProcessMessages;
      until (ReadBytes <> Sizeof(Buffer)) or Stopped;
      InputStream.Free;
      SetLength(Digest,DCP_md5.HashSize div 8);
      DCP_md5.Final(Digest[0]);;
      // convert into an hexadecimal string
      Hash:= '';
      for N:= 0 to Length(Digest) - 1 do
        Hash:= Hash + IntToHex(Digest[N],2);
      Hash:= lowercase(Hash);
    except
      Hash:= '';
    end;
  end;
  if Stopped then result:= ''
             else result:= Hash;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.SaveMd5(Hash, Filename: string);
var
  Md5sumsFilename: string;
  Text: string;
  F: textfile;
  GoSave: boolean;
  Info: string;
begin
  Md5sumsFilename:= CurrentPath + ChangeFileExt(Filename,'_md5sum.txt');
  if FileExists(Md5sumsFilename) then
  begin
    Info:= Format(GetMsg(85),[Md5sumsFilename]);
    GoSave:= MessageDlg(Info, mtInformation, [mbOk, mbCancel], 0) = mrOk;
  end
  else GoSave:= true;
  if GoSave then
  begin
    Text:= Hash + '  ' + ExtractFileName(Filename) + chr(10);
    AssignFile(F, Md5sumsFilename);
    Rewrite(F);
    Write(F, Text);
    CloseFile(F);
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.SaveMd5sumsFile(Filename: string);
var
  Md5sumsFilename: string;
  Text: string;
  Data: string;
  F: textfile;
  N: longint;
  ListItem: TListItem;
  GoSave: boolean;
  Info: string;
begin
  Md5sumsFilename:= CurrentPath + Filename;
  if FileExists(Md5sumsFilename) then
  begin
    Info:= Format(GetMsg(85),[Md5sumsFilename]);
    GoSave:= MessageDlg(Info, mtInformation, [mbOk, mbCancel], 0) = mrOk;
  end
  else GoSave:= true;
  if GoSave then
  begin
    Text:= '';
    for N:= 0 to ListView.Items.Count - 1 do
    begin
      ListItem:= ListView.Items[N];
      Data:= ListItem.SubItems[0] + '  ' + ListItem.Caption + chr(13) + chr(10);
      Text:= Text + Data;
    end;
    AssignFile(F, Md5sumsFilename);
    Rewrite(F);
    Write(F, Text);
    CloseFile(F);
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.SplitLine(aLine: string; var Hash, Filename: string);
// Split one line of an md5sum file into hash and filename
var
  Divider: integer;
begin
  Hash:= '';
  Filename:= '';
  Divider:= pos(' ',aLine);
  if Divider = 0 then Divider:= pos(chr(9),aLine);
  if Divider <> 0 then
  begin
    Hash:= Copy(aLine,0,Divider - 1);
    Filename:= Copy(aLine, Divider+1, length(aLine) - Divider);
    Filename:= trim(Filename);
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.GetMd5FromFile(MD5Filename: string; var aList: TStringList);
// Get all md5sum found in that file and store them in the list
var
  FullMD5Filename: string;
  Contents: TStrings;
  L: integer;
  aLine: string;
  Hash: string;
  Filename: string;
  anObject: TStrings;
begin
  FullMD5Filename :=  CurrentPath + MD5Filename;
  if FileExists(FullMD5Filename) then
  begin
    Contents:= TStringList.Create;
    Contents.LoadFromFile(FullMD5Filename);
    for L:= 0 to Contents.Count-1 do
    begin
      aLine:= Contents[L];
      SplitLine(aLine, Hash, Filename);
      if (Filename <> '') and (Hash <> '') then
      begin
        anObject:= TStringList.Create;
        anObject.add(Hash);
        anObject.add(MD5Filename);
        aList.AddObject(Filename, anObject);
      end;
      if Stopped then exit;
      Application.ProcessMessages;
    end;
    Contents.Free;
    aList.Sort;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.GetMd5sumsFileContents;
// Get the contents of all general md5sum files
var
  N: integer;
begin
  HashList.Clear;
  HashList.Sorted:= false;
  for N:= 0 to 1 do
  begin
    GetMd5FromFile(FrmSaveFormat.SaveFormat[N], HashList);
  end;
  HashList.Sorted:= true;
  HashList.Sort;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.GetCompareHash(Filename: string; var CompareHash, CompareFilename: string);
// Find files with couple hash / filename and return:
// - filename in which we found it
// - found hash
//
// Compare filename could be in the form:
// - md5sums
// - md5sums.txt
// - Filename without extension + '_md5sum.txt'
var
  SingleMD5Filename: string;
  SingleMD5FullFilename: string;
  Index: integer;
  anObject: TStrings;
  TmpList: TStringList;
begin
  CompareHash:= '';
  CompareFilename:= '';
  SingleMD5Filename:= ChangeFileExt(Filename, FrmSaveFormat.SaveFormat[2]);
  SingleMD5FullFilename:= CurrentPath + SingleMD5Filename;
  if FileExists(SingleMD5FullFilename) then
  begin
    TmpList:= TStringList.Create;
    TmpList.Sorted:= false;
    GetMd5FromFile(SingleMD5Filename, TmpList);
    TmpList.Sorted:= true;
    TmpList.Sort;
    if TmpList.Find(Filename, Index) then
    begin
      anObject:= TStrings(TmpList.Objects[Index]);
      CompareHash:= anObject[0];
      CompareFilename:= anObject[1];
    end;
    TmpList.Free;
  end;
  if CompareFilename = '' then
  begin
    GetMd5sumsFileContents;
    if HashList.Find(Filename, Index) then
    begin
      anObject:= TStrings(HashList.Objects[Index]);
      CompareHash:= anObject[0];
      CompareFilename:= anObject[1];
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.UpdateUI;
// Update UI for proper language
begin
  // File menu
  MnuFile.Caption:= GetMsg(01);
  MnuFile.Hint:= GetMsg(02);
  ActFileOpen.Caption:= GetMsg(03);
  ActFileOpen.Hint:= GetMsg(04);
  ActFileSave.Caption:= GetMsg(05);
  ActFileSave.Hint:= GetMsg(06);
  ActFilePrint.Caption:= GetMsg(07);
  ActFilePrint.Hint:= GetMsg(08);
  ActFileQuit.Caption:= GetMsg(19);
  ActFileQuit.Hint:= GetMsg(20);
  // Tools menu
  MnuTools.Caption:= GetMsg(21);
  MnuTools.Hint:= GetMsg(22);
  MnuToolsLanguage.Caption:= GetMsg(23);
  MnuToolsLanguage.Hint:= GetMsg(24);
  // Help menu
  MnuHelp.Caption:= GetMsg(31);
  MnuHelp.Hint:= GetMsg(32);
  ActHelpAbout.Caption:= GetMsg(33);
  ActHelpAbout.Hint:= GetMsg(34);
  // Stop
  ActStop.Caption:= GetMsg(35);
  ActStop.Hint:= GetMsg(36);
  // Hash listview
  ListView.Columns[0].Caption:= GetMsg(41);
  ListView.Columns[1].Caption:= GetMsg(42);
  ListView.Columns[2].Caption:= GetMsg(43);
  // Current folder label
  LblCurrentPath.Caption:= Format(GetMsg(51),[CurrentPath]);

  TbrMenu.Menu:= nil;
  TbrMenu.Menu:= MainMnu;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.LockUI;
// Lock UI during computation
begin
  SBrMain.SimplePanel:= false;
  PgrBr.Visible:= true;
  TBSpacer3.Visible:= true;
  TBtnStop.Visible:= true;
  ActFileOpen.Enabled:= false;
  ActFileSave.Enabled:= false;
  ActFilePrint.Enabled:= false;
  ActHelpAbout.Enabled:= false;
  MnuToolsLanguage.Enabled:= false;
  Stopped:= false;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.ResetUI;
// Reset UI after computation
begin

  SBrMain.Panels[1].Text:= '';
  SBrMain.Panels[2].Text:= '';
  SBrMain.SimplePanel:= true;
  PgrBr.Visible:= false;
  TBSpacer3.Visible:= false;
  TBtnStop.Visible:= false;
  ActFileOpen.Enabled:= true;
  ActHelpAbout.Enabled:= true;
  MnuToolsLanguage.Enabled:= true;
  Stopped:= false;
end;

{*****************************************************************************}
{ Public                                                                      }
{*****************************************************************************}

procedure TFrmMain.ActFileOpenExecute(Sender: TObject);
// Select files for which we need to compute and compare hash
const
  GreenCheck = 4;
  RedCross = 5;
var
  N: longint;
  FullFilename: string;
  Filename: string;
  Hash: string;
  ListItem: TListItem;
  CompareHash: string;
  CompareFilename: string;
  //Start: cardinal;
  //Timing: cardinal;
begin
  OpenDlg.InitialDir:= CurrentPath;
  OpenDlg.Filter:= GetMsg(53);
  if OpenDlg.Execute then
  begin
    LockUI;
    FileList.Clear;
    ListView.Items.Clear;
    with OpenDlg.Files do
    begin
      for N := 0 to Count - 1 do
      begin
        //Start:= GetTickCount;
        FullFilename:= Strings[N];
        Filename:= ExtractFileName(FullFilename);
        CurrentPath:= ExtractFilePath(FullFilename);
        LblCurrentPath.Caption:= Format(GetMsg(51),[CurrentPath]);
        SbrMain.Panels[1].Text:= Format(GetMsg(52),[Filename]);
        FileList.Add(FullFilename);
        ListItem := ListView.Items.Add;
        ListItem.Caption := Filename;
        Hash:= ComputeMD5(FullFilename);
        ListItem.SubItems.Add(Hash);
        GetCompareHash(Filename, CompareHash, CompareFilename);

        if Stopped then
        begin
          ResetUI;
          Exit;
        end;

        if CompareFilename <> '' then
        begin
          ListItem.SubItems.Add(CompareFilename);
          if CompareHash = Hash then ListItem.StateIndex:= GreenCheck
                                else ListItem.StateIndex:= RedCross;
        end
        else ListItem.StateIndex:= -1;

        //Timing:= GetTickCount - start;
        //SBrMain.Panels[2].Text:= IntToStr(Timing) + ' ms';

      end;
      LblCurrentPath.Caption:= Format(GetMsg(51),[CurrentPath]);
      ActFileSave.Enabled:= true;
      ActFilePrint.Enabled:= true;
      ResetUI;
     end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.ActFileSaveExecute(Sender: TObject);
// Save files hash in a single file or in separate files
var
  Filename: string;
  Hash: string;
  N: integer;
begin
  // Save format selection box
  FrmSaveFormat.Title:= GetMsg(81);
  FrmSaveFormat.RgpTitle:= GetMsg(82);
  FrmSaveFormat.BtnOkTitle:= GetMsg(83);
  FrmSaveFormat.BtnCancelTitle:= GetMsg(84);
  if FrmSaveFormat.Execute then
  begin
    case FrmSaveFormat.SelectedFormat of
      0..1: begin
              Filename:= FrmSaveFormat.SaveFormat[FrmSaveFormat.SelectedFormat];
              SaveMd5sumsFile(Filename);
            end;
      2: begin
           for N:= 0 to ListView.Items.Count - 1 do
           begin
             Filename:= ListView.Items[N].Caption;
             Hash:= ListView.Items[N].SubItems[0];
             SaveMd5(Hash, Filename);
           end;
         end;
    end;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.ActFilePrintExecute(Sender: TObject);
// Print files hash
var
  N: longint;
  Data: string;
  Size: integer;
  ListItem: TListItem;
  Text: TStrings;
  r: TRect;
begin
  if PrintDlg.Execute then
  begin
    Text:= TStringList.Create;
    // Caption
    Data:= ListView.Columns[1].Caption;
    Size:= length(ListView.Items[0].SubItems[0]);
    while length(Data) < Size do Data:= Data + ' ';
    Data:= Data + ' | ';
    Data := Data + ListView.Columns[0].Caption;
    Text.Add(Data);
    // Line
    Data:= '';
    Size:= 2 * Size;
    while length(Data) < Size do Data:= Data + '-';
    Text.Add(Data);
    // Contents
    for N:= 0 to ListView.Items.Count - 1 do
    begin
      ListItem:= ListView.Items[N];
      Data:= ListItem.SubItems[0] + ' | ';
      Data:= Data + ListItem.Caption;
      Text.Add(Data);
    end;

    with Printer do
    begin
      Title:= FrmMain.Caption;
      Canvas.Font.Name:= 'Courier New';
      Canvas.Font.Size:= 14;
      r := Rect(200,200,(Pagewidth - 200),(PageHeight - 200));
      BeginDoc;
      Canvas.TextOut(200,200 + Canvas.TextHeight(FrmMain.Caption),FrmMain.Caption);
      Canvas.Font.Size:= 10;
      for N := 0 to Text.Count-1 do
        Canvas.TextOut(200,200 + ((N + 4) * Canvas.TextHeight(Text[N])),Text[N]);
      Canvas.Brush.Color := clBlack;
      Canvas.FrameRect(r);
      EndDoc;
    end;

    Text.Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.ActFileQuitExecute(Sender: TObject);
// Exit that application
begin
  Stopped:= true;
  Close;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.ActHelpAboutExecute(Sender: TObject);
// Display an about box
begin
  with TFrmAbout.Create(Application) do
  try
    SoftVersion:= Version;
    BtnOkTitle:= GetMsg(83);
    Showmodal;
  finally
    Free;
  end;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.ActStopExecute(Sender: TObject);
// Stop MD5 processing
begin
  Stopped:= true;
end;

{------------------------------------------------------------------------------}

procedure TFrmMain.MnuToolsLanguageClick(Sender: TObject);
// Change user interface language
begin
  with Sender as TMenuItem do
  begin
    MnuToolsEnglish.Checked:= false;
    MnuToolsFrench.Checked:= false;
    Checked:= true;
    CurrentLanguage:= TComponent(Sender).Tag;
  end;
  UpdateUI;
end;

end.
